//============================================================================
//  The contents of this file are covered by the Viskores license. See
//  LICENSE.txt for details.
//
//  By contributing to this file, all contributors agree to the Developer
//  Certificate of Origin Version 1.1 (DCO 1.1) as stated in DCO.txt.
//============================================================================

//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================

#ifndef viskores_filter_image_processing_ImageDifference_h
#define viskores_filter_image_processing_ImageDifference_h

#include <viskores/filter/Filter.h>
#include <viskores/filter/image_processing/viskores_filter_image_processing_export.h>

namespace viskores
{
namespace filter
{
namespace image_processing
{
/// \brief Construct an ImageDifference of a given DataSet
///
/// The dataset generated by executing this filter is a Dataset with two Fields:
///   - "image-diff": Uniform Structured Dataset, difference values of image A - B
///   - "threshold-output": Uniform Structured Dataset, the magnitudes of the pixel differences
///
/// The threshold-output is calculated for each pixel using the `viskores::Magnitude` vector function
/// on the individual pixel difference.
///
class VISKORES_FILTER_IMAGE_PROCESSING_EXPORT ImageDifference : public viskores::filter::Filter
{
public:
  VISKORES_CONT ImageDifference();

  VISKORES_CONT viskores::IdComponent GetAverageRadius() const { return this->AverageRadius; }
  VISKORES_CONT void SetAverageRadius(const viskores::IdComponent& averageRadius)
  {
    this->AverageRadius = averageRadius;
  }

  VISKORES_CONT viskores::IdComponent GetPixelShiftRadius() const { return this->PixelShiftRadius; }
  VISKORES_CONT void SetPixelShiftRadius(const viskores::IdComponent& pixelShiftRadius)
  {
    this->PixelShiftRadius = pixelShiftRadius;
  }

  VISKORES_CONT viskores::FloatDefault GetAllowedPixelErrorRatio() const
  {
    return this->AllowedPixelErrorRatio;
  }
  VISKORES_CONT void SetAllowedPixelErrorRatio(const viskores::FloatDefault& pixelErrorRatio)
  {
    this->AllowedPixelErrorRatio = pixelErrorRatio;
  }

  VISKORES_CONT viskores::FloatDefault GetPixelDiffThreshold() const
  {
    return this->PixelDiffThreshold;
  }
  VISKORES_CONT void SetPixelDiffThreshold(const viskores::FloatDefault& threshold)
  {
    this->PixelDiffThreshold = threshold;
  }

  VISKORES_CONT bool GetImageDiffWithinThreshold() const { return this->ImageDiffWithinThreshold; }
  VISKORES_CONT void SetThresholdFieldName(const std::string& name)
  {
    this->ThresholdFieldName = name;
  }
  VISKORES_CONT std::string GetThresholdFieldName() const { return this->ThresholdFieldName; }

  /// Choose the primary field to operate on.  For Image difference A - B, A is the
  /// primary field.
  VISKORES_CONT
  void SetPrimaryField(
    const std::string& name,
    viskores::cont::Field::Association association = viskores::cont::Field::Association::Any)
  {
    this->SetActiveField(name, association);
  }

  VISKORES_CONT std::string GetPrimaryFieldName() const { return this->GetActiveFieldName(); }
  VISKORES_CONT viskores::cont::Field::Association GetPrimaryFieldAssociation() const
  {
    return this->GetActiveFieldAssociation();
  }

  /// Choose the secondary field to operate on.  For Image difference A - B, B is the
  /// secondary field.
  VISKORES_CONT
  void SetSecondaryField(
    const std::string& name,
    viskores::cont::Field::Association association = viskores::cont::Field::Association::Any)
  {
    this->SetActiveField(1, name, association);
  }

  VISKORES_CONT std::string GetSecondaryFieldName() const { return this->GetActiveFieldName(1); }
  VISKORES_CONT viskores::cont::Field::Association GetSecondaryFieldAssociation() const
  {
    return this->GetActiveFieldAssociation(1);
  }

private:
  VISKORES_CONT viskores::cont::DataSet DoExecute(
    const viskores::cont::DataSet& primaryArray) override;

  viskores::IdComponent AverageRadius = 0;
  viskores::IdComponent PixelShiftRadius = 0;
  viskores::FloatDefault AllowedPixelErrorRatio = 0.00025f;
  viskores::FloatDefault PixelDiffThreshold = 0.05f;
  bool ImageDiffWithinThreshold = true;
  std::string ThresholdFieldName = "threshold-output";
};
} // namespace image_processing
} // namespace filter
} // namespace viskores

#endif // viskores_filter_image_processing_ImageDifference_h
