import { Thread, type ThreadOptions } from '../Thread.ts';
/**
 * Creates a thread from a web worker. This function can be used either from a JavaScript
 * environment that *created* a web worker, or from within a web worker that has been
 * created.
 *
 * @see https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API/Using_web_workers
 *
 * @example
 * import {ThreadWebWorker} from '@quilted/threads';
 *
 * // If inside a web worker:
 * const thread = new ThreadWebWorker(self, {
 *   exports: {
 *     async sendMessage(message) {
 *      console.log(message);
 *     },
 *   },
 * });
 *
 * // If in an environment that creates a worker:
 * const worker = new Worker('worker.js');
 * const thread = new ThreadWebWorker(worker);
 *
 * await thread.imports.sendMessage('Hello world!');
 */
export declare class ThreadWebWorker<Imports = Record<string, never>, Exports = Record<string, never>> extends Thread<Imports, Exports> {
    readonly worker: Worker;
    static from<Imports = Record<string, never>, Exports = Record<string, never>>(worker: Worker, options?: ThreadOptions<Imports, Exports>): ThreadWebWorker<Imports, Exports>;
    /**
     * Starts a thread wrapped around a `Worker` object, and returns the imports
     * of the thread.
     *
     * @example
     * ```ts
     * import {ThreadWebWorker} from '@quilted/threads';
     *
     * // On the main thread:
     * const worker = new Worker('worker.js');
     * const {getMessage} = ThreadWebWorker.import(worker);
     * const message = await getMessage(); // 'Hello, world!'
     *
     * // Inside a web worker:
     *
     * import {ThreadWebWorker} from '@quilted/threads';
     *
     * ThreadWebWorker.export(self, {
     *   async getMessage() {
     *     return 'Hello, world!';
     *   },
     * });
     * ```
     */
    static import<Imports = Record<string, never>>(worker: Worker, options?: Omit<ThreadOptions<Imports, Record<string, never>>, 'imports'>): import("../Thread.ts").ThreadImports<Imports>;
    /**
     * Starts a thread wrapped around a `Worker` object, providing the second
     * argument as the exports of the thread.
     *
     * @example
     * ```ts
     * import {ThreadWebWorker} from '@quilted/threads';
     *
     * // Inside a web worker:
     * ThreadWebWorker.export(self, {
     *   async getMessage() {
     *     return 'Hello, world!';
     *   },
     * });
     *
     * // On the main thread:
     *
     * import {ThreadWebWorker} from '@quilted/threads';
     *
     * const worker = new Worker('worker.js');
     * const {getMessage} = ThreadWebWorker.import(worker);
     * const message = await getMessage(); // 'Hello, world!'
     * ```
     */
    static export<Exports = Record<string, never>>(worker: Worker, exports: Exports, options?: Omit<ThreadOptions<Record<string, never>, Exports>, 'exports'>): void;
    static self: (<Imports_1 = Record<string, never>, Exports_1 = Record<string, never>>(options?: ThreadOptions<Imports_1, Exports_1>) => ThreadWebWorker<Imports_1, Exports_1>) & {
        import: <Imports_1 = Record<string, never>>(options?: Omit<ThreadOptions<Imports_1, Record<string, never>>, "imports">) => import("../Thread.ts").ThreadImports<Imports_1>;
        export: <Exports_1 = Record<string, never>>(exports: Exports_1, options?: Omit<ThreadOptions<Record<string, never>, Exports_1>, "exports">) => void;
    };
    constructor(worker: Worker, options?: ThreadOptions<Imports, Exports>);
}
//# sourceMappingURL=ThreadWebWorker.d.ts.map