/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.parser;

import ghidra.app.services.DataTypeQueryService;
import ghidra.app.util.cparser.C.ParseException;
import ghidra.program.model.data.Array;
import ghidra.program.model.data.CategoryPath;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.DataTypePath;
import ghidra.program.model.data.Dynamic;
import ghidra.program.model.data.FactoryDataType;
import ghidra.program.model.data.FunctionDefinitionDataType;
import ghidra.program.model.data.InvalidDataTypeException;
import ghidra.program.model.data.ParameterDefinition;
import ghidra.program.model.data.ParameterDefinitionImpl;
import ghidra.program.model.data.Pointer;
import ghidra.program.model.data.TypeDef;
import ghidra.program.model.listing.FunctionSignature;
import ghidra.util.data.DataTypeParser;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;

public class FunctionSignatureParser {
    private static final String REPLACEMENT_DT_NAME = "__REPLACE_DT_NAME__";
    private static final String REPLACE_NAME = "__REPLACE_NAME__";
    private DataTypeParser dataTypeParser;
    private Map<String, DataType> dtMap = new HashMap<String, DataType>();
    private Map<String, String> nameMap = new HashMap<String, String>();
    private DataTypeManager destDataTypeManager;
    private ParserDataTypeManagerService dtmService;

    public FunctionSignatureParser(DataTypeManager destDataTypeManager, DataTypeQueryService service) {
        this.destDataTypeManager = destDataTypeManager;
        if (destDataTypeManager == null && service == null) {
            throw new IllegalArgumentException("Destination DataTypeManager or DataTypeManagerService provider required");
        }
        if (service != null) {
            this.dtmService = new ParserDataTypeManagerService(service);
        }
        this.dataTypeParser = new DataTypeParser(destDataTypeManager, destDataTypeManager, this.dtmService, DataTypeParser.AllowedDataTypes.FIXED_LENGTH);
    }

    public FunctionDefinitionDataType parse(FunctionSignature originalSignature, String signatureText) throws ParseException, CancelledException {
        this.dtMap.clear();
        this.nameMap.clear();
        if (this.dtmService != null) {
            this.dtmService.clearCache();
        }
        if (originalSignature != null) {
            this.initDataTypeMap(originalSignature);
            signatureText = this.cleanUpSignatureText(signatureText, originalSignature);
        }
        String functionName = this.extractFunctionName(signatureText);
        FunctionDefinitionDataType function = new FunctionDefinitionDataType(functionName, this.destDataTypeManager);
        function.setReturnType(this.extractReturnType(signatureText));
        function.setArguments(this.extractArguments(signatureText));
        function.setVarArgs(this.hasVarArgs(signatureText));
        return function;
    }

    private void initDataTypeMap(FunctionSignature signature) {
        this.cacheDataType(signature.getReturnType());
        for (ParameterDefinition p : signature.getArguments()) {
            this.cacheDataType(p.getDataType());
        }
    }

    private void cacheDataType(DataType dataType) {
        if (dataType == null || dataType instanceof Dynamic || dataType instanceof FactoryDataType) {
            return;
        }
        DataType baseType = null;
        if (dataType instanceof Pointer) {
            baseType = ((Pointer)dataType).getDataType();
        } else if (dataType instanceof Array) {
            baseType = ((Array)dataType).getDataType();
        } else if (dataType instanceof TypeDef) {
            baseType = ((TypeDef)dataType).getDataType();
        }
        this.dtMap.put(dataType.getName(), dataType);
        this.cacheDataType(baseType);
    }

    private boolean hasVarArgs(String newSignatureText) {
        int startIndex = newSignatureText.lastIndexOf(44);
        int endIndex = newSignatureText.indexOf(41);
        if (startIndex < 0 || endIndex < 0 || startIndex >= endIndex) {
            return false;
        }
        String lastArg = newSignatureText.substring(startIndex + 1, endIndex).trim();
        return "...".equals(lastArg);
    }

    private ParameterDefinition[] extractArguments(String newSignatureText) throws ParseException, CancelledException {
        String[] split;
        int startIndex = newSignatureText.indexOf(40);
        int endIndex = newSignatureText.indexOf(41);
        if (startIndex < 0 || endIndex < 0 || startIndex >= endIndex) {
            throw new ParseException("Can't parse function arguments");
        }
        String trailingText = newSignatureText.substring(endIndex + 1);
        if (trailingText.trim().length() > 0) {
            throw new ParseException("Unexpected trailing text at end of function: " + trailingText);
        }
        String argString = newSignatureText.substring(startIndex + 1, endIndex).trim();
        if (argString.length() == 0) {
            return new ParameterDefinition[0];
        }
        if ("void".equalsIgnoreCase(argString)) {
            return new ParameterDefinition[0];
        }
        ArrayList<ParameterDefinition> parameterList = new ArrayList<ParameterDefinition>();
        for (String arg : split = argString.split(",")) {
            this.addParameter(parameterList, arg.trim());
        }
        return parameterList.toArray(new ParameterDefinition[parameterList.size()]);
    }

    private void addParameter(List<ParameterDefinition> parameterList, String arg) throws ParseException, CancelledException {
        if ("...".equals(arg)) {
            return;
        }
        if (arg.length() == 0) {
            throw new ParseException("Missing parameter");
        }
        DataType dt = this.resolveDataType(arg);
        if (dt != null) {
            parameterList.add((ParameterDefinition)new ParameterDefinitionImpl(null, dt, null));
            return;
        }
        int spaceIndex = arg.lastIndexOf(32);
        if (spaceIndex < 0) {
            throw new ParseException("Can't resolve datatype: " + arg);
        }
        int starIndex = arg.lastIndexOf(42);
        int nameIndex = Math.max(spaceIndex, starIndex) + 1;
        String name = this.resolveName(arg.substring(nameIndex).trim());
        String dtName = arg.substring(0, nameIndex).trim();
        dt = this.resolveDataType(dtName);
        if (dt == null) {
            throw new ParseException("Can't resolve datatype: " + dtName);
        }
        parameterList.add((ParameterDefinition)new ParameterDefinitionImpl(name, dt, null));
    }

    String cleanUpSignatureText(String text, FunctionSignature signature) {
        ParameterDefinition[] arguments;
        DataType returnType = signature.getReturnType();
        text = this.replaceDataTypeIfNeeded(text, returnType, REPLACEMENT_DT_NAME);
        text = this.replaceNameIfNeeded(text, signature.getName(), REPLACE_NAME);
        for (ParameterDefinition argument : arguments = signature.getArguments()) {
            text = this.replaceDataTypeIfNeeded(text, argument.getDataType(), REPLACEMENT_DT_NAME + argument.getOrdinal());
            text = this.replaceNameIfNeeded(text, argument.getName(), REPLACE_NAME + argument.getOrdinal());
        }
        return text;
    }

    private String replaceDataTypeIfNeeded(String text, DataType dataType, String replacementName) {
        String datatypeName = dataType.getName();
        if (this.canParseType(datatypeName)) {
            return text;
        }
        this.dtMap.put(replacementName, dataType);
        return this.substitute(text, datatypeName, replacementName);
    }

    private String replaceNameIfNeeded(String text, String name, String replacementName) {
        if (this.canParseName(name)) {
            return text;
        }
        this.nameMap.put(replacementName, name);
        return this.substitute(text, name, replacementName);
    }

    DataType extractReturnType(String signatureText) throws ParseException, CancelledException {
        int parenIndex = signatureText.indexOf(40);
        if (parenIndex < 0) {
            throw new ParseException("Can't find return type");
        }
        Object[] split = StringUtils.split((String)signatureText.substring(0, parenIndex));
        if (split.length < 2) {
            throw new ParseException("Can't find return type");
        }
        String returnTypeName = StringUtils.join((Object[])split, (String)" ", (int)0, (int)(split.length - 1));
        DataType dt = this.resolveDataType(returnTypeName);
        if (dt == null) {
            throw new ParseException("Can't resolve return type: " + returnTypeName);
        }
        return dt;
    }

    private DataType resolveDataType(String dataTypeName) throws CancelledException {
        if (this.dtMap.containsKey(dataTypeName)) {
            return this.dtMap.get(dataTypeName);
        }
        DataType dataType = null;
        try {
            dataType = this.dataTypeParser.parse(dataTypeName);
        }
        catch (InvalidDataTypeException invalidDataTypeException) {
            // empty catch block
        }
        return dataType;
    }

    String extractFunctionName(String signatureText) throws ParseException {
        int parenIndex = signatureText.indexOf(40);
        if (parenIndex < 0) {
            throw new ParseException("Can't find function name");
        }
        String[] split = StringUtils.split((String)signatureText.substring(0, parenIndex));
        if (split.length < 2) {
            throw new ParseException("Can't find function name");
        }
        String name = split[split.length - 1];
        return this.resolveName(name);
    }

    private String resolveName(String name) throws ParseException {
        if (this.nameMap.containsKey(name)) {
            return this.nameMap.get(name);
        }
        if (!this.canParseName(name)) {
            throw new ParseException("Can't parse name: " + name);
        }
        return name;
    }

    String substitute(String text, String searchString, String replacementString) {
        return text.replaceFirst(Pattern.quote(searchString), replacementString);
    }

    private boolean canParseName(String text) {
        return !StringUtils.containsAny((CharSequence)text, (CharSequence)"()*[], ");
    }

    private boolean canParseType(String text) {
        return !StringUtils.containsAny((CharSequence)text, (CharSequence)"()<>,");
    }

    private static class ParserDataTypeManagerService
    implements DataTypeQueryService {
        private Map<String, DataType> dtCache = new HashMap<String, DataType>();
        private final DataTypeQueryService service;

        ParserDataTypeManagerService(DataTypeQueryService service) {
            this.service = service;
        }

        void clearCache() {
            this.dtCache.clear();
        }

        @Override
        public List<DataType> getSortedDataTypeList() {
            return this.service.getSortedDataTypeList();
        }

        @Override
        public List<CategoryPath> getSortedCategoryPathList() {
            return this.service.getSortedCategoryPathList();
        }

        @Override
        public DataType getDataType(String filterText) {
            return this.promptForDataType(filterText);
        }

        @Override
        public DataType promptForDataType(String filterText) {
            DataType dt = this.dtCache.get(filterText);
            if (dt == null && (dt = this.service.promptForDataType(filterText)) != null) {
                this.dtCache.put(filterText, dt);
            }
            return dt;
        }

        @Override
        public List<DataType> getDataTypesByPath(DataTypePath path) {
            return this.service.getDataTypesByPath(path);
        }

        @Override
        public DataType getProgramDataTypeByPath(DataTypePath path) {
            return this.service.getProgramDataTypeByPath(path);
        }

        @Override
        public List<DataType> findDataTypes(String name, TaskMonitor monitor) {
            return this.service.findDataTypes(name, monitor);
        }
    }
}

