import { F as Message, ay as Task, aQ as TaskStatusUpdateEvent, aS as TaskArtifactUpdateEvent, z as PushNotificationConfig, ae as AgentCard, x as MessageSendParams, X as TaskQueryParams, Z as TaskIdParams, $ as TaskPushNotificationConfig, a3 as GetTaskPushNotificationConfigParams, a7 as ListTaskPushNotificationConfigParams, a9 as DeleteTaskPushNotificationConfigParams, j as JSONRPCResponse, aw as JSONRPCError } from '../extensions-DvruCIzw.cjs';
import { S as ServerCallContext, A as A2ARequestHandler } from '../a2a_request_handler-B3LxMq3P.cjs';
export { a as UnauthenticatedUser, U as User } from '../a2a_request_handler-B3LxMq3P.cjs';

type AgentExecutionEvent = Message | Task | TaskStatusUpdateEvent | TaskArtifactUpdateEvent;
/**
 * Event names supported by ExecutionEventBus.
 */
type ExecutionEventName = 'event' | 'finished';
interface ExecutionEventBus {
    publish(event: AgentExecutionEvent): void;
    on(eventName: ExecutionEventName, listener: (event: AgentExecutionEvent) => void): this;
    off(eventName: ExecutionEventName, listener: (event: AgentExecutionEvent) => void): this;
    once(eventName: ExecutionEventName, listener: (event: AgentExecutionEvent) => void): this;
    removeAllListeners(eventName?: ExecutionEventName): this;
    finished(): void;
}
/**
 * Web-compatible ExecutionEventBus using EventTarget.
 * Works across all modern runtimes: Node.js 15+, browsers, Cloudflare Workers, Deno, Bun.
 *
 * This implementation provides the subset of EventEmitter methods defined in the
 * ExecutionEventBus interface. Users extending DefaultExecutionEventBus should note
 * that other EventEmitter methods (e.g., listenerCount, rawListeners) are not available.
 */
declare class DefaultExecutionEventBus extends EventTarget implements ExecutionEventBus {
    private readonly eventListeners;
    private readonly finishedListeners;
    publish(event: AgentExecutionEvent): void;
    finished(): void;
    /**
     * EventEmitter-compatible 'on' method.
     * Wraps the listener to extract event detail from CustomEvent.
     * Supports multiple registrations of the same listener (like EventEmitter).
     * @param eventName The event name to listen for.
     * @param listener The callback function to invoke when the event is emitted.
     * @returns This instance for method chaining.
     */
    on(eventName: ExecutionEventName, listener: (event: AgentExecutionEvent) => void): this;
    /**
     * EventEmitter-compatible 'off' method.
     * Uses the stored wrapped listener for proper removal.
     * Removes at most one instance of a listener per call (like EventEmitter).
     * @param eventName The event name to stop listening for.
     * @param listener The callback function to remove.
     * @returns This instance for method chaining.
     */
    off(eventName: ExecutionEventName, listener: (event: AgentExecutionEvent) => void): this;
    /**
     * EventEmitter-compatible 'once' method.
     * Listener is automatically removed after first invocation.
     * Supports multiple registrations of the same listener (like EventEmitter).
     * @param eventName The event name to listen for once.
     * @param listener The callback function to invoke when the event is emitted.
     * @returns This instance for method chaining.
     */
    once(eventName: ExecutionEventName, listener: (event: AgentExecutionEvent) => void): this;
    /**
     * EventEmitter-compatible 'removeAllListeners' method.
     * Removes all listeners for a specific event or all events.
     * @param eventName Optional event name to remove listeners for. If omitted, removes all.
     * @returns This instance for method chaining.
     */
    removeAllListeners(eventName?: ExecutionEventName): this;
    /**
     * Adds a wrapped listener to the tracking map.
     */
    private trackListener;
    /**
     * Removes a wrapped listener from the tracking map (for once cleanup).
     */
    private untrackWrappedListener;
    private addEventListenerInternal;
    private removeEventListenerInternal;
    private addEventListenerOnceInternal;
    private addFinishedListenerInternal;
    private removeFinishedListenerInternal;
    private addFinishedListenerOnceInternal;
}

declare class RequestContext {
    readonly userMessage: Message;
    readonly taskId: string;
    readonly contextId: string;
    readonly task?: Task;
    readonly referenceTasks?: Task[];
    readonly context?: ServerCallContext;
    constructor(userMessage: Message, taskId: string, contextId: string, task?: Task, referenceTasks?: Task[], context?: ServerCallContext);
}

interface AgentExecutor {
    /**
     * Executes the agent logic based on the request context and publishes events.
     * @param requestContext The context of the current request.
     * @param eventBus The bus to publish execution events to.
     */
    execute: (requestContext: RequestContext, eventBus: ExecutionEventBus) => Promise<void>;
    /**
     * Method to explicitly cancel a running task.
     * The implementation should handle the logic of stopping the execution
     * and publishing the final 'canceled' status event on the provided event bus.
     * @param taskId The ID of the task to cancel.
     * @param eventBus The event bus associated with the task's execution.
     */
    cancelTask: (taskId: string, eventBus: ExecutionEventBus) => Promise<void>;
}

interface ExecutionEventBusManager {
    createOrGetByTaskId(taskId: string): ExecutionEventBus;
    getByTaskId(taskId: string): ExecutionEventBus | undefined;
    cleanupByTaskId(taskId: string): void;
}
declare class DefaultExecutionEventBusManager implements ExecutionEventBusManager {
    private taskIdToBus;
    /**
     * Creates or retrieves an existing ExecutionEventBus based on the taskId.
     * @param taskId The ID of the task.
     * @returns An instance of ExecutionEventBus.
     */
    createOrGetByTaskId(taskId: string): ExecutionEventBus;
    /**
     * Retrieves an existing ExecutionEventBus based on the taskId.
     * @param taskId The ID of the task.
     * @returns An instance of ExecutionEventBus or undefined if not found.
     */
    getByTaskId(taskId: string): ExecutionEventBus | undefined;
    /**
     * Removes the event bus for a given taskId.
     * This should be called when an execution flow is complete to free resources.
     * @param taskId The ID of the task.
     */
    cleanupByTaskId(taskId: string): void;
}

/**
 * An async queue that subscribes to an ExecutionEventBus for events
 * and provides an async generator to consume them.
 */
declare class ExecutionEventQueue {
    private eventBus;
    private eventQueue;
    private resolvePromise?;
    private stopped;
    private boundHandleEvent;
    constructor(eventBus: ExecutionEventBus);
    private handleEvent;
    private handleFinished;
    /**
     * Provides an async generator that yields events from the event bus.
     * Stops when a Message event is received or a TaskStatusUpdateEvent with final=true is received.
     */
    events(): AsyncGenerator<AgentExecutionEvent, void, undefined>;
    /**
     * Stops the event queue from processing further events.
     */
    stop(): void;
}

/**
 * Simplified interface for task storage providers.
 * Stores and retrieves the task.
 */
interface TaskStore {
    /**
     * Saves a task.
     * Overwrites existing data if the task ID exists.
     * @param task The task to save.
     * @param context The context of the current call.
     * @returns A promise resolving when the save operation is complete.
     */
    save(task: Task, context?: ServerCallContext): Promise<void>;
    /**
     * Loads a task by task ID.
     * @param taskId The ID of the task to load.
     * @param context The context of the current call.
     * @returns A promise resolving to an object containing the Task, or undefined if not found.
     */
    load(taskId: string, context?: ServerCallContext): Promise<Task | undefined>;
}
declare class InMemoryTaskStore implements TaskStore {
    private store;
    load(taskId: string): Promise<Task | undefined>;
    save(task: Task): Promise<void>;
}

interface PushNotificationStore {
    save(taskId: string, pushNotificationConfig: PushNotificationConfig): Promise<void>;
    load(taskId: string): Promise<PushNotificationConfig[]>;
    delete(taskId: string, configId?: string): Promise<void>;
}
declare class InMemoryPushNotificationStore implements PushNotificationStore {
    private store;
    save(taskId: string, pushNotificationConfig: PushNotificationConfig): Promise<void>;
    load(taskId: string): Promise<PushNotificationConfig[]>;
    delete(taskId: string, configId?: string): Promise<void>;
}

interface PushNotificationSender {
    send(task: Task): Promise<void>;
}

declare class DefaultRequestHandler implements A2ARequestHandler {
    private readonly agentCard;
    private readonly taskStore;
    private readonly agentExecutor;
    private readonly eventBusManager;
    private readonly pushNotificationStore?;
    private readonly pushNotificationSender?;
    private readonly extendedAgentCardProvider?;
    constructor(agentCard: AgentCard, taskStore: TaskStore, agentExecutor: AgentExecutor, eventBusManager?: ExecutionEventBusManager, pushNotificationStore?: PushNotificationStore, pushNotificationSender?: PushNotificationSender, extendedAgentCardProvider?: AgentCard | ExtendedAgentCardProvider);
    getAgentCard(): Promise<AgentCard>;
    getAuthenticatedExtendedAgentCard(context?: ServerCallContext): Promise<AgentCard>;
    private _createRequestContext;
    private _processEvents;
    sendMessage(params: MessageSendParams, context?: ServerCallContext): Promise<Message | Task>;
    sendMessageStream(params: MessageSendParams, context?: ServerCallContext): AsyncGenerator<Message | Task | TaskStatusUpdateEvent | TaskArtifactUpdateEvent, void, undefined>;
    getTask(params: TaskQueryParams, context?: ServerCallContext): Promise<Task>;
    cancelTask(params: TaskIdParams, context?: ServerCallContext): Promise<Task>;
    setTaskPushNotificationConfig(params: TaskPushNotificationConfig, context?: ServerCallContext): Promise<TaskPushNotificationConfig>;
    getTaskPushNotificationConfig(params: TaskIdParams | GetTaskPushNotificationConfigParams, context?: ServerCallContext): Promise<TaskPushNotificationConfig>;
    listTaskPushNotificationConfigs(params: ListTaskPushNotificationConfigParams, context?: ServerCallContext): Promise<TaskPushNotificationConfig[]>;
    deleteTaskPushNotificationConfig(params: DeleteTaskPushNotificationConfigParams, context?: ServerCallContext): Promise<void>;
    resubscribe(params: TaskIdParams, context?: ServerCallContext): AsyncGenerator<Task | TaskStatusUpdateEvent | TaskArtifactUpdateEvent, void, undefined>;
    private _sendPushNotificationIfNeeded;
    private _handleProcessingError;
}
type ExtendedAgentCardProvider = (context?: ServerCallContext) => Promise<AgentCard>;

declare class ResultManager {
    private readonly taskStore;
    private readonly serverCallContext?;
    private currentTask?;
    private latestUserMessage?;
    private finalMessageResult?;
    constructor(taskStore: TaskStore, serverCallContext?: ServerCallContext);
    setContext(latestUserMessage: Message): void;
    /**
     * Processes an agent execution event and updates the task store.
     * @param event The agent execution event.
     */
    processEvent(event: AgentExecutionEvent): Promise<void>;
    private saveCurrentTask;
    /**
     * Gets the final result, which could be a Message or a Task.
     * This should be called after the event stream has been fully processed.
     * @returns The final Message or the current Task.
     */
    getFinalResult(): Message | Task | undefined;
    /**
     * Gets the task currently being managed by this ResultManager instance.
     * This task could be one that was started with or one created during agent execution.
     * @returns The current Task or undefined if no task is active.
     */
    getCurrentTask(): Task | undefined;
}

/**
 * Handles JSON-RPC transport layer, routing requests to A2ARequestHandler.
 */
declare class JsonRpcTransportHandler {
    private requestHandler;
    constructor(requestHandler: A2ARequestHandler);
    /**
     * Handles an incoming JSON-RPC request.
     * For streaming methods, it returns an AsyncGenerator of JSONRPCResult.
     * For non-streaming methods, it returns a Promise of a single JSONRPCMessage (Result or ErrorResponse).
     */
    handle(requestBody: any, context?: ServerCallContext): Promise<JSONRPCResponse | AsyncGenerator<JSONRPCResponse, void, undefined>>;
    private isRequestValid;
    private paramsAreValid;
}

/**
 * Custom error class for A2A server operations, incorporating JSON-RPC error codes.
 */
declare class A2AError extends Error {
    code: number;
    data?: Record<string, unknown>;
    taskId?: string;
    constructor(code: number, message: string, data?: Record<string, unknown>, taskId?: string);
    /**
     * Formats the error into a standard JSON-RPC error object structure.
     */
    toJSONRPCError(): JSONRPCError;
    static parseError(message: string, data?: Record<string, unknown>): A2AError;
    static invalidRequest(message: string, data?: Record<string, unknown>): A2AError;
    static methodNotFound(method: string): A2AError;
    static invalidParams(message: string, data?: Record<string, unknown>): A2AError;
    static internalError(message: string, data?: Record<string, unknown>): A2AError;
    static taskNotFound(taskId: string): A2AError;
    static taskNotCancelable(taskId: string): A2AError;
    static pushNotificationNotSupported(): A2AError;
    static unsupportedOperation(operation: string): A2AError;
    static authenticatedExtendedCardNotConfigured(): A2AError;
}

interface DefaultPushNotificationSenderOptions {
    /**
     * Timeout in milliseconds for the abort controller. Defaults to 5000ms.
     */
    timeout?: number;
    /**
     * Custom header name for the token. Defaults to 'X-A2A-Notification-Token'.
     */
    tokenHeaderName?: string;
}
declare class DefaultPushNotificationSender implements PushNotificationSender {
    private readonly pushNotificationStore;
    private notificationChain;
    private readonly options;
    constructor(pushNotificationStore: PushNotificationStore, options?: DefaultPushNotificationSenderOptions);
    send(task: Task): Promise<void>;
    private _dispatchNotification;
}

export { A2AError, A2ARequestHandler, type AgentExecutionEvent, type AgentExecutor, DefaultExecutionEventBus, DefaultExecutionEventBusManager, DefaultPushNotificationSender, type DefaultPushNotificationSenderOptions, DefaultRequestHandler, type ExecutionEventBus, type ExecutionEventBusManager, type ExecutionEventName, ExecutionEventQueue, type ExtendedAgentCardProvider, InMemoryPushNotificationStore, InMemoryTaskStore, JsonRpcTransportHandler, type PushNotificationSender, type PushNotificationStore, RequestContext, ResultManager, ServerCallContext, type TaskStore };
