/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { loadSettings, SettingScope, } from '../../config/settings.js';
import { debugLogger } from '@google/gemini-cli-core';
import { exitCli } from '../utils.js';
export async function handleEnable(args) {
    const { name, scope } = args;
    const workspaceDir = process.cwd();
    const settings = loadSettings(workspaceDir);
    const currentDisabled = settings.forScope(scope).settings.skills?.disabled || [];
    const newDisabled = currentDisabled.filter((d) => d !== name);
    if (currentDisabled.length === newDisabled.length) {
        debugLogger.log(`Skill "${name}" is already enabled in scope "${scope}".`);
        return;
    }
    settings.setValue(scope, 'skills.disabled', newDisabled);
    debugLogger.log(`Skill "${name}" successfully enabled in scope "${scope}".`);
}
export const enableCommand = {
    command: 'enable <name>',
    describe: 'Enables an agent skill.',
    builder: (yargs) => yargs
        .positional('name', {
        describe: 'The name of the skill to enable.',
        type: 'string',
        demandOption: true,
    })
        .option('scope', {
        alias: 's',
        describe: 'The scope to enable the skill in (user or project).',
        type: 'string',
        default: 'user',
        choices: ['user', 'project'],
    }),
    handler: async (argv) => {
        const scope = argv['scope'] === 'project' ? SettingScope.Workspace : SettingScope.User;
        await handleEnable({
            name: argv['name'],
            scope,
        });
        await exitCli();
    },
};
//# sourceMappingURL=enable.js.map