import {
  A2A_ERROR_CODE,
  SSE_HEADERS,
  formatSSEErrorEvent,
  formatSSEEvent
} from "../../chunk-SJNAG4AL.js";
import {
  AGENT_CARD_PATH,
  HTTP_EXTENSION_HEADER
} from "../../chunk-3QDLXHKS.js";
import {
  A2AError,
  JsonRpcTransportHandler,
  ServerCallContext,
  UnauthenticatedUser
} from "../../chunk-LTPINR5K.js";
import {
  Extensions
} from "../../chunk-ZX6KNMCP.js";

// src/server/express/a2a_express_app.ts
import express3 from "express";

// src/server/express/json_rpc_handler.ts
import express from "express";
function jsonRpcHandler(options) {
  const jsonRpcTransportHandler = new JsonRpcTransportHandler(options.requestHandler);
  const router = express.Router();
  router.use(express.json(), jsonErrorHandler);
  router.post("/", async (req, res) => {
    try {
      const user = await options.userBuilder(req);
      const context = new ServerCallContext(
        Extensions.parseServiceParameter(req.header(HTTP_EXTENSION_HEADER)),
        user ?? new UnauthenticatedUser()
      );
      const rpcResponseOrStream = await jsonRpcTransportHandler.handle(req.body, context);
      if (context.activatedExtensions) {
        res.setHeader(HTTP_EXTENSION_HEADER, Array.from(context.activatedExtensions));
      }
      if (typeof rpcResponseOrStream?.[Symbol.asyncIterator] === "function") {
        const stream = rpcResponseOrStream;
        Object.entries(SSE_HEADERS).forEach(([key, value]) => {
          res.setHeader(key, value);
        });
        res.flushHeaders();
        try {
          for await (const event of stream) {
            res.write(formatSSEEvent(event));
          }
        } catch (streamError) {
          console.error(`Error during SSE streaming (request ${req.body?.id}):`, streamError);
          let a2aError;
          if (streamError instanceof A2AError) {
            a2aError = streamError;
          } else {
            a2aError = A2AError.internalError(
              streamError instanceof Error && streamError.message || "Streaming error."
            );
          }
          const errorResponse = {
            jsonrpc: "2.0",
            id: req.body?.id || null,
            // Use original request ID if available
            error: a2aError.toJSONRPCError()
          };
          if (!res.headersSent) {
            res.status(500).json(errorResponse);
          } else {
            res.write(formatSSEErrorEvent(errorResponse));
          }
        } finally {
          if (!res.writableEnded) {
            res.end();
          }
        }
      } else {
        const rpcResponse = rpcResponseOrStream;
        res.status(200).json(rpcResponse);
      }
    } catch (error) {
      console.error("Unhandled error in JSON-RPC POST handler:", error);
      const a2aError = error instanceof A2AError ? error : A2AError.internalError("General processing error.");
      const errorResponse = {
        jsonrpc: "2.0",
        id: req.body?.id || null,
        error: a2aError.toJSONRPCError()
      };
      if (!res.headersSent) {
        res.status(500).json(errorResponse);
      } else if (!res.writableEnded) {
        res.end();
      }
    }
  });
  return router;
}
var jsonErrorHandler = (err, _req, res, next) => {
  if (err instanceof SyntaxError && "body" in err) {
    const a2aError = A2AError.parseError("Invalid JSON payload.");
    const errorResponse = {
      jsonrpc: "2.0",
      id: null,
      error: a2aError.toJSONRPCError()
    };
    return res.status(400).json(errorResponse);
  }
  next(err);
};

// src/server/express/agent_card_handler.ts
import express2 from "express";
function agentCardHandler(options) {
  const router = express2.Router();
  const provider = typeof options.agentCardProvider === "function" ? options.agentCardProvider : options.agentCardProvider.getAgentCard.bind(options.agentCardProvider);
  router.get("/", async (_req, res) => {
    try {
      const agentCard = await provider();
      res.json(agentCard);
    } catch (error) {
      console.error("Error fetching agent card:", error);
      res.status(500).json({ error: "Failed to retrieve agent card" });
    }
  });
  return router;
}

// src/server/express/common.ts
var UserBuilder = {
  noAuthentication: () => Promise.resolve(new UnauthenticatedUser())
};

// src/server/express/a2a_express_app.ts
var A2AExpressApp = class {
  requestHandler;
  userBuilder;
  constructor(requestHandler, userBuilder = UserBuilder.noAuthentication) {
    this.requestHandler = requestHandler;
    this.userBuilder = userBuilder;
  }
  /**
   * Adds A2A routes to an existing Express app.
   * @param app Optional existing Express app.
   * @param baseUrl The base URL for A2A endpoints (e.g., "/a2a/api").
   * @param middlewares Optional array of Express middlewares to apply to the A2A routes.
   * @param agentCardPath Optional custom path for the agent card endpoint (defaults to .well-known/agent-card.json).
   * @returns The Express app with A2A routes.
   */
  setupRoutes(app, baseUrl = "", middlewares, agentCardPath = AGENT_CARD_PATH) {
    const router = express3.Router();
    router.use(express3.json(), jsonErrorHandler);
    if (middlewares && middlewares.length > 0) {
      router.use(middlewares);
    }
    router.use(
      jsonRpcHandler({
        requestHandler: this.requestHandler,
        userBuilder: this.userBuilder
      })
    );
    router.use(`/${agentCardPath}`, agentCardHandler({ agentCardProvider: this.requestHandler }));
    app.use(baseUrl, router);
    return app;
  }
};

// src/server/express/rest_handler.ts
import express4 from "express";

// src/server/transports/rest/rest_transport_handler.ts
var HTTP_STATUS = {
  OK: 200,
  CREATED: 201,
  ACCEPTED: 202,
  NO_CONTENT: 204,
  BAD_REQUEST: 400,
  UNAUTHORIZED: 401,
  NOT_FOUND: 404,
  CONFLICT: 409,
  INTERNAL_SERVER_ERROR: 500,
  NOT_IMPLEMENTED: 501
};
function mapErrorToStatus(errorCode) {
  switch (errorCode) {
    case A2A_ERROR_CODE.PARSE_ERROR:
    case A2A_ERROR_CODE.INVALID_REQUEST:
    case A2A_ERROR_CODE.INVALID_PARAMS:
      return HTTP_STATUS.BAD_REQUEST;
    case A2A_ERROR_CODE.METHOD_NOT_FOUND:
    case A2A_ERROR_CODE.TASK_NOT_FOUND:
      return HTTP_STATUS.NOT_FOUND;
    case A2A_ERROR_CODE.TASK_NOT_CANCELABLE:
      return HTTP_STATUS.CONFLICT;
    case A2A_ERROR_CODE.PUSH_NOTIFICATION_NOT_SUPPORTED:
    case A2A_ERROR_CODE.UNSUPPORTED_OPERATION:
      return HTTP_STATUS.BAD_REQUEST;
    default:
      return HTTP_STATUS.INTERNAL_SERVER_ERROR;
  }
}
function toHTTPError(error) {
  const errorObject = {
    code: error.code,
    message: error.message
  };
  if (error.data !== void 0) {
    errorObject.data = error.data;
  }
  return errorObject;
}
var RestTransportHandler = class _RestTransportHandler {
  requestHandler;
  constructor(requestHandler) {
    this.requestHandler = requestHandler;
  }
  // ==========================================================================
  // Public API Methods
  // ==========================================================================
  /**
   * Gets the agent card (for capability checks).
   */
  async getAgentCard() {
    return this.requestHandler.getAgentCard();
  }
  /**
   * Gets the authenticated extended agent card.
   */
  async getAuthenticatedExtendedAgentCard(context) {
    return this.requestHandler.getAuthenticatedExtendedAgentCard(context);
  }
  /**
   * Sends a message to the agent.
   * Accepts both snake_case and camelCase input, returns camelCase.
   */
  async sendMessage(params, context) {
    const normalized = this.normalizeMessageParams(params);
    return this.requestHandler.sendMessage(normalized, context);
  }
  /**
   * Sends a message with streaming response.
   * Accepts both snake_case and camelCase input, returns camelCase stream.
   * @throws {A2AError} UnsupportedOperation if streaming not supported
   */
  async sendMessageStream(params, context) {
    await this.requireCapability("streaming");
    const normalized = this.normalizeMessageParams(params);
    return this.requestHandler.sendMessageStream(normalized, context);
  }
  /**
   * Gets a task by ID.
   * Validates historyLength parameter if provided.
   */
  async getTask(taskId, context, historyLength) {
    const params = { id: taskId };
    if (historyLength !== void 0) {
      params.historyLength = this.parseHistoryLength(historyLength);
    }
    return this.requestHandler.getTask(params, context);
  }
  /**
   * Cancels a task.
   */
  async cancelTask(taskId, context) {
    const params = { id: taskId };
    return this.requestHandler.cancelTask(params, context);
  }
  /**
   * Resubscribes to task updates.
   * Returns camelCase stream of task updates.
   * @throws {A2AError} UnsupportedOperation if streaming not supported
   */
  async resubscribe(taskId, context) {
    await this.requireCapability("streaming");
    const params = { id: taskId };
    return this.requestHandler.resubscribe(params, context);
  }
  /**
   * Sets a push notification configuration.
   * Accepts both snake_case and camelCase input, returns camelCase.
   * @throws {A2AError} PushNotificationNotSupported if push notifications not supported
   */
  async setTaskPushNotificationConfig(config, context) {
    await this.requireCapability("pushNotifications");
    const normalized = this.normalizeTaskPushNotificationConfig(config);
    return this.requestHandler.setTaskPushNotificationConfig(normalized, context);
  }
  /**
   * Lists all push notification configurations for a task.
   */
  async listTaskPushNotificationConfigs(taskId, context) {
    return this.requestHandler.listTaskPushNotificationConfigs({ id: taskId }, context);
  }
  /**
   * Gets a specific push notification configuration.
   */
  async getTaskPushNotificationConfig(taskId, configId, context) {
    return this.requestHandler.getTaskPushNotificationConfig(
      { id: taskId, pushNotificationConfigId: configId },
      context
    );
  }
  /**
   * Deletes a push notification configuration.
   */
  async deleteTaskPushNotificationConfig(taskId, configId, context) {
    await this.requestHandler.deleteTaskPushNotificationConfig(
      { id: taskId, pushNotificationConfigId: configId },
      context
    );
  }
  // ==========================================================================
  // Private Transformation Methods
  // ==========================================================================
  // All type conversion between REST (snake_case) and internal (camelCase) formats
  /**
   * Validates and normalizes message parameters.
   * Accepts both snake_case and camelCase input.
   * @throws {A2AError} InvalidParams if message is missing or conversion fails
   */
  normalizeMessageParams(input) {
    if (!input.message) {
      throw A2AError.invalidParams("message is required");
    }
    try {
      return this.normalizeMessageSendParams(input);
    } catch (error) {
      if (error instanceof A2AError) throw error;
      throw A2AError.invalidParams(
        error instanceof Error ? error.message : "Invalid message parameters"
      );
    }
  }
  /**
   * Static map of capability to error for missing capabilities.
   */
  static CAPABILITY_ERRORS = {
    streaming: () => A2AError.unsupportedOperation("Agent does not support streaming"),
    pushNotifications: () => A2AError.pushNotificationNotSupported()
  };
  /**
   * Validates that the agent supports a required capability.
   * @throws {A2AError} UnsupportedOperation for streaming, PushNotificationNotSupported for push notifications
   */
  async requireCapability(capability) {
    const agentCard = await this.getAgentCard();
    if (!agentCard.capabilities?.[capability]) {
      throw _RestTransportHandler.CAPABILITY_ERRORS[capability]();
    }
  }
  /**
   * Parses and validates historyLength query parameter.
   */
  parseHistoryLength(value) {
    if (value === void 0 || value === null) {
      throw A2AError.invalidParams("historyLength is required");
    }
    const parsed = parseInt(String(value), 10);
    if (isNaN(parsed)) {
      throw A2AError.invalidParams("historyLength must be a valid integer");
    }
    if (parsed < 0) {
      throw A2AError.invalidParams("historyLength must be non-negative");
    }
    return parsed;
  }
  /**
   * Normalizes Part input - accepts both snake_case and camelCase for file mimeType.
   */
  normalizePart(part) {
    if (part.kind === "text") return { kind: "text", text: part.text };
    if (part.kind === "file") {
      const file = this.normalizeFile(part.file);
      return { kind: "file", file, metadata: part.metadata };
    }
    return { kind: "data", data: part.data, metadata: part.metadata };
  }
  /**
   * Normalizes File input - accepts both snake_case (mime_type) and camelCase (mimeType).
   */
  normalizeFile(f) {
    const file = f;
    const mimeType = file.mimeType ?? file.mime_type;
    if ("bytes" in file) {
      return { bytes: file.bytes, mimeType, name: file.name };
    }
    return { uri: file.uri, mimeType, name: file.name };
  }
  /**
   * Normalizes Message input - accepts both snake_case and camelCase.
   */
  normalizeMessage(input) {
    const m = input;
    const messageId = m.messageId ?? m.message_id;
    if (!messageId) {
      throw A2AError.invalidParams("message.messageId is required");
    }
    if (!m.parts || !Array.isArray(m.parts)) {
      throw A2AError.invalidParams("message.parts must be an array");
    }
    return {
      contextId: m.contextId ?? m.context_id,
      extensions: m.extensions,
      kind: "message",
      messageId,
      metadata: m.metadata,
      parts: m.parts.map((p) => this.normalizePart(p)),
      referenceTaskIds: m.referenceTaskIds ?? m.reference_task_ids,
      role: m.role,
      taskId: m.taskId ?? m.task_id
    };
  }
  /**
   * Normalizes MessageSendParams - accepts both snake_case and camelCase.
   */
  normalizeMessageSendParams(input) {
    const p = input;
    const config = p.configuration;
    return {
      configuration: config ? {
        acceptedOutputModes: config.acceptedOutputModes ?? config.accepted_output_modes,
        blocking: config.blocking,
        historyLength: config.historyLength ?? config.history_length
      } : void 0,
      message: this.normalizeMessage(p.message),
      metadata: p.metadata
    };
  }
  /**
   * Normalizes TaskPushNotificationConfig - accepts both snake_case and camelCase.
   */
  normalizeTaskPushNotificationConfig(input) {
    const c = input;
    const taskId = c.taskId ?? c.task_id;
    if (!taskId) {
      throw A2AError.invalidParams("taskId is required");
    }
    const pnConfig = c.pushNotificationConfig ?? c.push_notification_config;
    if (!pnConfig) {
      throw A2AError.invalidParams("pushNotificationConfig is required");
    }
    return {
      pushNotificationConfig: pnConfig,
      taskId
    };
  }
};

// src/server/express/rest_handler.ts
var restErrorHandler = (err, _req, res, next) => {
  if (err instanceof SyntaxError && "body" in err) {
    const a2aError = A2AError.parseError("Invalid JSON payload.");
    return res.status(400).json(toHTTPError(a2aError));
  }
  next(err);
};
function restHandler(options) {
  const router = express4.Router();
  const restTransportHandler = new RestTransportHandler(options.requestHandler);
  router.use(express4.json(), restErrorHandler);
  const buildContext = async (req) => {
    const user = await options.userBuilder(req);
    return new ServerCallContext(
      Extensions.parseServiceParameter(req.header(HTTP_EXTENSION_HEADER)),
      user
    );
  };
  const setExtensionsHeader = (res, context) => {
    if (context.activatedExtensions) {
      res.setHeader(HTTP_EXTENSION_HEADER, Array.from(context.activatedExtensions));
    }
  };
  const sendResponse = (res, statusCode, context, body) => {
    setExtensionsHeader(res, context);
    res.status(statusCode);
    if (statusCode === HTTP_STATUS.NO_CONTENT) {
      res.end();
    } else {
      res.json(body);
    }
  };
  const sendStreamResponse = async (res, stream, context) => {
    const iterator = stream[Symbol.asyncIterator]();
    let firstResult;
    try {
      firstResult = await iterator.next();
    } catch (error) {
      const a2aError = error instanceof A2AError ? error : A2AError.internalError(error instanceof Error ? error.message : "Streaming error");
      const statusCode = mapErrorToStatus(a2aError.code);
      sendResponse(res, statusCode, context, toHTTPError(a2aError));
      return;
    }
    Object.entries(SSE_HEADERS).forEach(([key, value]) => {
      res.setHeader(key, value);
    });
    setExtensionsHeader(res, context);
    res.flushHeaders();
    try {
      if (!firstResult.done) {
        res.write(formatSSEEvent(firstResult.value));
      }
      for await (const event of { [Symbol.asyncIterator]: () => iterator }) {
        res.write(formatSSEEvent(event));
      }
    } catch (streamError) {
      console.error("SSE streaming error:", streamError);
      const a2aError = streamError instanceof A2AError ? streamError : A2AError.internalError(
        streamError instanceof Error ? streamError.message : "Streaming error"
      );
      if (!res.writableEnded) {
        res.write(formatSSEErrorEvent(toHTTPError(a2aError)));
      }
    } finally {
      if (!res.writableEnded) {
        res.end();
      }
    }
  };
  const handleError = (res, error) => {
    if (res.headersSent) {
      if (!res.writableEnded) {
        res.end();
      }
      return;
    }
    const a2aError = error instanceof A2AError ? error : A2AError.internalError(error instanceof Error ? error.message : "Internal server error");
    const statusCode = mapErrorToStatus(a2aError.code);
    res.status(statusCode).json(toHTTPError(a2aError));
  };
  const asyncHandler = (handler) => {
    return async (req, res) => {
      try {
        await handler(req, res);
      } catch (error) {
        handleError(res, error);
      }
    };
  };
  router.get(
    "/v1/card",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const result = await restTransportHandler.getAuthenticatedExtendedAgentCard(context);
      sendResponse(res, HTTP_STATUS.OK, context, result);
    })
  );
  router.post(
    "/v1/message\\:send",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const result = await restTransportHandler.sendMessage(req.body, context);
      sendResponse(res, HTTP_STATUS.CREATED, context, result);
    })
  );
  router.post(
    "/v1/message\\:stream",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const stream = await restTransportHandler.sendMessageStream(req.body, context);
      await sendStreamResponse(res, stream, context);
    })
  );
  router.get(
    "/v1/tasks/:taskId",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const result = await restTransportHandler.getTask(
        req.params.taskId,
        context,
        req.query.historyLength
      );
      sendResponse(res, HTTP_STATUS.OK, context, result);
    })
  );
  router.post(
    "/v1/tasks/:taskId\\:cancel",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const result = await restTransportHandler.cancelTask(req.params.taskId, context);
      sendResponse(res, HTTP_STATUS.ACCEPTED, context, result);
    })
  );
  router.post(
    "/v1/tasks/:taskId\\:subscribe",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const stream = await restTransportHandler.resubscribe(req.params.taskId, context);
      await sendStreamResponse(res, stream, context);
    })
  );
  router.post(
    "/v1/tasks/:taskId/pushNotificationConfigs",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const config = {
        ...req.body,
        taskId: req.params.taskId,
        task_id: req.params.taskId
      };
      const result = await restTransportHandler.setTaskPushNotificationConfig(config, context);
      sendResponse(res, HTTP_STATUS.CREATED, context, result);
    })
  );
  router.get(
    "/v1/tasks/:taskId/pushNotificationConfigs",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const result = await restTransportHandler.listTaskPushNotificationConfigs(
        req.params.taskId,
        context
      );
      sendResponse(res, HTTP_STATUS.OK, context, result);
    })
  );
  router.get(
    "/v1/tasks/:taskId/pushNotificationConfigs/:configId",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      const result = await restTransportHandler.getTaskPushNotificationConfig(
        req.params.taskId,
        req.params.configId,
        context
      );
      sendResponse(res, HTTP_STATUS.OK, context, result);
    })
  );
  router.delete(
    "/v1/tasks/:taskId/pushNotificationConfigs/:configId",
    asyncHandler(async (req, res) => {
      const context = await buildContext(req);
      await restTransportHandler.deleteTaskPushNotificationConfig(
        req.params.taskId,
        req.params.configId,
        context
      );
      sendResponse(res, HTTP_STATUS.NO_CONTENT, context);
    })
  );
  return router;
}
export {
  A2AExpressApp,
  UserBuilder,
  agentCardHandler,
  jsonRpcHandler,
  restHandler
};
