! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
#define ESMF_FILENAME "ESMF_FieldBundle.F90"
!==============================================================================

!
! ESMF FieldBundle Module
module ESMF_FieldBundleMod
!
!==============================================================================
!
! This file contains the F90 implementation of the fieldbundle class.
!
!------------------------------------------------------------------------------
! INCLUDES
#include "ESMF.h"


!==============================================================================
!BOPI
! !MODULE: ESMF_FieldBundleMod
!

! F90 implementation of fieldbundle
!
!------------------------------------------------------------------------------

! !USES:
  use ESMF_UtilTypesMod ! ESMF utility types
  use ESMF_InitMacrosMod ! ESMF initializer macros
  use ESMF_BaseMod ! ESMF base class
  use ESMF_LogErrMod ! ESMF error handling
  use ESMF_F90InterfaceMod ! ESMF F90-C++ interface helper
  use ESMF_UtilStringMod
  use ESMF_ArraySpecMod
  use ESMF_ArrayMod
  use ESMF_ArrayBundleMod
  use ESMF_ContainerMod
  use ESMF_GridMod
  use ESMF_XGridMod
  use ESMF_MeshMod
  use ESMF_LocStreamMod
  use ESMF_GeomMod
  use ESMF_FieldMod
  use ESMF_FieldCreateMod
  use ESMF_FieldGetMod
  use ESMF_FieldPrMod
  use ESMF_FieldSMMMod
  use ESMF_FieldRegridMod
  use ESMF_FieldWrMod
  use ESMF_RHandleMod
  use ESMF_RegridMod
  use ESMF_StaggerLocMod
  use ESMF_VMMod
  use ESMF_IOMod
  use ESMF_FactorReadMod ! Read weight factors from netCDF file.

  implicit none

!------------------------------------------------------------------------------
! !PRIVATE TYPES:
  private
!------------------------------------------------------------------------------
! ! ESMF_FieldBundleStatus
!
!------------------------------------------------------------------------------

  type ESMF_FieldBundleStatus
#ifndef ESMF_NO_SEQUENCE
  sequence
#endif
    integer :: status
  end type

  type(ESMF_FieldBundleStatus), parameter :: ESMF_FBSTATUS_UNINIT = ESMF_FieldBundleStatus(1), &
                                  ESMF_FBSTATUS_EMPTY = ESMF_FieldBundleStatus(2), &
                                  ESMF_FBSTATUS_GRIDSET = ESMF_FieldBundleStatus(3)

!------------------------------------------------------------------------------
! ! ESMF_FieldBundle
!
!------------------------------------------------------------------------------

  type ESMF_FieldBundleType
#ifndef ESMF_NO_SEQUENCE
  sequence
#endif
    type(ESMF_Base) :: base ! base class object
    type(ESMF_Geom) :: geom ! base class object
    type(ESMF_Container) :: container ! internal storage implementation
    type(ESMF_FieldBundleStatus) :: status ! status of this FieldBundle
    logical :: is_proxy ! true if this is a proxy FB
    logical :: is_packed ! true if this is a packed FB
    character(len=:), allocatable:: encodeName! storage for packed FB
    ESMF_INIT_DECLARE
  end type

  ! F90 class type to hold pointer to FieldBundleType
  type ESMF_FieldBundle
#ifndef ESMF_NO_SEQUENCE
  sequence
#endif
    type(ESMF_FieldBundleType), pointer :: this
    ESMF_INIT_DECLARE_NAMED_ALIAS
  end type

  type ESMF_RHL
    type(ESMF_GeomType_Flag) :: srcGeomtype, dstGeomtype
    type(ESMF_Grid) :: srcGrid, dstGrid
    type(ESMF_Mesh) :: srcMesh, dstMesh
    ! field specific items, TODO: push into a FieldMatch() method
    type(ESMF_ArraySpec) :: srcArraySpec, dstArraySpec
    type(ESMF_StaggerLoc) :: srcStaggerLoc, dstStaggerLoc
    type(ESMF_MeshLoc) :: srcMeshLoc, dstMeshLoc
    integer, pointer :: srcGridToFieldMap(:)
    integer, pointer :: dstGridToFieldMap(:)
    integer, pointer :: srcUngriddedLBound(:)
    integer, pointer :: srcUngriddedUBound(:)
    integer, pointer :: dstUngriddedLBound(:)
    integer, pointer :: dstUngriddedUBound(:)
    ! remap specific items
    ! - guards
    logical :: guard_srcMaskValues
    logical :: guard_dstMaskValues
    logical :: guard_regridmethod
    logical :: guard_polemethod
    logical :: guard_regridPoleNPnts
    logical :: guard_lineType
    logical :: guard_normType
    logical :: guard_extrapMethod
    logical :: guard_extrapNumSrcPnts
    logical :: guard_extrapDistExponent
    logical :: guard_extrapNumLevels
    logical :: guard_unmappedaction
    logical :: guard_ignoreDegenerate
    logical :: guard_srcTermProcessing
    logical :: guard_pipelineDepth
    ! - remap options
    integer(ESMF_KIND_I4), pointer :: srcMaskValues(:)
    integer(ESMF_KIND_I4), pointer :: dstMaskValues(:)
    type(ESMF_RegridMethod_Flag) :: regridmethod
    type(ESMF_PoleMethod_Flag) :: polemethod
    integer :: regridPoleNPnts
    type(ESMF_LineType_Flag) :: lineType
    type(ESMF_NormType_Flag) :: normType
    type(ESMF_ExtrapMethod_Flag) :: extrapMethod
    integer :: extrapNumSrcPnts
    real :: extrapDistExponent
    integer :: extrapNumLevels
    type(ESMF_UnmappedAction_Flag) :: unmappedaction
    logical :: ignoreDegenerate
    integer :: srcTermProcessing
    integer :: pipelineDepth
    ! misc
    type(ESMF_RouteHandle) :: rh
    logical :: factorAllocFlag
    integer(ESMF_KIND_I4), pointer :: factorIndexList(:,:)
    real(ESMF_KIND_R8), pointer :: factorList(:)
    type(ESMF_RHL), pointer :: prev
  end type

!------------------------------------------------------------------------------
! !PUBLIC TYPES:
  public ESMF_FieldBundle
  public ESMF_FieldBundleType
  public ESMF_FieldBundleStatus
  public ESMF_RHL
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
!
! !PUBLIC MEMBER FUNCTIONS:

! - ESMF-public methods:
  public operator(==)
  public operator(/=)

  public ESMF_FieldBundleAdd
  public ESMF_FieldBundleAddReplace
  public ESMF_FieldBundleCreate
  public ESMF_FieldBundleDeserialize
  public ESMF_FieldBundleDestroy
  public ESMF_FieldBundleGet
  public ESMF_FieldBundleHalo
  public ESMF_FieldBundleHaloRelease
  public ESMF_FieldBundleHaloStore
  public ESMF_FieldBundleIsCreated
  public ESMF_FieldBundlePrint
  public ESMF_FieldBundleRead
  public ESMF_FieldBundleRedist
  public ESMF_FieldBundleRedistRelease
  public ESMF_FieldBundleRedistStore
  public ESMF_FieldBundleRegrid
  public ESMF_FieldBundleRegridRelease
  public ESMF_FieldBundleRegridStore
  public ESMF_FieldBundleRegridStorePair
  public ESMF_FieldBundleRemove
  public ESMF_FieldBundleReplace
  public ESMF_FieldBundleSerialize
  public ESMF_FieldBundleSet
  public ESMF_FieldBundleSMM
  public ESMF_FieldBundleSMMRelease
  public ESMF_FieldBundleSMMStore
  public ESMF_FieldBundleValidate
  public ESMF_FieldBundleWrite


!------------------------------------------------------------------------------
!------------------------------------------------------------------------------
! - ESMF-internal methods:
  public ESMF_FieldBundleGetInit
  public ESMF_FieldBundleDestruct
  public ESMF_FieldBundleTypeDeepCopy

!EOPI
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
  character(*), parameter, private :: version = &
    '$Id$'

!==============================================================================
!
! INTERFACE BLOCKS
!
!==============================================================================

! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_FieldBundleAdd -- Generic interface

! !INTERFACE:
  interface ESMF_FieldBundleAdd

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldBundleAddList
!EOPI

  end interface

! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_FieldBundleCreate -- Overloaded FieldBunleCreate interface

! !INTERFACE:
  interface ESMF_FieldBundleCreate

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldBundleCreateDefault
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldBundleCreateGridDataPtr1DI1 
 module procedure ESMF_FieldBundleCreateGridDataPtr2DI1 
 module procedure ESMF_FieldBundleCreateGridDataPtr3DI1 
 module procedure ESMF_FieldBundleCreateGridDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleCreateGridDataPtr5DI1 
 module procedure ESMF_FieldBundleCreateGridDataPtr6DI1 
 module procedure ESMF_FieldBundleCreateGridDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldBundleCreateGridDataPtr1DI2 
 module procedure ESMF_FieldBundleCreateGridDataPtr2DI2 
 module procedure ESMF_FieldBundleCreateGridDataPtr3DI2 
 module procedure ESMF_FieldBundleCreateGridDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleCreateGridDataPtr5DI2 
 module procedure ESMF_FieldBundleCreateGridDataPtr6DI2 
 module procedure ESMF_FieldBundleCreateGridDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldBundleCreateGridDataPtr1DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr1DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr1DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr1DR8 
 module procedure ESMF_FieldBundleCreateGridDataPtr2DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr2DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr2DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr2DR8 
 module procedure ESMF_FieldBundleCreateGridDataPtr3DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr3DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr3DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr3DR8 
 module procedure ESMF_FieldBundleCreateGridDataPtr4DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr4DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr4DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleCreateGridDataPtr5DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr5DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr5DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr5DR8 
 module procedure ESMF_FieldBundleCreateGridDataPtr6DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr6DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr6DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr6DR8 
 module procedure ESMF_FieldBundleCreateGridDataPtr7DI4 
 module procedure ESMF_FieldBundleCreateGridDataPtr7DI8 
 module procedure ESMF_FieldBundleCreateGridDataPtr7DR4 
 module procedure ESMF_FieldBundleCreateGridDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldBundleCreateMeshDataPtr1DI1 
 module procedure ESMF_FieldBundleCreateMeshDataPtr2DI1 
 module procedure ESMF_FieldBundleCreateMeshDataPtr3DI1 
 module procedure ESMF_FieldBundleCreateMeshDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleCreateMeshDataPtr5DI1 
 module procedure ESMF_FieldBundleCreateMeshDataPtr6DI1 
 module procedure ESMF_FieldBundleCreateMeshDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldBundleCreateMeshDataPtr1DI2 
 module procedure ESMF_FieldBundleCreateMeshDataPtr2DI2 
 module procedure ESMF_FieldBundleCreateMeshDataPtr3DI2 
 module procedure ESMF_FieldBundleCreateMeshDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleCreateMeshDataPtr5DI2 
 module procedure ESMF_FieldBundleCreateMeshDataPtr6DI2 
 module procedure ESMF_FieldBundleCreateMeshDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldBundleCreateMeshDataPtr1DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr1DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr1DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr1DR8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr2DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr2DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr2DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr2DR8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr3DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr3DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr3DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr3DR8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr4DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr4DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr4DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleCreateMeshDataPtr5DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr5DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr5DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr5DR8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr6DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr6DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr6DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr6DR8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr7DI4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr7DI8 
 module procedure ESMF_FieldBundleCreateMeshDataPtr7DR4 
 module procedure ESMF_FieldBundleCreateMeshDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_FieldBundleCreate} subroutines.

  end interface

!EOPI

! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_FieldBundleGet -- Generic interface

! !INTERFACE:
  interface ESMF_FieldBundleGet

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldBundleGetListAll
    module procedure ESMF_FieldBundleGetItem
    module procedure ESMF_FieldBundleGetList
    module procedure ESMF_FieldBundleGetIndex
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldBundleGetDataPtr1DI1 
 module procedure ESMF_FieldBundleGetDataPtr2DI1 
 module procedure ESMF_FieldBundleGetDataPtr3DI1 
 module procedure ESMF_FieldBundleGetDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleGetDataPtr5DI1 
 module procedure ESMF_FieldBundleGetDataPtr6DI1 
 module procedure ESMF_FieldBundleGetDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldBundleGetDataPtr1DI2 
 module procedure ESMF_FieldBundleGetDataPtr2DI2 
 module procedure ESMF_FieldBundleGetDataPtr3DI2 
 module procedure ESMF_FieldBundleGetDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleGetDataPtr5DI2 
 module procedure ESMF_FieldBundleGetDataPtr6DI2 
 module procedure ESMF_FieldBundleGetDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldBundleGetDataPtr1DI4 
 module procedure ESMF_FieldBundleGetDataPtr1DI8 
 module procedure ESMF_FieldBundleGetDataPtr1DR4 
 module procedure ESMF_FieldBundleGetDataPtr1DR8 
 module procedure ESMF_FieldBundleGetDataPtr2DI4 
 module procedure ESMF_FieldBundleGetDataPtr2DI8 
 module procedure ESMF_FieldBundleGetDataPtr2DR4 
 module procedure ESMF_FieldBundleGetDataPtr2DR8 
 module procedure ESMF_FieldBundleGetDataPtr3DI4 
 module procedure ESMF_FieldBundleGetDataPtr3DI8 
 module procedure ESMF_FieldBundleGetDataPtr3DR4 
 module procedure ESMF_FieldBundleGetDataPtr3DR8 
 module procedure ESMF_FieldBundleGetDataPtr4DI4 
 module procedure ESMF_FieldBundleGetDataPtr4DI8 
 module procedure ESMF_FieldBundleGetDataPtr4DR4 
 module procedure ESMF_FieldBundleGetDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldBundleGetDataPtr5DI4 
 module procedure ESMF_FieldBundleGetDataPtr5DI8 
 module procedure ESMF_FieldBundleGetDataPtr5DR4 
 module procedure ESMF_FieldBundleGetDataPtr5DR8 
 module procedure ESMF_FieldBundleGetDataPtr6DI4 
 module procedure ESMF_FieldBundleGetDataPtr6DI8 
 module procedure ESMF_FieldBundleGetDataPtr6DR4 
 module procedure ESMF_FieldBundleGetDataPtr6DR8 
 module procedure ESMF_FieldBundleGetDataPtr7DI4 
 module procedure ESMF_FieldBundleGetDataPtr7DI8 
 module procedure ESMF_FieldBundleGetDataPtr7DR4 
 module procedure ESMF_FieldBundleGetDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!EOPI

  end interface

! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_FieldBundleSet -- Generic interface

! !INTERFACE:
  interface ESMF_FieldBundleSet

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldBundleSetGeom
    module procedure ESMF_FieldBundleSetGrid
    module procedure ESMF_FieldBundleSetLS
    module procedure ESMF_FieldBundleSetMesh
    module procedure ESMF_FieldBundleSetXGrid
!EOPI

  end interface

! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_FieldBundleRedist -- Generic interface

! !INTERFACE:
  interface ESMF_FieldBundleRedistStore

! !PRIVATE MEMBER FUNCTIONS:
    module procedure ESMF_FieldBundleRedistStoreI4
    module procedure ESMF_FieldBundleRedistStoreI8
    module procedure ESMF_FieldBundleRedistStoreR4
    module procedure ESMF_FieldBundleRedistStoreR8
    module procedure ESMF_FieldBundleRedistStoreNF
!
!EOPI

  end interface

! -------------------------- ESMF-public method -------------------------------
!BOPI
! !IROUTINE: ESMF_FieldBundleSMM -- Generic interface

! !INTERFACE:
  interface ESMF_FieldBundleSMMStore

! !PRIVATE MEMBER FUNCTIONS:
    module procedure ESMF_FieldBundleSMMStoreI4
    module procedure ESMF_FieldBundleSMMStoreI8
    module procedure ESMF_FieldBundleSMMStoreR4
    module procedure ESMF_FieldBundleSMMStoreR8
    module procedure ESMF_FieldBundleSMMStoreNF
    module procedure ESMF_FieldBundleSMMStoreFromFile
!
!EOPI

  end interface

!===============================================================================
! FieldBundleOperator() interfaces
!===============================================================================

! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_FieldBundleAssignment(=) - FieldBundle assignment
!
! !INTERFACE:
! interface assignment(=)
! fieldbundle1 = fieldbundle2
!
! !ARGUMENTS:
! type(ESMF_FieldBundle) :: fieldbundle1
! type(ESMF_FieldBundle) :: fieldbundle2
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Assign fieldbundle1 as an alias to the same ESMF fieldbundle object in memory
! as fieldbundle2. If fieldbundle2 is invalid, then fieldbundle1 will be equally invalid after
! the assignment.
!
! The arguments are:
! \begin{description}
! \item[fieldbundle1]
! The {\tt ESMF\_FieldBundle} object on the left hand side of the assignment.
! \item[fieldbundle2]
! The {\tt ESMF\_FieldBundle} object on the right hand side of the assignment.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_FieldBundleOperator(==) - FieldBundle equality operator
!
! !INTERFACE:
  interface operator(==)
! if (fieldbundle1 == fieldbundle2) then ... endif
! OR
! result = (fieldbundle1 == fieldbundle2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_FieldBundle), intent(in) :: fieldbundle1
! type(ESMF_FieldBundle), intent(in) :: fieldbundle2
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Test whether fieldbundle1 and fieldbundle2 are valid aliases to the same ESMF
! fieldbundle object in memory. For a more general comparison of two ESMF FieldBundles,
! going beyond the simple alias test, the ESMF\_FieldBundleMatch() function (not yet
! implemented) must be used.
! \end{sloppypar}
!
! The arguments are:
! \begin{description}
! \item[fieldbundle1]
! The {\tt ESMF\_FieldBundle} object on the left hand side of the equality
! operation.
! \item[fieldbundle2]
! The {\tt ESMF\_FieldBundle} object on the right hand side of the equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_FieldBundleEQ
    module procedure ESMF_FieldBundleStatusEQ

  end interface
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_FieldBundleOperator(/=) - FieldBundle not equal operator
!
! !INTERFACE:
  interface operator(/=)
! if (fieldbundle1 /= fieldbundle2) then ... endif
! OR
! result = (fieldbundle1 /= fieldbundle2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_FieldBundle), intent(in) :: fieldbundle1
! type(ESMF_FieldBundle), intent(in) :: fieldbundle2
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Test whether fieldbundle1 and fieldbundle2 are {\it not} valid aliases to the
! same ESMF fieldbundle object in memory. For a more general comparison of two ESMF
! FieldBundles, going beyond the simple alias test, the ESMF\_FieldBundleMatch() function
! (not yet implemented) must be used.
! \end{sloppypar}
!
! The arguments are:
! \begin{description}
! \item[fieldbundle1]
! The {\tt ESMF\_FieldBundle} object on the left hand side of the non-equality
! operation.
! \item[fieldbundle2]
! The {\tt ESMF\_FieldBundle} object on the right hand side of the non-equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_FieldBundleNE
    module procedure ESMF_FieldBundleStatusNE

  end interface
!------------------------------------------------------------------------------


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

contains

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleEQ()"
!BOPI
! !IROUTINE: ESMF_FieldBundleEQ - Compare two FieldBundles for equality
!
! !INTERFACE:
  impure elemental function ESMF_FieldBundleEQ(fieldbundle1, fieldbundle2)
!
! !RETURN VALUE:
    logical :: ESMF_FieldBundleEQ

! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle1
    type(ESMF_FieldBundle), intent(in) :: fieldbundle2

! !DESCRIPTION:
! Test if both {\tt fieldbundle1} and {\tt fieldbundle2} alias the same ESMF fieldbundle
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE fbinit1, fbinit2
    integer :: localrc1, localrc2
    logical :: lval1, lval2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ! TODO: Consider moving this logic to C++: use Base class? status?
    ! Or replicate logic for C interface also.

    ! check inputs
    fbinit1 = ESMF_FieldBundleGetInit(fieldbundle1)
    fbinit2 = ESMF_FieldBundleGetInit(fieldbundle2)

    ! TODO: this line must remain split in two for SunOS f90 8.3 127000-03
    if (fbinit1 .eq. ESMF_INIT_CREATED .and. &
      fbinit2 .eq. ESMF_INIT_CREATED) then
      ESMF_FieldBundleEQ = associated(fieldbundle1%this,fieldbundle2%this)
    else
      ESMF_FieldBundleEQ = ESMF_FALSE
    endif

  end function ESMF_FieldBundleEQ
!-------------------------------------------------------------------------------


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleNE()"
!BOPI
! !IROUTINE: ESMF_FieldBundleNE - Compare two FieldBundles for non-equality
!
! !INTERFACE:
  impure elemental function ESMF_FieldBundleNE(fieldbundle1, fieldbundle2)
!
! !RETURN VALUE:
    logical :: ESMF_FieldBundleNE

! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle1
    type(ESMF_FieldBundle), intent(in) :: fieldbundle2

! !DESCRIPTION:
! Test if both {\tt fieldbundle1} and {\tt fieldbundle2} alias the same ESMF fieldbundle
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE fbinit1, fbinit2
    integer :: localrc1, localrc2
    logical :: lval1, lval2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ESMF_FieldBundleNE = .not.ESMF_FieldBundleEQ(fieldbundle1, fieldbundle2)

  end function ESMF_FieldBundleNE
!-------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleAddList()"
!BOP
! !IROUTINE: ESMF_FieldBundleAdd - Add Fields to a FieldBundle
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleAdd()
    subroutine ESMF_FieldBundleAddList(fieldbundle, fieldList, keywordEnforcer, &
      multiflag, relaxedflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_Field), intent(in) :: fieldList(:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: multiflag
    logical, intent(in), optional :: relaxedflag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Add Field(s) to a FieldBundle. It is an error if {\tt fieldList} contains
! Fields that match by name Fields already contained in
! {\tt fieldbundle} when multiflag
! is set to {\tt .false.} and relaxedflag is set to {\tt .false.}.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be added to.
! \item [fieldList]
! List of {\tt ESMF\_Field} objects to be added.
! \item [{[multiflag]}]
! A setting of {\tt .true.} allows multiple items with the same name
! to be added to {\tt ESMF\_FieldBundle}. For {\tt .false.} added items must
! have unique names. The default setting is {\tt .false.}.
! \item [{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "add"
! under {\tt multiflag=.false.} mode, where it is {\em not} an error if
! {\tt fieldList} contains items with names that are also found in
! {\tt ESMF\_FieldBundle}. The {\tt ESMF\_FieldBundle} is left unchanged for these items.
! For {\tt .false.} this is treated as an error condition.
! The default setting is {\tt .false.}.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    ! Cannot add Field to a packed FieldBundle
    if(fieldbundle%this%is_packed) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - Fields cannot be added to packed FieldBundle after it is created.", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    call ESMF_FieldBundleAddListInternal(fieldbundle, fieldList, &
      multiflag=multiflag, relaxedflag = relaxedflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleAddList

    ! Private name; call using ESMF_FieldBundleAdd()
    subroutine ESMF_FieldBundleAddListInternal(fieldbundle, fieldList, &
      multiflag, relaxedflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_Field), intent(in) :: fieldList(:)
    logical, intent(in), optional :: multiflag
    logical, intent(in), optional :: relaxedflag
    integer, intent(out), optional :: rc

    integer :: localrc ! local return code
    integer :: fieldCount, i, j, addedIndex, garbageSize
    type(ESMF_Logical) :: linkChange
    type(ESMF_Field), pointer :: garbageList(:), addedList(:)
    logical :: isGarbage

    ! Determine the number of FieldList elements
    fieldCount = size(fieldList)

    call ESMF_ContainerGarbageOn(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageClear(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerAdd(fieldbundle%this%container, fieldList, &
      multiflag=multiflag, relaxedflag=relaxedflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    nullify(garbageList)
    call ESMF_ContainerGarbageGet(fieldbundle%this%container, garbageList=garbageList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageOff(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! There are garbage, some fields are not added.
    ! Deduce the list of Field actually got added.
    nullify(addedList)
    garbageSize = size(garbageList)
    if(garbageSize /= 0) then

      ! There are garbage after ContainerAdd, figure out which fields are added
      ! error checking
      if(garbageSize .gt. fieldCount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - there are more garbage in garbageList than fields in FieldList", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      ! at least 1 Field was added
      ! No need to handle the case when garbageSize == fieldCount when no Field is added
      if(garbageSize .lt. fieldCount) then

        allocate(addedList(fieldCount - garbageSize), stat=localrc)
        if(localrc /= 0) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
            msg = " - cannot allocate addedList", &
            ESMF_CONTEXT, rcToReturn=rc)
          return
        endif

        ! TODO: this is a performance bottlnect
        ! if this causes problem, the container add method should return
        ! a list of field actually are added
        addedIndex = 1
        do i = 1, fieldCount
          isGarbage = .false.
          do j = 1, garbageSize
            if(fieldList(i) == garbageList(j)) isGarbage = .true.
          enddo

          if(.not. isGarbage) then
            addedList(addedIndex) = fieldList(i)
            addedIndex = addedIndex + 1
          endif
        enddo

        if(size(addedList) .ge. 1) then
          call ESMF_FieldBundleSetGeom(fieldbundle, addedList(1), rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
                        ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        deallocate(addedList)

      endif ! partial add

    else
      ! No garbage, all fieldList should be linked
      !
      ! Obviously the attribute linking of the geom object is a bit tricky,
      ! right now, we do not restrict the geom to be matched among the fields in the
      ! fieldBundle. But for attribute hierarchy, we actually assume that all fields in
      ! the fieldBundle should be built on the same geom object.
      !
      ! 3/7/13
      ! New behavior for geom:
      ! 1. FieldBundleSet() can always change the geom object in a FieldBundle
      ! 2. Add/AddReplace/Replace can change the geom object in a FieldBundle if it is not set
      ! 3. Add/AddReplace/Replace can NOT change geom object in a FieldBundle after its set
      !
      ! Attribute link
      if(size(fieldList) .ge. 1 .and. (fieldbundle%this%status /= ESMF_FBSTATUS_GRIDSET) ) then
        ! setgeom links grid geom automatically
        call ESMF_FieldBundleSetGeom(fieldbundle, fieldList(1), rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif ! non-empty fieldlist

    endif ! garbageSize /= 0

    ! always deallocate garbageList because it is always associated returning from Container
    deallocate(garbageList)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleAddListInternal
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleAddItem()"
!BOPI
! !IROUTINE: ESMF_FieldBundleAdd - Add one Field to a FieldBundle
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleAdd()
    subroutine ESMF_FieldBundleAddItem(fieldbundle, field, keywordEnforcer, &
      multiflag, relaxedflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_Field), intent(in) :: field
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: multiflag
    logical, intent(in), optional :: relaxedflag
    integer, intent(out), optional :: rc
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Add a single field to a FieldBundle. It is an error if {\tt field}
! match by name to what is already contained in {\tt fieldbundle} when multiflag
! is set to {\tt .false.} and relaxedflag is set to {\tt .false.} by default.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be added to.
! \item [field]
! {\tt ESMF\_Field} object to be added.
! \item [{[multiflag]}]
! A setting of {\tt .true.} allows multiple items with the same name
! to be added to {\tt ESMF\_FieldBundle}. For {\tt .false.} added items must
! have unique names. The default setting is {\tt .false.}.
! \item [{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "add"
! under {\tt multiflag=.false.} mode, where it is {\em not} an error if
! {\tt fieldList} contains items with names that are also found in
! {\tt ESMF\_FieldBundle}. The {\tt ESMF\_FieldBundle} is left unchanged for these items.
! For {\tt .false.} this is treated as an error condition.
! The default setting is {\tt .false.}.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Call into the list version
    call ESMF_FieldBundleAddList(fieldbundle, (/field/), multiflag=multiflag, &
      relaxedflag=relaxedflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleAddItem
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleAddReplace()"
!BOP
! !IROUTINE: ESMF_FieldBundleAddReplace - Conditionally add or replace Fields in a FieldBundle
!
! !INTERFACE:
    subroutine ESMF_FieldBundleAddReplace(fieldbundle, fieldList, keywordEnforcer, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_Field), intent(in) :: fieldList(:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Fields in {\tt fieldList} that do not match any Fields by name in
! {\tt fieldbundle} are added to the FieldBundle. Fields in {\tt fieldList}
! that match any Fields by name in {\tt fieldbundle} replace those Fields.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be manipulated.
! \item [fieldList]
! List of {\tt ESMF\_Field} objects to be added or used as replacement.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: fieldCount, i
    type(ESMF_Field), pointer :: garbageList(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    ! Cannot add Field to a packed FieldBundle
    if(fieldbundle%this%is_packed) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - Fields cannot be added to packed FieldBundle after it is created.", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    ! Determine the number of FieldList elements
    fieldCount = size(fieldList)
    nullify(garbageList)

    call ESMF_ContainerGarbageOn(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageClear(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Call into container addreplace
    call ESMF_ContainerAddReplace(fieldbundle%this%container, fieldList, &
      rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageGet(fieldbundle%this%container, garbageList=garbageList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageOff(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Attribute link
    if(size(fieldList) .ge. 1 .and. (fieldbundle%this%status /= ESMF_FBSTATUS_GRIDSET) ) then
      ! setgeom links grid geom automatically
      call ESMF_FieldBundleSetGeom(fieldbundle, fieldList(1), rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif ! non-empty fieldlist

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleAddReplace
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleCreateDefault()"
!BOP
! !IROUTINE: ESMF_FieldBundleCreate - Create a non packed FieldBundle from a list of Fields
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleCreate()
  function ESMF_FieldBundleCreateDefault(keywordEnforcer, fieldList, &
      multiflag, relaxedflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateDefault
!
! !ARGUMENTS:
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Field), intent(in), optional :: fieldList(:)
    logical, intent(in), optional :: multiflag
    logical, intent(in), optional :: relaxedflag
    character (len=*),intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Create an {\tt ESMF\_FieldBundle} object from a list of existing Fields.
!
! The creation of a FieldBundle leaves the bundled Fields unchanged, they
! remain valid individual objects. a FieldBundle is a light weight container
! of Field references. The actual data remains in place, there are no
! data movements or duplications associated with the creation of an
! FieldBundle.
!
! \begin{description}
! \item [{[fieldList]}]
! List of {\tt ESMF\_Field} objects to be bundled.
! \item [{[multiflag]}]
! A setting of {\tt .true.} allows multiple items with the same name
! to be added to {\tt fieldbundle}. For {\tt .false.} added items must
! have unique names. The default setting is {\tt .false.}.
! \item [{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "add"
! under {\tt multiflag=.false.} mode, where it is {\em not} an error if
! {\tt fieldList} contains items with names that are also found in
! {\tt fieldbundle}. The {\tt fieldbundle} is left unchanged for these items.
! For {\tt .false.} this is treated as an error condition.
! The default setting is {\tt .false.}.
! \item [{[name]}]
! Name of the created {\tt ESMF\_FieldBundle}. A default name is generated
! if not specified.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    type(ESMF_FieldBundleType), pointer :: this
    integer :: fieldCount, i
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    type(ESMF_XGrid) :: xgrid
    type(ESMF_Mesh) :: mesh
    type(ESMF_LocStream) :: locstream
    type(ESMF_FieldStatus_Flag) :: fstatus

    ! Initialize return code
    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    localrc = ESMF_RC_NOT_IMPL

    ! Determine the number of FieldList elements
    if (present(fieldList)) then
      fieldCount = size(fieldList)
    else
      fieldCount = 0
    endif

    ! Check init status of arguments
    do i=1, fieldCount
      ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit, fieldList(i), rc)
    enddo

    ! Initialize
    nullify(this)
    nullify(ESMF_FieldBundleCreateDefault%this)

    ! Create the internal objects
    allocate(this, stat=localrc)
    if (ESMF_LogFoundAllocError(localrc, &
        msg="- Allocating FieldBundle Type", &
        ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    this%container = ESMF_ContainerCreate(rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    this%status = ESMF_FBSTATUS_EMPTY

    ! Set up proxy flag
    this%is_proxy = .false.
    this%is_packed = .false.

    ! Set return value
    ESMF_FieldBundleCreateDefault%this => this

    ! Add reference to this object into ESMF garbage collection table
    ! Only call this in those Create() methods that call Construct()
    call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateDefault, ESMF_ID_FIELDBUNDLE%objectID)

    ! Set init code
    ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateDefault)

    if(present(fieldList)) then
      call ESMF_FieldBundleAdd(ESMF_FieldBundleCreateDefault, fieldList, &
        multiflag=multiflag, relaxedflag=relaxedflag, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldBundleCreateDefault
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldBundleCreate - Create a packed FieldBundle from Fortran array pointer and Grid 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldBundleCreate() 
! function ESMF_FieldBundleCreateGrid<rank><type><kind>(fieldNameList, & 
! farrayPtr, grid, fieldDim, keywordEnforcer, & 
! indexflag, staggerLoc, & 
! gridToFieldMap, & 
! totalLWidth, totalUWidth, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! character(len=*), intent(in) :: fieldNameList(:) 
! <type> (ESMF_KIND_<kind>), dimension(<rank>), pointer :: farrayPtr 
! type(ESMF_Grid), intent(in) :: grid 
! integer, intent(in) :: fieldDim 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_Index_Flag), intent(in), optional :: indexflag 
! type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! integer, intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create a packed FieldBundle from user supplied list of field names, pre-allocated 
! Fortran array pointer, and {\tt ESMF\_Grid} object. 
! 
! The arguments are: 
! \begin{description} 
! \item [fieldNameList] 
! A list of field names for the Fields held by the packed {\tt FieldBundle}. 
! \item[farrayPtr] 
! Pre-allocated Fortran array pointer holding the memory of the list of Fields. 
! \item [grid] 
! The {\tt ESMF\_Grid} object on which the Fields in the packed FieldBundle are built. 
! \item [fieldDim] 
! The dimension in the {\tt farrayPtr} that contains the indices of Fields to be packed. 
! \item [{[indexflag]}] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! All Fields in packed FieldBundle use identical indexflag setting. 
! \item [{[staggerloc]}] 
! Stagger location of data in grid cells. For valid 
! predefined values see section \ref{const:staggerloc}. 
! To create a custom stagger location see section 
! \ref{sec:usage:staggerloc:adv}. The default 
! value is {\tt ESMF\_STAGGERLOC\_CENTER}. 
! All Fields in packed FieldBundle use identical staggerloc setting. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. 
! The default is to 
! map all of the {\tt grid}'s dimensions against the lowest dimensions of 
! the {\tt farrayPtr} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farrayPtr} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farrayPtr} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt grid}. Ungridded dimensions must be in the same order they are 
! stored in the {\t farrayPtr}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! All Fields in packed FieldBundle use identical gridToFieldMap setting. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farrayPtr}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt farrayPtr}. That is, for each gridded dimension the 
! {\tt farrayPtr} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! All Fields in packed FieldBundle use identical totalLWidth setting. 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farrayPtr}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt farrayPtr}. That is, for each gridded dimension the 
! {\tt farrayPtr} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! All Fields in packed FieldBundle use identical totalUWidth setting. 
! \item [{[name]}] 
! FieldBundle name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr1Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr1Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr1Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr1Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr1Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr1Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr1Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr2Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr2Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr2Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr2Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr2Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr2Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr2Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr3Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr3Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr3Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr3Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr3Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr3Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr3Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr4Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr4Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr4Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr4Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr4Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr4Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr4Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr5Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr5Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr5Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr5Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr5Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr5Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr5Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr6Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr6Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr6Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr6Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr6Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr6Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr6Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr7Di1(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr7Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr7Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr7Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr7Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr7Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr7Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr1Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr1Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr1Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr1Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr1Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr1Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr1Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr2Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr2Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr2Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr2Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr2Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr2Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr2Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr3Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr3Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr3Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr3Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr3Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr3Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr3Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr4Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr4Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr4Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr4Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr4Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr4Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr4Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr5Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr5Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr5Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr5Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr5Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr5Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr5Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr6Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr6Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr6Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr6Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr6Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr6Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr6Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr7Di2(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr7Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr7Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr7Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr7Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr7Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr7Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr1Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr1Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr1Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr1Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr1Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr1Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr1Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr1Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr1Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr1Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr1Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr1Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr1Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr1Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr1Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr1Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr1Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr1Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr1Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr1Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr1Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr1Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr1Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr1Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr1Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr1Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr1Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr1Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr2Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr2Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr2Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr2Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr2Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr2Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr2Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr2Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr2Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr2Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr2Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr2Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr2Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr2Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr2Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr2Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr2Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr2Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr2Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr2Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr2Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr2Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr2Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr2Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr2Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr2Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr2Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr2Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr3Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr3Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr3Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr3Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr3Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr3Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr3Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr3Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr3Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr3Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr3Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr3Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr3Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr3Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr3Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr3Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr3Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr3Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr3Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr3Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr3Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr3Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr3Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr3Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr3Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr3Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr3Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr3Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr4Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr4Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr4Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr4Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr4Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr4Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr4Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr4Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr4Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr4Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr4Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr4Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr4Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr4Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr4Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr4Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr4Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr4Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr4Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr4Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr4Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr4Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr4Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr4Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr4Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr4Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr4Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr4Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr5Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr5Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr5Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr5Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr5Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr5Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr5Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr5Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr5Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr5Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr5Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr5Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr5Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr5Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr5Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr5Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr5Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr5Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr5Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr5Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr5Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr5Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr5Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr5Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr5Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr5Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr5Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr5Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr6Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr6Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr6Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr6Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr6Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr6Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr6Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr6Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr6Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr6Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr6Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr6Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr6Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr6Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr6Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr6Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr6Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr6Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr6Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr6Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr6Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr6Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr6Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr6Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr6Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr6Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr6Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr6Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr7Di4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr7Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr7Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr7Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr7Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr7Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr7Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr7Di8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr7Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr7Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr7Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr7Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr7Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr7Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr7Dr4(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr7Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr7Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr7Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr7Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr7Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr7Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr7Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateGridDataPtr" 
 function ESMF_FieldBundleCreateGridDataPtr7Dr8(fieldNameList, & 
 farrayPtr, grid, fieldDim, keywordEnforcer, & 
 indexflag, staggerLoc, & 
 gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateGridDataPtr7Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Grid), intent(in) :: grid 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_Index_Flag), intent(in), optional :: indexflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateGridDataPtr7Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateGridDataPtr7Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateGridDataPtr7Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateGridDataPtr7Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 ! Create an internal g2f map 
 if(present(gridToFieldMap)) then 
 allocate(ig2fmap(size(gridToFieldMap))) 
 do i = 1, size(gridToFieldMap) 
 if(gridToFieldMap(i) < fieldDim) then 
 ig2fmap(i) = gridToFieldMap(i) 
 else 
 ig2fmap(i) = gridToFieldMap(i)+1 
 endif 
 enddo 
 else 
 call ESMF_GridGet(grid, gridDimCount, rc=rc) 
 allocate(ig2fmap(gridDimCount)) 
 do i = 1, gridDimCount 
 if(i < fieldDim) then 
 ig2fmap(i) = i 
 else 
 ig2fmap(i) = i+1 
 endif 
 enddo 
 endif 

 fieldList(1) = ESMF_FieldCreate(grid, farrayPtr, staggerloc=staggerloc, & 
 gridToFieldMap=gridToFieldMap, totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateGridDataPtr7Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateGridDataPtr7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!----------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldBundleCreate - Create a packed FieldBundle from Fortran array pointer and Mesh 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldBundleCreate() 
! function ESMF_FieldBundleCreateMesh<rank><type><kind>(fieldNameList, & 
! farrayPtr, Mesh, fieldDim, keywordEnforcer, & 
! meshLoc, gridToFieldMap, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! character(len=*), intent(in) :: fieldNameList(:) 
! <type> (ESMF_KIND_<kind>), dimension(<rank>), pointer :: farrayPtr 
! type(ESMF_Mesh), intent(in) :: mesh 
! integer, intent(in) :: fieldDim 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_MeshLoc), intent(in), optional:: meshloc 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create a packed FieldBundle from user supplied list of field names, pre-allocated 
! Fortran array pointer, and {\tt ESMF\_Mesh} object. 
! 
! The arguments are: 
! \begin{description} 
! \item [fieldNameList] 
! A list of field names for the Fields held by the packed {\tt FieldBundle}. 
! \item[farrayPtr] 
! Pre-allocated Fortran array pointer holding the memory of the list of Fields. 
! \item [mesh] 
! The {\tt ESMF\_Mesh} object on which the Fields in the packed FieldBundle are built. 
! \item [fieldDim] 
! The dimension in the {\tt farrayPtr} that contains the indices of Fields to be packed. 
! \item [{[meshloc]}] 
! \begin{sloppypar} 
! The part of the Mesh on which to build the Field. For valid 
! predefined values see Section~\ref{const:meshloc}. 
! If not set, defaults to {\tt ESMF\_MESHLOC\_NODE}. 
! \end{sloppypar} 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt mesh}'s dimCount. The list elements map each dimension 
! of the {\tt mesh} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. 
! The default is to 
! map all of the {\tt mesh}'s dimensions against the lowest dimensions of 
! the {\tt farrayPtr} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farrayPtr} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farrayPtr} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt mesh}. Ungridded dimensions must be in the same order they are 
! stored in the {\t farrayPtr}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! All Fields in packed FieldBundle use identical gridToFieldMap setting. 
! \item [{[name]}] 
! FieldBundle name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr1Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr1Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr1Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr1Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr1Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr1Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr1Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr2Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr2Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr2Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr2Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr2Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr2Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr2Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr3Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr3Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr3Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr3Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr3Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr3Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr3Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr4Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr4Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr4Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr4Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr4Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr4Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr4Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr5Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr5Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr5Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr5Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr5Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr5Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr5Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr6Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr6Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr6Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr6Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr6Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr6Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr6Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr7Di1(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr7Di1 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr7Di1%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr7Di1%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr7Di1, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr7Di1) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr7Di1, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr1Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr1Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr1Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr1Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr1Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr1Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr1Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr2Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr2Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr2Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr2Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr2Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr2Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr2Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr3Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr3Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr3Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr3Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr3Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr3Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr3Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr4Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr4Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr4Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr4Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr4Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr4Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr4Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr5Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr5Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr5Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr5Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr5Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr5Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr5Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr6Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr6Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr6Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr6Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr6Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr6Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr6Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr7Di2(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr7Di2 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr7Di2%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr7Di2%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr7Di2, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr7Di2) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr7Di2, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr1Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr1Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr1Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr1Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr1Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr1Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr1Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr1Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr1Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr1Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr1Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr1Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr1Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr1Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr1Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr1Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr1Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr1Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr1Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr1Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr1Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr1Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr1Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr1Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr1Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr1Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr1Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr1Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr2Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr2Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr2Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr2Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr2Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr2Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr2Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr2Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr2Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr2Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr2Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr2Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr2Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr2Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr2Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr2Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr2Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr2Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr2Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr2Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr2Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr2Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr2Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr2Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr2Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr2Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr2Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr2Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr3Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr3Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr3Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr3Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr3Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr3Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr3Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr3Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr3Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr3Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr3Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr3Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr3Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr3Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr3Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr3Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr3Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr3Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr3Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr3Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr3Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr3Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr3Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr3Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr3Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr3Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr3Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr3Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr4Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr4Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr4Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr4Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr4Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr4Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr4Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr4Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr4Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr4Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr4Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr4Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr4Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr4Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr4Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr4Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr4Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr4Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr4Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr4Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr4Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr4Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr4Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr4Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr4Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr4Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr4Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr4Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr5Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr5Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr5Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr5Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr5Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr5Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr5Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr5Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr5Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr5Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr5Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr5Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr5Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr5Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr5Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr5Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr5Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr5Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr5Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr5Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr5Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr5Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr5Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr5Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr5Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr5Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr5Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr5Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr6Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr6Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr6Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr6Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr6Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr6Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr6Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr6Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr6Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr6Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr6Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr6Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr6Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr6Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr6Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr6Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr6Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr6Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr6Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr6Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr6Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr6Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr6Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr6Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr6Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr6Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr6Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr6Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr7Di4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr7Di4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr7Di4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr7Di4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr7Di4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr7Di4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr7Di4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr7Di8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr7Di8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr7Di8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr7Di8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr7Di8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr7Di8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr7Di8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr7Dr4(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr7Dr4 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr7Dr4%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr7Dr4%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr7Dr4, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr7Dr4) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr7Dr4, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr7Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleCreateMeshDataPtr" 
 function ESMF_FieldBundleCreateMeshDataPtr7Dr8(fieldNameList, & 
 farrayPtr, mesh, fieldDim, keywordEnforcer, & 
 meshLoc, gridToFieldMap, name, rc) 

 ! return value 
 type(ESMF_FieldBundle) :: ESMF_FieldBundleCreateMeshDataPtr7Dr8 

 !ARGUMENTS: 
 character(len=*), intent(in) :: fieldNameList(:) 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer, intent(in) :: fieldDim 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_MeshLoc), intent(in), optional:: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character(len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 integer :: gridDimCount, fieldCount 
 type(ESMF_Field) :: fieldList(1) 
 integer, allocatable :: ig2fmap(:) 
 character(len=:), allocatable :: encodeName 
 integer :: i, localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Initialize 
 nullify(this) 
 nullify(ESMF_FieldBundleCreateMeshDataPtr7Dr8%this) 
 allocate(character(1024) :: encodeName) ! Initial allocation 

 if (size(fieldNameList) /= size(farrayPtr, fieldDim) ) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg = " - size of fieldNameList is not equal to size of the fieldCount indicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 fieldCount = size(fieldNameList) 

 ! Create the internal objects 
 allocate(this, stat=localrc) 
 if (ESMF_LogFoundAllocError(localrc, & 
 msg="- Allocating FieldBundle Type", & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_BaseCreate(this%base, "FieldBundle", name, 0, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%container = ESMF_ContainerCreate(rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 this%status = ESMF_FBSTATUS_EMPTY 

 ! Set up proxy flag 
 this%is_proxy = .false. 
 this%is_packed = .true. 

 ! Set return value 
 ESMF_FieldBundleCreateMeshDataPtr7Dr8%this => this 

 ! Add reference to this object into ESMF garbage collection table 
 ! Only call this in those Create() methods that call Construct() 
 call c_ESMC_VMAddFObject(ESMF_FieldBundleCreateMeshDataPtr7Dr8, ESMF_ID_FIELDBUNDLE%objectID) 

 ! Set init code 
 ESMF_INIT_SET_CREATED(ESMF_FieldBundleCreateMeshDataPtr7Dr8) 

 encodeName = trim(fieldNameList(1)) 
 do i = 2, size(fieldNameList) 
 encodeName = concatName(trim(encodeName), ':', trim(fieldNameList(i))) 
 enddo 
 this%encodeName = encodeName 

 fieldList(1) = ESMF_FieldCreate(mesh, farrayPtr, meshloc=meshloc, & 
 gridToFieldMap=gridToFieldMap, & 
 name='packedFieldBundleField', rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldBundleAddListInternal( & 
 ESMF_FieldBundleCreateMeshDataPtr7Dr8, fieldList, & 
 multiflag=.false., relaxedflag=.false., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end function ESMF_FieldBundleCreateMeshDataPtr7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleDestroy()"
!BOP
! !IROUTINE: ESMF_FieldBundleDestroy - Release resources associated with a FieldBundle

! !INTERFACE:
  subroutine ESMF_FieldBundleDestroy(fieldbundle, keywordEnforcer, noGarbage, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: noGarbage
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[7.0.0] Added argument {\tt noGarbage}.
! The argument provides a mechanism to override the default garbage collection
! mechanism when destroying an ESMF object.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Destroy an {\tt ESMF\_FieldBundle} object. The member Fields are not
! touched by this operation and remain valid objects that need to be
! destroyed individually if necessary.
!
! The arguments are:
! \begin{description}
! \item[fieldbundle]
! {\tt ESMF\_FieldBundle} object to be destroyed.
! \item[{[noGarbage]}]
! If set to {\tt .TRUE.} the object will be fully destroyed and removed
! from the ESMF garbage collection system. Note however that under this
! condition ESMF cannot protect against accessing the destroyed object
! through dangling aliases -- a situation which may lead to hard to debug
! application crashes.
!
! It is generally recommended to leave the {\tt noGarbage} argument
! set to {\tt .FALSE.} (the default), and to take advantage of the ESMF
! garbage collection system which will prevent problems with dangling
! aliases or incorrect sequences of destroy calls. However this level of
! support requires that a small remnant of the object is kept in memory
! past the destroy call. This can lead to an unexpected increase in memory
! consumption over the course of execution in applications that use
! temporary ESMF objects. For situations where the repeated creation and
! destruction of temporary objects leads to memory issues, it is
! recommended to call with {\tt noGarbage} set to {\tt .TRUE.}, fully
! removing the entire temporary object from memory.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! Initialize return code
    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    localrc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    ! more input checking
    if (.not.associated(fieldbundle%this)) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg="Uninitialized or already destroyed FieldBundle: this pointer unassociated", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    ! Destruct all fieldbundle internals and then free fieldbundle memory.
    call ESMF_FieldBundleDestruct(fieldbundle%this, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if (present(noGarbage)) then
      if (noGarbage) then
        ! destroy Base object (which also removes it from garbage collection)
        call ESMF_BaseDestroy(fieldbundle%this%base, noGarbage, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        ! remove reference to this object from ESMF garbage collection table
        call c_ESMC_VMRmFObject(fieldbundle)
        ! deallocate the actual fieldbundle data structure
        deallocate(fieldbundle%this, stat=localrc)
        if (ESMF_LogFoundDeallocError(localrc, &
          msg="Deallocating FieldBundle information", &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif
    endif

    ! Mark this fieldbundle as invalid
    nullify(fieldbundle%this)

    ! Set init code
    ESMF_INIT_SET_DELETED(fieldbundle)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleDestroy
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleDestruct()"
!BOPI
! !IROUTINE: ESMF_FieldBundleDestruct - Destruct the FieldBundleType
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleDestruct()
  subroutine ESMF_FieldBundleDestruct(this, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundleType), pointer :: this
    integer, intent(out), optional :: rc
!
!
! !DESCRIPTION:
! Destruct the FieldBundleType
!
! \begin{description}
! \item [this]
! {\tt ESMF\_FieldBundleType} to be destructed.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI
!------------------------------------------------------------------------------

    integer :: localrc, fcount, i
    type(ESMF_Status) :: basestatus
    type(ESMF_Field), pointer :: flist(:)

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    call ESMF_BaseGetStatus(this%base, basestatus, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if (basestatus .eq. ESMF_STATUS_READY) then

      ! Destroy internal geom and mark this fieldBundle UNINIT
      if(this%status == ESMF_FBSTATUS_GRIDSET) then
        call ESMF_GeomDestroy(this%geom, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        this%status = ESMF_FBSTATUS_UNINIT
      endif

      ! Destroy all the internal Fields if this is a proxy fieldBundle
      ! Destroy internal container
      call ESMF_ContainerDestroy(this%container, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    endif


    ! Mark base object invalid
    call ESMF_BaseSetStatus(this%base, ESMF_STATUS_INVALID, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleDestruct
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleGetListAll()"
!BOP
! !IROUTINE: ESMF_FieldBundleGet - Get object-wide information from a FieldBundle
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleGet()
    subroutine ESMF_FieldBundleGetListAll(fieldbundle, keywordEnforcer, &
      itemorderflag, geomtype, geom, grid, locstream, mesh, xgrid, &
      fieldCount, fieldList, fieldNameList, isPacked, name, vm, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_ItemOrder_Flag), intent(in), optional :: itemorderflag
    type(ESMF_GeomType_Flag), intent(out), optional :: geomtype
    type(ESMF_Geom), intent(out), optional :: geom
    type(ESMF_Grid), intent(out), optional :: grid
    type(ESMF_LocStream), intent(out), optional :: locstream
    type(ESMF_Mesh), intent(out), optional :: mesh
    type(ESMF_XGrid), intent(out), optional :: xgrid
    integer, intent(out), optional :: fieldCount
    type(ESMF_Field), intent(out), optional :: fieldList(:)
    character(len=*), intent(out), optional :: fieldNameList(:)
    logical, intent(out), optional :: isPacked
    character(len=*), intent(out), optional :: name
    type(ESMF_VM), intent(out), optional :: vm
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[6.1.0] Added argument {\tt itemorderflag}.
! The new argument gives the user control over the order in which
! the items are returned.
! \item[8.0.0] Added argument {\tt isPacked}.
! The new argument allows the user to query if this is a packed FieldBundle.
! \item[8.8.0] Added argument {\tt vm} in order to offer information about the
! VM on which the FieldBundle was created.
! \item[8.9.0] Added argument {\tt geom} to allow the user to retrieve a generic representation of
! the FieldBundle's geometry.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Get the list of all Fields and field names bundled in a FieldBundle.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be queried.
! \item [{[itemorderflag]}]
! Specifies the order of the returned items in the {\tt fieldList} or the
! {\tt fieldNameList}.
! The default is {\tt ESMF\_ITEMORDER\_ABC}.
! See \ref{const:itemorderflag} for a full list of options.
! \item[{[geomtype]}]
! Flag that indicates what type of geometry this FieldBundle object holds.
! Can be {\tt ESMF\_GEOMTYPE\_GRID}, {\tt ESMF\_GEOMTYPE\_MESH}, {\tt ESMF\_GEOMTYPE\_LOCSTREAM},
! {\tt ESMF\_GEOMTYPE\_XGRID}
! \item[{[geom]}]
! The Geom object that this FieldBundle object holds.
! \item[{[grid]}]
! The Grid object that this FieldBundle object holds.
! \item[{[locstream]}]
! The LocStream object that this FieldBundle object holds.
! \item[{[mesh]}]
! The Mesh object that this FieldBundle object holds.
! \item[{[xgrid]}]
! The XGrid object that this FieldBundle object holds.
! \item [{[fieldCount]}]
! Upon return holds the number of Fields bundled in the fieldbundle.
! \item [{[fieldList]}]
! Upon return holds a list of Fields bundled in {\tt ESMF\_FieldBundle}. The
! argument must be allocated to be at least of size {\tt fieldCount}.
! \item [{[fieldNameList]}]
! Upon return holds a list of the names of the fields bundled in
! {\tt ESMF\_FieldBundle}. The argument must be allocated to be at least of
! size {\tt fieldCount}.
! \item [{[isPacked]}]
! Upon return holds the information if this FieldBundle is packed.
! \item [{[name]}]
! Name of the FieldBundle object.
! \item [{[vm}]
! The VM on which the FieldBundle object was created.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: l_fieldCount, i ! helper variable
    type(ESMF_Field), pointer :: l_fieldList(:)
    type(ESMF_FieldBundleType), pointer :: this
    type(ESMF_GeomType_Flag) :: l_geomtype
    type(ESMF_ItemOrder_Flag) :: l_itemorderflag
    character(len=:), allocatable :: packedName
    character(len=ESMF_MAXSTR), pointer :: fieldNameListPtr(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    l_itemorderflag = ESMF_ITEMORDER_ABC
    if (present(itemorderflag)) l_itemorderflag = itemorderflag

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    nullify(this)
    nullify(l_fieldList)
    allocate(character(1024) :: packedName) ! Initial allocation

    this => fieldbundle%this

    ! geomtype
    if(present(geomtype)) then
      if(this%status /= ESMF_FBSTATUS_GRIDSET) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - fieldbundle does not have a geom object stored", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      call ESMF_GeomGet(this%geom, geomtype=geomtype, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Get Geom
    if (present(geom)) then
       geom=this%geom
    endif

    ! Get Grid
    if(present(grid)) then
      if(this%status /= ESMF_FBSTATUS_GRIDSET) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - fieldbundle does not have a geom object stored", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      call ESMF_GeomGet(this%geom, grid=grid, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(present(xgrid)) then
      if(this%status /= ESMF_FBSTATUS_GRIDSET) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - fieldbundle does not have a geom object stored", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      call ESMF_GeomGet(this%geom, xgrid=xgrid, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(present(mesh)) then
      if(this%status /= ESMF_FBSTATUS_GRIDSET) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - fieldbundle does not have a geom object stored", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      call ESMF_GeomGet(this%geom, mesh=mesh, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(present(locstream)) then
      if(this%status /= ESMF_FBSTATUS_GRIDSET) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - fieldbundle does not have a geom object stored", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      call ESMF_GeomGet(this%geom, locstream=locstream, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif


    ! Call into the container
    call ESMF_ContainerGet(fieldbundle%this%container, &
      itemCount=l_fieldCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(present(fieldCount)) fieldCount = l_fieldCount

    allocate(l_fieldList(l_fieldCount), stat=localrc)
    if(localrc /= ESMF_SUCCESS) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - cannot allocate l_fieldList internally", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    call ESMF_ContainerGet(fieldbundle%this%container, &
      itemList=l_fieldList, itemorderflag=l_itemorderflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(present(fieldList)) then
      if(size(fieldList) .lt. l_fieldCount .and. (.not. fieldbundle%this%is_packed)) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - Input fieldList size is too small", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
      fieldList(1:l_fieldCount) = l_fieldList(1:l_fieldCount)
    endif

    ! Special call to get name out of Base class
    if (present(fieldNameList)) then
      if(size(fieldNameList) .lt. l_fieldCount .and. (.not. fieldbundle%this%is_packed)) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - Input fieldNameList size is too small", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
      do i = 1, l_fieldCount
        call ESMF_FieldGet(l_fieldList(i), name=fieldNameList(i), rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      enddo
    endif

    ! If fieldbundle is not packed, evaluate the results
    if(fieldbundle%this%is_packed) then
      if(l_fieldCount .ne. 1) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - Inconsistent fieldCount from internal container when fieldbundle is packed.", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
      call decodeName(fieldBundle%this%encodeName, ':', l_fieldCount, fieldNameListPtr)
      if(present(fieldCount)) fieldCount = l_fieldCount
      if(present(fieldList)) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg = " - Field within packed FieldBundle cannot be retrieved directly. Query the data pointer directly.", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
      if (present(fieldNameList)) then
        if(size(fieldNameList) .lt. l_fieldCount) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
            msg = " - Input fieldNameList size is too small", &
            ESMF_CONTEXT, rcToReturn=rc)
          return
        endif
        do i = 1, l_fieldCount
          fieldNameList(i) = fieldNameListPtr(i)
        enddo
      endif
      if(associated(fieldNameListPtr)) deallocate(fieldNameListPtr)
    endif

    deallocate(l_fieldList)

    if (present(isPacked)) then
      isPacked = fieldbundle%this%is_packed
    endif

    if (present(name)) then
      if (fieldbundle%isNamedAlias) then
        name = trim(fieldbundle%name)
      else
        call ESMF_GetName(fieldbundle%this%base, name, localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif
    endif

    if (present(vm)) then
      call ESMF_GetVM(fieldbundle%this%base, vm, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleGetListAll
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleGetItem()"
!BOP
! !IROUTINE: ESMF_FieldBundleGet - Get information about a Field by name and optionally return a Field
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleGet()
    subroutine ESMF_FieldBundleGetItem(fieldbundle, fieldName, &
      keywordEnforcer, field, fieldCount, isPresent, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
    character(len=*), intent(in) :: fieldName
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Field), intent(out), optional :: field
    integer, intent(out), optional :: fieldCount
    logical, intent(out), optional :: isPresent
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Get information about items that match {\tt fieldName} in FieldBundle.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be queried.
! \item [fieldName]
! Specified name.
! \item [{[field]}]
! Upon return holds the requested field item. It is an error if this
! argument was specified and there is not exactly one field item in
! {\tt ESMF\_FieldBundle} that matches {\tt fieldName}.
! \item [{[fieldCount]}]
! Number of Fields with {\tt fieldName} in {\tt ESMF\_FieldBundle}.
! \item [{[isPresent]}]
! Upon return indicates whether field(s) with {\tt fieldName} exist
! in {\tt ESMF\_FieldBundle}.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    if(fieldbundle%this%is_packed) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
msg="This call does not work with packed FieldBundle.",&
          ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    if (present(fieldCount)) then
      call ESMF_ContainerGet(fieldbundle%this%container, &
        itemName=trim(fieldName), itemCount=fieldCount, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(isPresent)) then
      call ESMF_ContainerGet(fieldbundle%this%container, &
        itemName=trim(fieldName), isPresent=isPresent, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(present(fieldCount)) then
      if(fieldCount .gt. 1) then
        if(present(field)) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
msg="field argument cannot be specified when fieldCount is greater than 1",&
              ESMF_CONTEXT, rcToReturn=rc)
          return
        endif
      endif
    endif

    if (present(field)) then
      call ESMF_ContainerGet(fieldbundle%this%container, &
        itemName=trim(fieldName), item=field, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleGetItem
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleGetList()"
!BOP
! !IROUTINE: ESMF_FieldBundleGet - Get a list of Fields by name
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleGet()
    subroutine ESMF_FieldBundleGetList(fieldbundle, fieldName, fieldList, &
      keywordEnforcer, itemorderflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
    character(len=*), intent(in) :: fieldName
    type(ESMF_Field), intent(out) :: fieldList(:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_ItemOrder_Flag), intent(in), optional :: itemorderflag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[6.1.0] Added argument {\tt itemorderflag}.
! The new argument gives the user control over the order in which
! the items are returned.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Get the list of Fields from fieldbundle that match fieldName.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be queried.
! \item [fieldName]
! Specified name.
! \item [fieldList]
! List of Fields in {\tt ESMF\_FieldBundle} that match {\tt fieldName}. The
! argument must be allocated to be at least of size {\tt fieldCount}
! returned for this {\tt fieldName}.
! \item [{[itemorderflag]}]
! Specifies the order of the returned items in the {\tt fieldList}.
! The default is {\tt ESMF\_ITEMORDER\_ABC}.
! See \ref{const:itemorderflag} for a full list of options.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: fieldCount
    type(ESMF_Field), pointer :: l_fieldList(:)
    type(ESMF_ItemOrder_Flag) :: l_itemorderflag

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    l_itemorderflag = ESMF_ITEMORDER_ABC
    if(present(itemorderflag)) l_itemorderflag = itemorderflag

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    if(fieldbundle%this%is_packed) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
msg="This call does not work with packed FieldBundle.",&
          ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    nullify(l_fieldList)
    ! Check size
    call ESMF_ContainerGet(fieldbundle%this%container, trim(fieldName), &
      itemCount=fieldCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(size(fieldList) .lt. fieldCount) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, &
      msg=" - Input argument fieldList size is too small", &
      ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    allocate(l_fieldList(fieldCount), stat=localrc)
    if(localrc /= ESMF_SUCCESS) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - cannot allocate l_fieldList internally", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    call ESMF_ContainerGet(fieldbundle%this%container, trim(fieldName), &
      l_fieldList, itemorderflag=l_itemorderflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    fieldList(1:fieldCount) = l_fieldList(1:fieldCount)

    deallocate(l_fieldList)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleGetList
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleGetIndex()"
!BOPI
! !IROUTINE: ESMF_FieldBundleGet - Access the Field at a specific index in a FieldBundle
!
! !INTERFACE:
    ! Private name; call using ESMF_FieldBundleGet()
    subroutine ESMF_FieldBundleGetIndex(fieldbundle, fieldIndex, field, keywordEnforcer, &
      rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
    integer, intent(in) :: fieldIndex
    type(ESMF_Field), intent(inout) :: field
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Get the fieldIndex-th Field from a FieldBundle. The order of the Fields in FieldBundle
! is not guranteed. If this call is used iteratively, then any Add, Replace, or Remove
! call on the FieldBundle can change the order and/or number of Fields in that FieldBundle.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} to be queried.
! \item [fieldIndex]
! The index of the Field to be returned.
! \item [field]
! The fieldIndex-th Field to be returned.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: l_fieldCount, i ! helper variable
    type(ESMF_Field), pointer :: l_fieldList(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    if(fieldbundle%this%is_packed) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
msg="This call does not work with packed FieldBundle.",&
          ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    nullify(l_fieldList)

    ! Call into the container
    call ESMF_ContainerGet(fieldbundle%this%container, &
      itemCount=l_fieldCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(fieldIndex .lt. 1 .or. fieldIndex .gt. l_fieldCount) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = "fieldIndex must be between 1 and fieldCount in the FieldBundle",&
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    allocate(l_fieldList(l_fieldCount), stat=localrc)
    if(localrc /= ESMF_SUCCESS) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - cannot allocate l_fieldList internally", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    call ESMF_ContainerGet(fieldbundle%this%container, &
      itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    field = l_fieldList(fieldIndex)

    deallocate(l_fieldList)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleGetIndex
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldBundleGet - Get Fortran array pointer from a packed FieldBundle 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldBundleGet() 
! function ESMF_FieldBundleGetDataPtr<rank><type><kind>(fieldBundle, & 
! localDe, farrayPtr, keywordEnforcer, & 
! rc) 
! 
! !RETURN VALUE: 
! type(ESMF_FieldBundle) :: ESMF_FieldBundleGetDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_FieldBundle), intent(in) :: fieldBundle 
! integer, intent(in), optional :: localDe 
! <type> (ESMF_KIND_<kind>), dimension(<rank>), pointer :: farrayPtr 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Get a Fortran pointer to DE-local memory allocation within packed FieldBundle. 
! It's erroneous to perform this call on a FieldBundle that's not packed. 
! 
! The arguments are: 
! \begin{description} 
! \item [fieldBundle] 
! {\tt ESMF\_FieldBundle} object. 
! \item [{[localDe]}] 
! Local DE for which information is requested. [0,..,localDeCount-1]. 
! For localDeCount==1 the localDe argument may be omitted, 
! in which case it will default to localDe=0. In the case where 
! packed FieldBundle is created on a Grid, the number of localDes can 
! be queried from the Grid attached to the FieldBundle. In the case where 
! packed FieldBundle is created on a Mesh, the number of localDes is 1. 
! \item[farrayPtr] 
! Fortran array pointer which will be pointed at DE-local memory allocation 
! in packed FieldBundle. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr1Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr2Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr3Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr4Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr5Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr6Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr7Di1( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr1Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr2Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr3Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr4Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr5Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr6Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr7Di2( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr1Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr1Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr1Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr1Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr2Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr2Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr2Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr2Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr3Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr3Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr3Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr3Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr4Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr4Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr4Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr4Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr5Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr5Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr5Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr5Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr6Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr6Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr6Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr6Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr7Di4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr7Di8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr7Dr4( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr7Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldBundleGetDataPtr" 
 subroutine ESMF_FieldBundleGetDataPtr7Dr8( fieldBundle, & 
 localDe, farrayPtr, keywordEnforcer, & 
 rc) 

 ! return value 

 !ARGUMENTS: 
 type(ESMF_FieldBundle), intent(in) :: fieldBundle 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 

 ! local variables 
 type(ESMF_FieldBundleType), pointer :: this 
 type(ESMF_Field), pointer :: l_fieldList(:) 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! Check init status of arguments 
 ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc) 

 if(.not. FieldBundle%this%is_packed) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, & 
 msg = "this call only works with packed FieldBundle.",& 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 nullify(l_fieldList)

 call ESMF_ContainerGet(fieldbundle%this%container, & 
 itemorderflag=ESMF_ITEMORDER_ADDORDER, itemList=l_fieldList, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldGet(l_fieldList(1), localDe=localDe, & 
 farrayPtr = farrayPtr, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Return successfully 
 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldBundleGetDataPtr7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleHalo()"
!BOP
! !IROUTINE: ESMF_FieldBundleHalo - Execute a FieldBundle halo operation
!
! !INTERFACE:
  subroutine ESMF_FieldBundleHalo(fieldbundle, routehandle, keywordEnforcer, &
    checkflag, rc)
!
! !ARGUMENTS:
        type(ESMF_FieldBundle), intent(inout) :: fieldbundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: checkflag
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Execute a precomputed halo operation for the Fields in {\tt fieldbundle}.
! The FieldBundle must match the respective FieldBundle used during
! {\tt ESMF\_FieldBundleHaloStore()} in {\em type}, {\em kind}, and
! memory layout of the {\em gridded} dimensions. However, the size, number,
! and index order of {\em ungridded} dimensions may be different. See section
! \ref{RH:Reusability} for a more detailed discussion of RouteHandle
! reusability.
! \end{sloppypar}
!
! See {\tt ESMF\_FieldBundleHaloStore()} on how to precompute
! {\tt routehandle}.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} with source data. The data in this
! FieldBundle may be destroyed by this call.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[checkflag]}]
! If set to {\tt .TRUE.} the input FieldBundle pair will be checked for
! consistency with the precomputed operation provided by {\tt routehandle}.
! If set to {\tt .FALSE.} {\em (default)} only a very basic input check
! will be performed, leaving many inconsistencies undetected. Set
! {\tt checkflag} to {\tt .FALSE.} to achieve highest performance.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! local variables to buffer optional arguments
        logical :: l_checkflag! helper variable
        type(ESMF_Field) :: l_field ! helper variable
        type(ESMF_Field), allocatable :: l_fieldList(:) ! helper variable

        ! local internal variables
        integer :: fcount, i

        type(ESMF_ArrayBundle) :: arrayBundle
        type(ESMF_Array), allocatable :: arrays(:)

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments, deal with optional FieldBundle args
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

        ! Set default flags
        l_checkflag = ESMF_FALSE
        if (present(checkflag)) l_checkflag = checkflag

        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

        call ESMF_FieldBundleGet(fieldbundle, fieldCount=fcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        allocate(l_fieldList(fcount))

        call ESMF_FieldBundleGet(fieldbundle, fieldList=l_fieldList, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! build arrayBundle on-the-fly
        allocate(arrays(fcount))
        do i = 1, fcount
            call ESMF_FieldGet(l_fieldList(i), array=arrays(i), rc=localrc)
            if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
        enddo
        arrayBundle = ESMF_ArrayBundleCreate(arrayList=arrays, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        deallocate(arrays)

        call ESMF_ArrayBundleHalo(arrayBundle, routehandle, &
            checkflag=l_checkflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(arrayBundle, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        deallocate(l_fieldList)

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleHalo

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleHaloRelease()"
!BOP
! !IROUTINE: ESMF_FieldBundleHaloRelease - Release resources associated with a FieldBundle halo operation
!
! !INTERFACE:
  subroutine ESMF_FieldBundleHaloRelease(routehandle, keywordEnforcer, &
    noGarbage, rc)
!
! !ARGUMENTS:
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: noGarbage
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.0.0] Added argument {\tt noGarbage}.
! The argument provides a mechanism to override the default garbage collection
! mechanism when destroying an ESMF object.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Release resources associated with a FieldBundle halo operation. After this call
! {\tt routehandle} becomes invalid.
!
! \begin{description}
! \item [routehandle]
! Handle to the precomputed Route.
! \item[{[noGarbage]}]
! If set to {\tt .TRUE.} the object will be fully destroyed and removed
! from the ESMF garbage collection system. Note however that under this
! condition ESMF cannot protect against accessing the destroyed object
! through dangling aliases -- a situation which may lead to hard to debug
! application crashes.
!
! It is generally recommended to leave the {\tt noGarbage} argument
! set to {\tt .FALSE.} (the default), and to take advantage of the ESMF
! garbage collection system which will prevent problems with dangling
! aliases or incorrect sequences of destroy calls. However this level of
! support requires that a small remnant of the object is kept in memory
! past the destroy call. This can lead to an unexpected increase in memory
! consumption over the course of execution in applications that use
! temporary ESMF objects. For situations where the repeated creation and
! destruction of temporary objects leads to memory issues, it is
! recommended to call with {\tt noGarbage} set to {\tt .TRUE.}, fully
! removing the entire temporary object from memory.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

        ! Call into the RouteHandle code
        call ESMF_RouteHandleRelease(routehandle, noGarbage=noGarbage, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleHaloRelease

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleHaloStore()"
!BOP
! !IROUTINE: ESMF_FieldBundleHaloStore - Precompute a FieldBundle halo operation
!
! !INTERFACE:
    subroutine ESMF_FieldBundleHaloStore(fieldbundle, routehandle, &
      keywordEnforcer, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Store a FieldBundle halo operation over the data in {\tt fieldbundle}.
! By definition, all elements in the total Field regions that lie
! outside the exclusive regions will be considered potential destination
! elements for the halo operation. However, only those elements that have a corresponding
! halo source element, i.e. an exclusive element on one of the DEs, will be
! updated under the halo operation. Elements that have no associated source
! remain unchanged under halo.
!
! The routine returns an {\tt ESMF\_RouteHandle} that can be used to call
! {\tt ESMF\_FieldBundleHalo()} on any pair of FieldBundles that matches
! {\tt srcFieldBundle} and {\tt dstFieldBundle} in {\em type}, {\em kind},
! and memory layout of the {\em gridded} dimensions. However, the size,
! number, and index order of {\em ungridded} dimensions may be different.
! See section \ref{RH:Reusability} for a more detailed discussion of
! RouteHandle reusability.
!
! This call is {\em collective} across the current VM.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} containing data to be haloed. The data in this
! FieldBundle may be destroyed by this call.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        ! internal local variables
        integer :: localrc, fcount, i
        type(ESMF_Field) :: l_field
        type(ESMF_ArrayBundle) :: arrayBundle
        type(ESMF_Array), allocatable :: arrays(:)
        type(ESMF_Field), allocatable :: l_fieldList(:) ! helper variable

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

        ! build arrayBundle on-the-fly
        call ESMF_FieldBundleGet(fieldbundle, fieldCount=fcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        allocate(l_fieldList(fcount))

        call ESMF_FieldBundleGet(fieldbundle, fieldList=l_fieldList, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        allocate(arrays(fcount))
        do i = 1, fcount
            call ESMF_FieldGet(l_fieldList(i), array=arrays(i), rc=localrc)
            if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
        enddo
        arrayBundle = ESMF_ArrayBundleCreate(arrayList=arrays, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        deallocate(arrays)

        call ESMF_ArrayBundleHaloStore(arrayBundle, routehandle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(arrayBundle, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        deallocate(l_fieldList)

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleHaloStore
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleIsCreated()"
!BOP
! !IROUTINE: ESMF_FieldBundleIsCreated - Check whether a FieldBundle object has been created

! !INTERFACE:
  function ESMF_FieldBundleIsCreated(fieldbundle, keywordEnforcer, rc)
! !RETURN VALUE:
    logical :: ESMF_FieldBundleIsCreated
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(out), optional :: rc

! !DESCRIPTION:
! Return {\tt .true.} if the {\tt fieldbundle} has been created. Otherwise return
! {\tt .false.}. If an error occurs, i.e. {\tt rc /= ESMF\_SUCCESS} is
! returned, the return value of the function will also be {\tt .false.}.
!
! The arguments are:
! \begin{description}
! \item[fieldbundle]
! {\tt ESMF\_FieldBundle} queried.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
  !-----------------------------------------------------------------------------
    ESMF_FieldBundleIsCreated = .false. ! initialize
    if (present(rc)) rc = ESMF_SUCCESS
    if (ESMF_FieldBundleGetInit(fieldbundle)==ESMF_INIT_CREATED) &
      ESMF_FieldBundleIsCreated = .true.
  end function
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundlePrint()"
!BOP
! !IROUTINE: ESMF_FieldBundlePrint - Print FieldBundle information

! !INTERFACE:
  subroutine ESMF_FieldBundlePrint(fieldbundle, keywordEnforcer, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Print internal information of the specified {\tt fieldbundle} object.
!
! The arguments are:
! \begin{description}
! \item[fieldbundle]
! {\tt ESMF\_FieldBundle} object.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: l_fieldCount, i
    type(ESMF_Field), pointer :: l_fieldList(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    nullify(l_fieldList)

    ! Get all the fields
    call ESMF_FieldBundleGet(fieldbundle, fieldCount=l_fieldCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    allocate(l_fieldList(l_fieldCount), stat=localrc)
    if(localrc /= ESMF_SUCCESS) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - cannot allocate l_fieldList internally", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    call ESMF_ContainerGet(fieldbundle%this%container, &
      itemList=l_fieldList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    do i = 1, l_fieldCount
      call ESMF_FieldPrint(l_fieldList(i), rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    enddo

    deallocate(l_fieldList)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundlePrint
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRead()"
!BOP
! !IROUTINE: ESMF_FieldBundleRead - Read Fields to a FieldBundle from file(s)
! \label{api:FieldBundleRead}

! !INTERFACE:
  subroutine ESMF_FieldBundleRead(fieldbundle, fileName, &
    keywordEnforcer, singleFile, timeslice, iofmt, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    character(*), intent(in) :: fileName
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords for the below
    logical, intent(in), optional :: singleFile
    integer, intent(in), optional :: timeslice
    type(ESMF_IOFmt_Flag), intent(in), optional :: iofmt
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Read field data to a FieldBundle object from file(s).
! For this API to be functional, the environment variable {\tt ESMF\_PIO}
! should be set to either "internal" or "external" when the ESMF library is built.
! Please see the section on Data I/O,~\ref{io:dataio}.
!
! Limitations:
! \begin{itemize}
! \item For multi-tile Fields, all Fields in the FieldBundle must contain
! the same number of tiles.
! \end{itemize}
!
! The arguments are:
! \begin{description}
! \item[fieldbundle]
! An {\tt ESMF\_FieldBundle} object.
! \item[fileName]
! The name of the file from which FieldBundle data is read.
! If the FieldBundle contains multi-tile Arrays, then fileName must contain
! exactly one instance of "*"; this is a placeholder that will be replaced
! by the tile number, with each tile being read from a separate file. (For
! example, for a fileName of "myfile*.nc", tile 1 will be read from
! "myfile1.nc", tile 2 from "myfile2.nc", etc.)
! (This handling of the fileName for multi-tile I/O is subject to change.)
! \item[{[singleFile]}]
! A logical flag, the default is .true., i.e., all Fields in the bundle
! are stored in one single file. If .false., each field is stored
! in separate files; these files are numbered with the name based on the
! argument "file". That is, a set of files are named: [file\_name]001,
! [file\_name]002, [file\_name]003,...
! \item[{[timeslice]}]
! The time-slice number of the variable read from file.
! \item[{[iofmt]}]
! \begin{sloppypar}
! The I/O format. Please see Section~\ref{opt:iofmtflag} for the list
! of options. If not present, defaults to {\tt ESMF\_IOFMT\_NETCDF}.
! \end{sloppypar}
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    character(len=ESMF_MAXSTR) :: name
    integer :: fieldCount
    integer :: i
    type(ESMF_Field), allocatable :: fieldList(:)
    logical :: singlef
    character(len=3) :: cnum
    character(len=len (fileName) + 3) :: filename_num ! len (file)+len (cnum)
    type(ESMF_Array) :: array
    type(ESMF_IOFmt_Flag) :: opt_iofmt
    type(ESMF_IO) :: io ! The I/O object
    logical :: errorFound ! True if error condition
    integer :: file_ext_p
    integer :: time

#ifdef ESMF_PIO
    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    ! Check options
    singlef = .true.
    if (present(singleFile)) singlef = singleFile

    ! Set iofmt based on file name extension (if present)
    if (present (iofmt)) then
      opt_iofmt = iofmt
    else
      opt_iofmt = ESMF_IOFMT_NETCDF
    end if

    time = 0
    if(present(timeslice)) time = timeslice

    call ESMF_FieldBundleGet(fieldbundle, fieldCount=fieldCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    allocate (fieldList(fieldCount))
    call ESMF_FieldBundleGet(fieldbundle, fieldList=fieldList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create an I/O object
    io = ESMF_IOCreate(rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! From here on out, we need to clean up so no returning on error
    if (singlef) then
      ! Get and read the fields in the Bundle
      do i=1,fieldCount
        call ESMF_FieldGet(fieldList(i), array=array, name=name, rc=localrc)
        errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)
        if (errorFound) exit
        call ESMF_IOAddArray(io, array, variableName=name, rc=localrc)
        errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)
        if (errorFound) exit
      enddo
      if (.not. errorFound) then
        call ESMF_IORead(io, trim(fileName), timeslice=time, &
            iofmt=opt_iofmt, rc=localrc)
        errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)
      endif
    else
      do i=1,fieldCount
        ! Clear the IO object (only need to do this for i > 1)
        if (i .gt. 1) call ESMF_IOClear(io)
        write(cnum,"(i3.3)") i
        filename_num = ESMF_StringConcat(trim (fileName) , cnum)
        call ESMF_FieldGet(fieldList(i), array=array, name=name, rc=localrc)
        errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)
        if (errorFound) exit
        call ESMF_IOAddArray(io, array, variableName=name, rc=localrc)
        errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)
        if (errorFound) exit
        if (.not. errorFound) then
          call ESMF_IORead(io, trim(filename_num), timeslice=time, &
              iofmt=opt_iofmt, rc=localrc)
          errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)
        endif
      enddo
    endif

    ! Set rc here in case we had an error but destroy succeeds
    if (present(rc)) rc = localrc

    call ESMF_IODestroy(io, rc=localrc)
    ! Log error but donot reset rc
    errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=localrc)

    ! Last chance to return an error code (IODestroy failed)
    if (present(rc)) then
      if (rc == ESMF_SUCCESS) rc = localrc
    end if

#else
    ! Return indicating PIO not present
    if (present(rc)) rc = ESMF_RC_LIB_NOT_PRESENT
#endif

  end subroutine ESMF_FieldBundleRead
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedist()"
!BOP
! !IROUTINE: ESMF_FieldBundleRedist - Execute a FieldBundle redistribution
!
! !INTERFACE:
  subroutine ESMF_FieldBundleRedist(srcFieldBundle, dstFieldBundle, &
    routehandle, keywordEnforcer, checkflag, rc)
!
! !ARGUMENTS:
        type(ESMF_FieldBundle), intent(in), optional :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout), optional :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: checkflag
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Execute a precomputed redistribution from {\tt srcFieldBundle}
! to {\tt dstFieldBundle}.
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} must match the
! respective FieldBundles used during {\tt ESMF\_FieldBundleRedistStore()}
! in {\em type}, {\em kind}, and memory layout of the {\em gridded}
! dimensions. However, the size, number,
! and index order of {\em ungridded} dimensions may be different. See section
! \ref{RH:Reusability} for a more detailed discussion of RouteHandle
! reusability.
! \end{sloppypar}
!
! The {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments are optional in support of
! the situation where {\tt srcFieldBundle} and/or {\tt dstFieldBundle} are not defined on
! all PETs. The {\tt srcFieldBundle} and {\tt dstFieldBundle} must be specified on those
! PETs that hold source or destination DEs, respectively, but may be omitted
! on all other PETs. PETs that hold neither source nor destination DEs may
! omit both arguments.
!
! It is erroneous to specify the identical FieldBundle object for {\tt srcFieldBundle} and
! {\tt dstFieldBundle} arguments.
!
! See {\tt ESMF\_FieldBundleRedistStore()} on how to precompute
! {\tt routehandle}.
!
! This call is {\em collective} across the current VM.
!
! For examples and associated documentation regarding this method see Section
! \ref{sec:fieldbundle:usage:redist_1dptr}.
!
! \begin{description}
! \item [{[srcFieldBundle]}]
! {\tt ESMF\_FieldBundle} with source data.
! \item [{[dstFieldBundle]}]
! {\tt ESMF\_FieldBundle} with destination data.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[checkflag]}]
! If set to {\tt .TRUE.} the input FieldBundle pair will be checked for
! consistency with the precomputed operation provided by {\tt routehandle}.
! If set to {\tt .FALSE.} {\em (default)} only a very basic input check
! will be performed, leaving many inconsistencies undetected. Set
! {\tt checkflag} to {\tt .FALSE.} to achieve highest performance.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! local variables to buffer optional arguments
    logical :: l_checkflag! helper variable

    ! local internal variables
    logical :: src_bundle
    logical :: dst_bundle
    integer :: fcount, i
    type(ESMF_ArrayBundle) :: srcab, dstab
    type(ESMF_Array), allocatable :: srca(:), dsta(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments, deal with optional FieldBundle args
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

    ! Set default flags
    l_checkflag = ESMF_FALSE
    if (present(checkflag)) l_checkflag = checkflag

    src_bundle = .true.
    if (present(srcFieldBundle)) then
      srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    else
      src_bundle = .false.
    endif

    dst_bundle = .true.
    if (present(dstFieldBundle)) then
      dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    else
      dst_bundle = .false.
    endif

    ! perform FieldBundle redistribution
    if(src_bundle .and. dst_bundle) then
        call ESMF_ArrayBundleRedist(srcab, dstab, routehandle, &
            checkflag=l_checkflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
    else if(src_bundle .and. .not. dst_bundle) then
        call ESMF_ArrayBundleRedist(srcArrayBundle=srcab, routehandle=routehandle, &
            checkflag=l_checkflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
    else if(.not. src_bundle .and. dst_bundle) then
        call ESMF_ArrayBundleRedist(dstArrayBundle=dstab, routehandle=routehandle, &
            checkflag=l_checkflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
    else if(.not. src_bundle .and. .not. dst_bundle) then
        call ESMF_ArrayBundleRedist(routehandle=routehandle, &
            checkflag=l_checkflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! garbage collection
    if (present(srcFieldBundle)) then
      call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif
    if (present(dstFieldBundle)) then
      call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleRedist

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedistRelease()"
!BOP
! !IROUTINE: ESMF_FieldBundleRedistRelease - Release resources associated with a FieldBundle redistribution
!
! !INTERFACE:
  subroutine ESMF_FieldBundleRedistRelease(routehandle, keywordEnforcer, &
    noGarbage, rc)
!
! !ARGUMENTS:
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: noGarbage
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.0.0] Added argument {\tt noGarbage}.
! The argument provides a mechanism to override the default garbage collection
! mechanism when destroying an ESMF object.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Release resources associated with a FieldBundle redistribution. After this call
! {\tt routehandle} becomes invalid.
!
! \begin{description}
! \item [routehandle]
! Handle to the precomputed Route.
! \item[{[noGarbage]}]
! If set to {\tt .TRUE.} the object will be fully destroyed and removed
! from the ESMF garbage collection system. Note however that under this
! condition ESMF cannot protect against accessing the destroyed object
! through dangling aliases -- a situation which may lead to hard to debug
! application crashes.
!
! It is generally recommended to leave the {\tt noGarbage} argument
! set to {\tt .FALSE.} (the default), and to take advantage of the ESMF
! garbage collection system which will prevent problems with dangling
! aliases or incorrect sequences of destroy calls. However this level of
! support requires that a small remnant of the object is kept in memory
! past the destroy call. This can lead to an unexpected increase in memory
! consumption over the course of execution in applications that use
! temporary ESMF objects. For situations where the repeated creation and
! destruction of temporary objects leads to memory issues, it is
! recommended to call with {\tt noGarbage} set to {\tt .TRUE.}, fully
! removing the entire temporary object from memory.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

        ! Call into the RouteHandle code
        call ESMF_RouteHandleRelease(routehandle, noGarbage=noGarbage, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRedistRelease

!----------------------------------------------------------------------------
!BOP
! !IROUTINE: ESMF_FieldBundleRedistStore - Precompute a FieldBundle redistribution with local factor argument
!
! !INTERFACE:
! ! Private name; call using ESMF_FieldBundleRedistStore()
! subroutine ESMF_FieldBundleRedistStore<type><kind>(srcFieldBundle, &
! dstFieldBundle, routehandle, factor, keywordEnforcer, &
! ignoreUnmatchedIndicesFlag, srcToDstTransposeMap, rc)
!
! !ARGUMENTS:
! type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
! type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
! type(ESMF_RouteHandle), intent(inout) :: routehandle
! <type>(ESMF_KIND_<kind>), intent(in) :: factor
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
! integer, intent(in), optional :: srcToDstTransposeMap(:)
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt ignoreUnmatchedIndicesFlag} to support cases
! where source and destination side do not cover the exact same index space.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
!
! Store a FieldBundle redistribution operation from {\tt srcFieldBundle} to {\tt dstFieldBundle}.
! PETs that
! specify a {\tt factor} argument must use the <type><kind> overloaded interface. Other
! PETs call into the interface without {\tt factor} argument. If multiple PETs specify
! the {\tt factor} argument its type and kind as well as its value must match across
! all PETs. If none of the PETs specifies a {\tt factor} argument the default will be a
! factor of 1.
!
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} are interpreted as sequentialized
! vectors. The
! sequence is defined by the order of DistGrid dimensions and the order of
! tiles within the DistGrid or by user-supplied arbitrary sequence indices. See
! section \ref{Array:SparseMatMul} for details on the definition of {\em sequence indices}.
! Redistribution corresponds to an identity mapping of the source FieldBundle vector to
! the destination FieldBundle vector.
!
! Source and destination FieldBundles may be of different <type><kind>. Further source
! and destination FieldBundles may differ in shape, however, the number of elements
! must match.
!
! It is erroneous to specify the identical FieldBundle object for srcFieldBundle
! and dstFieldBundle arguments.
!
! The routine returns an {\tt ESMF\_RouteHandle} that can be used to call
! {\tt ESMF\_FieldBundleRedist()} on any pair of FieldBundles that matches
! {\tt srcFieldBundle} and {\tt dstFieldBundle} in {\em type}, {\em kind},
! and memory layout of the {\em gridded} dimensions. However, the size,
! number, and index order of {\em ungridded} dimensions may be different.
! See section \ref{RH:Reusability} for a more detailed discussion of
! RouteHandle reusability.
!
! This method is overloaded for:\newline
! {\tt ESMF\_TYPEKIND\_I4}, {\tt ESMF\_TYPEKIND\_I8},\newline
! {\tt ESMF\_TYPEKIND\_R4}, {\tt ESMF\_TYPEKIND\_R8}.
! \newline
!
! This call is collective across the current VM.
!
! For examples and associated documentation regarding this method see Section
! \ref{sec:fieldbundle:usage:redist_1dptr}.
!
! The arguments are:
! \begin{description}
! \item [srcFieldBundle]
! {\tt ESMF\_FieldBundle} with source data.
! \item [dstFieldBundle]
! {\tt ESMF\_FieldBundle} with destination data. The data in this
! FieldBundle may be destroyed by this call.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [factor]
! Factor by which to multiply source data.
! \item [{[ignoreUnmatchedIndicesFlag]}]
! If set to {.false.}, the {\em default}, source and destination side must
! cover the identical index space, using precisely matching sequence
! indices. If set to {.true.}, mismatching sequence indices between source
! and destination side are silently ignored.
! The size of this array argument must either be 1 or equal the number of
! Fields in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the handling of unmatched indices is specified for each
! Field pair separately. If only one element is specified, it is
! used for {\em all} Field pairs.
! \item [{[srcToDstTransposeMap]}]
! List with as many entries as there are dimensions in {\tt srcFieldBundle}. Each
! entry maps the corresponding {\tt srcFieldBundle} dimension
! against the specified {\tt dstFieldBundle}
! dimension. Mixing of distributed and undistributed dimensions is supported.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!----------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedistStoreI4"
!BOPI
! !IROUTINE: ESMF_FieldBundleRedistStore - Precompute a FieldBundle redistribution
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleRedistStore()
    subroutine ESMF_FieldBundleRedistStoreI4(srcFieldBundle, dstFieldBundle, &
      routehandle, factor, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
      srcToDstTransposeMap, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        integer(ESMF_KIND_I4), intent(in) :: factor
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(in) , optional :: srcToDstTransposeMap(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleRedistStore(srcab, dstab, routehandle, factor, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcToDstTransposeMap=srcToDstTransposeMap, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRedistStoreI4
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedistStoreI8"
!BOPI
! !IROUTINE: ESMF_FieldBundleRedistStore - Precompute a FieldBundle redistribution
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleRedistStore()
    subroutine ESMF_FieldBundleRedistStoreI8(srcFieldBundle, dstFieldBundle, &
      routehandle, factor, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
      srcToDstTransposeMap, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        integer(ESMF_KIND_I8), intent(in) :: factor
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(in) , optional :: srcToDstTransposeMap(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleRedistStore(srcab, dstab, routehandle, factor, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcToDstTransposeMap=srcToDstTransposeMap, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRedistStoreI8
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedistStoreR4"
!BOPI
! !IROUTINE: ESMF_FieldBundleRedistStore - Precompute a FieldBundle redistribution
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleRedistStore()
    subroutine ESMF_FieldBundleRedistStoreR4(srcFieldBundle, dstFieldBundle, &
      routehandle, factor, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
      srcToDstTransposeMap, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        real(ESMF_KIND_R4), intent(in) :: factor
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(in) , optional :: srcToDstTransposeMap(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleRedistStore(srcab, dstab, routehandle, factor, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcToDstTransposeMap=srcToDstTransposeMap, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRedistStoreR4
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedistStoreR8"
!BOPI
! !IROUTINE: ESMF_FieldBundleRedistStore - Precompute a FieldBundle redistribution
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleRedistStore()
    subroutine ESMF_FieldBundleRedistStoreR8(srcFieldBundle, dstFieldBundle, &
      routehandle, factor, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
      srcToDstTransposeMap, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        real(ESMF_KIND_R8), intent(in) :: factor
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(in) , optional :: srcToDstTransposeMap(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleRedistStore(srcab, dstab, routehandle, factor, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcToDstTransposeMap=srcToDstTransposeMap, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRedistStoreR8

!----------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRedistStoreNF"
!BOP
! !IROUTINE: ESMF_FieldBundleRedistStore - Precompute a FieldBundle redistribution without local factor argument
!
! !INTERFACE:
! ! Private name; call using ESMF_FieldBundleRedistStore()
  subroutine ESMF_FieldBundleRedistStoreNF(srcFieldBundle, dstFieldBundle, &
    routehandle, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
    srcToDstTransposeMap, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
    type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
    type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
    integer, intent(in), optional :: srcToDstTransposeMap(:)
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt ignoreUnmatchedIndicesFlag} to support cases
! where source and destination side do not cover the exact same index space.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
!
! \begin{sloppypar}
! Store a FieldBundle redistribution operation from {\tt srcFieldBundle}
! to {\tt dstFieldBundle}. PETs that specify non-zero matrix coefficients must use
! the <type><kind> overloaded interface and provide the {\tt factorList} and
! {\tt factorIndexList} arguments. Providing {\tt factorList} and
! {\tt factorIndexList} arguments with {\tt size(factorList) = (/0/)} and
! {\tt size(factorIndexList) = (/2,0/)} or {\tt (/4,0/)} indicates that a
! PET does not provide matrix elements. Alternatively, PETs that do not
! provide matrix elements may also call into the overloaded interface
! {\em without} {\tt factorList} and {\tt factorIndexList} arguments.
! \end{sloppypar}
!
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} are interpreted as sequentialized
! vectors. The
! sequence is defined by the order of DistGrid dimensions and the order of
! tiles within the DistGrid or by user-supplied arbitrary sequence indices. See
! section \ref{Array:SparseMatMul} for details on the definition of {\em sequence indices}.
! Redistribution corresponds to an identity mapping of the source FieldBundle vector to
! the destination FieldBundle vector.
!
! Source and destination Fields may be of different <type><kind>. Further source
! and destination Fields may differ in shape, however, the number of elements
! must match.
!
! It is erroneous to specify the identical FieldBundle object for srcFieldBundle and dstFieldBundle
! arguments.
!
! The routine returns an {\tt ESMF\_RouteHandle} that can be used to call
! {\tt ESMF\_FieldBundleRedist()} on any pair of FieldBundles that matches
! {\tt srcFieldBundle} and {\tt dstFieldBundle} in {\em type}, {\em kind},
! and memory layout of the {\em gridded} dimensions. However, the size,
! number, and index order of {\em ungridded} dimensions may be different.
! See section \ref{RH:Reusability} for a more detailed discussion of
! RouteHandle reusability.
!
! This call is collective across the current VM.
!
! For examples and associated documentation regarding this method see Section
! \ref{sec:fieldbundle:usage:redist_1dptr}.
!
! The arguments are:
! \begin{description}
! \item [srcFieldBundle]
! {\tt ESMF\_FieldBundle} with source data.
! \item [dstFieldBundle]
! {\tt ESMF\_FieldBundle} with destination data. The data in this
! FieldBundle may be destroyed by this call.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[ignoreUnmatchedIndicesFlag]}]
! If set to {.false.}, the {\em default}, source and destination side must
! cover the identical index space, using precisely matching sequence
! indices. If set to {.true.}, mismatching sequence indices between source
! and destination side are silently ignored.
! The size of this array argument must either be 1 or equal the number of
! Fields in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the handling of unmatched indices is specified for each
! Field pair separately. If only one element is specified, it is
! used for {\em all} Field pairs.
! \item [{[srcToDstTransposeMap]}]
! List with as many entries as there are dimensions in {\tt srcFieldBundle}. Each
! entry maps the corresponding {\tt srcFieldBundle} dimension
! against the specified {\tt dstFieldBundle}
! dimension. Mixing of distributed and undistributed dimensions is supported.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!----------------------------------------------------------------------------
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        ! TODO:
        ! internal grids match
        !if(ESMF_GridMatch(srcFieldBundle%btypep%grid, dstFieldBundle%btypep%grid) then
        ! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
        ! "src and dst FieldBundle must have matching grid", &
        ! ESMF_CONTEXT, rcToReturn=rc)
        ! return
        !endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleRedistStore(srcab, dstab, routehandle, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcToDstTransposeMap=srcToDstTransposeMap, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRedistStoreNF
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRegrid()"
!BOP
! !IROUTINE: ESMF_FieldBundleRegrid - Execute a FieldBundle regrid operation
!
! !INTERFACE:
  subroutine ESMF_FieldBundleRegrid(srcFieldBundle, dstFieldBundle, &
         routehandle, keywordEnforcer, zeroregion, termorderflag, checkflag, rc)
!
! !ARGUMENTS:
        type(ESMF_FieldBundle), intent(in), optional :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout), optional :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        type(ESMF_Region_Flag), intent(in), optional :: zeroregion
        type(ESMF_TermOrder_Flag), intent(in), optional :: termorderflag(:)
        logical, intent(in), optional :: checkflag
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[7.0.0] Added argument {\tt termorderflag}.
! The new argument gives the user control over the order in which
! the src terms are summed up.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Execute a precomputed regrid from {\tt srcFieldBundle}
! to {\tt dstFieldBundle}.
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} must match the
! respective FieldBundles used during {\tt ESMF\_FieldBundleRedistStore()}
! in {\em type}, {\em kind}, and memory layout of the {\em gridded}
! dimensions. However, the size, number,
! and index order of {\em ungridded} dimensions may be different. See section
! \ref{RH:Reusability} for a more detailed discussion of RouteHandle
! reusability.
! \end{sloppypar}
!
! The {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments are optional in support of
! the situation where {\tt srcFieldBundle} and/or {\tt dstFieldBundle} are not defined on
! all PETs. The {\tt srcFieldBundle} and {\tt dstFieldBundle} must be specified on those
! PETs that hold source or destination DEs, respectively, but may be omitted
! on all other PETs. PETs that hold neither source nor destination DEs may
! omit both arguments.
!
! It is erroneous to specify the identical FieldBundle object for {\tt srcFieldBundle} and
! {\tt dstFieldBundle} arguments.
!
! See {\tt ESMF\_FieldBundleRegridStore()} on how to precompute
! {\tt routehandle}.
!
! This call is {\em collective} across the current VM.
!
! \begin{description}
! \item [{[srcFieldBundle]}]
! {\tt ESMF\_FieldBundle} with source data.
! \item [{[dstFieldBundle]}]
! {\tt ESMF\_FieldBundle} with destination data.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[zeroregion]}]
! \begin{sloppypar}
! If set to {\tt ESMF\_REGION\_TOTAL} {\em (default)} the total regions of
! all DEs in {\tt dstFieldBundle} will be initialized to zero before updating the
! elements with the results of the sparse matrix multiplication. If set to
! {\tt ESMF\_REGION\_EMPTY} the elements in {\tt dstFieldBundle} will not be
! modified prior to the sparse matrix multiplication and results will be
! added to the incoming element values. Setting {\tt zeroregion} to
! {\tt ESMF\_REGION\_SELECT} will only zero out those elements in the
! destination FieldBundle that will be updated by the sparse matrix
! multiplication. See section \ref{const:region} for a complete list of
! valid settings.
! \end{sloppypar}
! \item [{[termorderflag]}]
! Specifies the order of the source side terms in all of the destination
! sums. The {\tt termorderflag} only affects the order of terms during
! the execution of the RouteHandle. See the \ref{RH:bfb} section for an
! in-depth discussion of {\em all} bit-for-bit reproducibility
! aspects related to route-based communication methods.
! See \ref{const:termorderflag} for a full list of options.
! The size of this array argument must either be 1 or equal the number of
! Fields in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the term order for each Field Regrid operation is
! indicated separately. If only one term order element is specified, it is
! used for {\em all} Field pairs.
! The default is {\tt (/ESMF\_TERMORDER\_FREE/)}, allowing maximum
! flexibility in the order of terms for optimum performance.
! \item [{[checkflag]}]
! If set to {\tt .TRUE.} the input FieldBundle pair will be checked for
! consistency with the precomputed operation provided by {\tt routehandle}.
! If set to {\tt .FALSE.} {\em (default)} only a very basic input check
! will be performed, leaving many inconsistencies undetected. Set
! {\tt checkflag} to {\tt .FALSE.} to achieve highest performance.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments, deal with optional FieldBundle args
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        call ESMF_FieldBundleSMM(srcFieldBundle=srcFieldBundle, &
          dstFieldBundle=dstFieldBundle, routehandle=routehandle, &
          zeroregion=zeroregion, termorderflag=termorderflag, &
          checkflag=checkflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRegrid
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRegridRelease()"
!BOP
! !IROUTINE: ESMF_FieldBundleRegridRelease - Release resources associated with a FieldBundle regrid operation
!
! !INTERFACE:
  subroutine ESMF_FieldBundleRegridRelease(routehandle, keywordEnforcer, &
    noGarbage, rc)
!
! !ARGUMENTS:
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: noGarbage
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.0.0] Added argument {\tt noGarbage}.
! The argument provides a mechanism to override the default garbage collection
! mechanism when destroying an ESMF object.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Release resources associated with a FieldBundle regrid operation. After this call
! {\tt routehandle} becomes invalid.
!
! \begin{description}
! \item [routehandle]
! Handle to the precomputed Route.
! \item[{[noGarbage]}]
! If set to {\tt .TRUE.} the object will be fully destroyed and removed
! from the ESMF garbage collection system. Note however that under this
! condition ESMF cannot protect against accessing the destroyed object
! through dangling aliases -- a situation which may lead to hard to debug
! application crashes.
!
! It is generally recommended to leave the {\tt noGarbage} argument
! set to {\tt .FALSE.} (the default), and to take advantage of the ESMF
! garbage collection system which will prevent problems with dangling
! aliases or incorrect sequences of destroy calls. However this level of
! support requires that a small remnant of the object is kept in memory
! past the destroy call. This can lead to an unexpected increase in memory
! consumption over the course of execution in applications that use
! temporary ESMF objects. For situations where the repeated creation and
! destruction of temporary objects leads to memory issues, it is
! recommended to call with {\tt noGarbage} set to {\tt .TRUE.}, fully
! removing the entire temporary object from memory.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

        ! Call into the RouteHandle code
        call ESMF_RouteHandleRelease(routehandle, noGarbage=noGarbage, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleRegridRelease
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRegridStore()"
!BOP
! !IROUTINE: ESMF_FieldBundleRegridStore - Precompute a FieldBundle regrid operation
! \label{api:esmf_fieldbundleregridstore}
! !INTERFACE:
  subroutine ESMF_FieldBundleRegridStore(srcFieldBundle, dstFieldBundle, keywordEnforcer, &
       srcMaskValues, dstMaskValues, regridmethod, polemethod, regridPoleNPnts, &
       lineType, normType, extrapMethod, extrapNumSrcPnts, extrapDistExponent, &
       extrapNumLevels, unmappedaction, ignoreDegenerate, srcTermProcessing, &
       pipelineDepth, routehandle, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
    type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer(ESMF_KIND_I4), target, intent(in), optional :: srcMaskValues(:)
    integer(ESMF_KIND_I4), target, intent(in), optional :: dstMaskValues(:)
    type(ESMF_RegridMethod_Flag), intent(in), optional :: regridmethod
    type(ESMF_PoleMethod_Flag), intent(in), optional :: polemethod
    integer, intent(in), optional :: regridPoleNPnts
    type(ESMF_LineType_Flag), intent(in), optional :: lineType
    type(ESMF_NormType_Flag), intent(in), optional :: normType
    type(ESMF_ExtrapMethod_Flag), intent(in), optional :: extrapMethod
    integer, intent(in), optional :: extrapNumSrcPnts
    real, intent(in), optional :: extrapDistExponent
    integer, intent(in), optional :: extrapNumLevels
    type(ESMF_UnmappedAction_Flag),intent(in), optional :: unmappedaction
    logical, intent(in), optional :: ignoreDegenerate
    integer, intent(inout), optional :: srcTermProcessing
    integer, intent(inout), optional :: pipelineDepth
    type(ESMF_RouteHandle), intent(inout), optional :: routehandle
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[7.0.0] Added arguments {\tt ignoreDegenerate}, {\tt lineType},
! and {\tt normType}. The argument {\tt ignoreDegenerate} allows the user to
! skip degenerate cells in the regridding instead of stopping with an error.
! The argument {\tt lineType} allows the user to control the path of the line
! between two points on a sphere surface. This allows the user to use their
! preferred line path for the calculation of distances and the shape of cells
! during regrid weight calculation on a sphere. The argument {\tt normType}
! allows the user to control the type of normalization done during
! conservative weight generation.
! \item[7.1.0r] Added argument {\tt srcTermProcessing}.
! Added argument {\tt pipelineDepth}.
! The new arguments provide access to the tuning parameters
! affecting the performance and bit-for-bit behavior when applying
! the regridding weights.
!
! Added arguments {\tt extrapMethod}, {\tt extrapNumSrcPnts}, and
! {\tt extrapDistExponent}. These three new extrapolation arguments allow the
! user to extrapolate destination points not mapped by the regrid method.
! {\tt extrapMethod} allows the user to choose the extrapolation method.
! {\tt extrapNumSrcPnts} and {\tt extrapDistExponent} are parameters that
! allow the user to tune the behavior of the
! {\tt ESMF\_EXTRAPMETHOD\_NEAREST\_IDAVG} method.
! \item[8.0.0] Added argument {\tt extrapNumLevels}. For level based
! extrapolation methods (e.g. {\tt ESMF\_EXTRAPMETHOD\_CREEP}) this argument
! allows the user to set how many levels to extrapolate.
!!
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Store a FieldBundle regrid operation over the data in {\tt srcFieldBundle}
! and {\tt dstFieldBundle} pair.
!
! The routine returns an {\tt ESMF\_RouteHandle} that can be used to call
! {\tt ESMF\_FieldBundleRegrid()} on any pair of FieldBundles that matches
! {\tt srcFieldBundle} and {\tt dstFieldBundle} in {\em type}, {\em kind},
! and memory layout of the {\em gridded} dimensions. However, the size,
! number, and index order of {\em ungridded} dimensions may be different.
! See section \ref{RH:Reusability} for a more detailed discussion of
! RouteHandle reusability.
!
! This call is {\em collective} across the current VM.
!
! \begin{description}
! \item [srcFieldbundle]
! Source {\tt ESMF\_FieldBundle} containing data to be regridded.
! \item [dstFieldbundle]
! Destination {\tt ESMF\_FieldBundle}. The data in this FieldBundle may be
! overwritten by this call.
! \item [{[srcMaskValues]}]
! Mask information can be set in the Grids (see~\ref{sec:usage:items})
! or Meshes (see~\ref{sec:mesh:mask}) upon which the Fields in the
! {\tt srcFieldbundle} are built. The {\tt srcMaskValues} argument
! specifies the values in that mask information which indicate a source
! point should be masked out. In other words, a location is masked if and
! only if the value for that location in the mask information matches one
! of the values listed in {\tt srcMaskValues}.
! If {\tt srcMaskValues} is not specified, no masking will occur.
! \item [{[dstMaskValues]}]
! Mask information can be set in the Grids (see~\ref{sec:usage:items}) or Meshes (see~\ref{sec:mesh:mask})
! upon which the Fields in the {\tt dstFieldbundle} are built.
! The {\tt dstMaskValues} argument specifies the values in that mask
! information which indicate a destination point should be masked out.
! In other words, a location is masked if and only if the value for that
! location in the mask information matches
! one of the values listed in {\tt dstMaskValues}.
! If {\tt dstMaskValues} is not specified, no masking will occur.
! \item [{[regridmethod]}]
! The type of interpolation. Please see Section~\ref{opt:regridmethod} for a list of
! valid options. If not specified, defaults to {\tt ESMF\_REGRIDMETHOD\_BILINEAR}.
! \item [{[polemethod]}]
! Which type of artificial pole
! to construct on the source Grid for regridding. Please see
! section~\ref{const:polemethod} for a list of valid options. If not
! specified, defaults to {\tt ESMF\_POLEMETHOD\_ALLAVG}.
! \item [{[regridPoleNPnts]}]
! If {\tt polemethod} is {\tt ESMF\_POLEMETHOD\_NPNTAVG}.
! This parameter indicates how many points should be averaged
! over. Must be specified if {\tt polemethod} is
! {\tt ESMF\_POLEMETHOD\_NPNTAVG}.
! \item [{[lineType]}]
! This argument controls the path of the line which connects two points on a sphere surface. This in
! turn controls the path along which distances are calculated and the shape of the edges that make
! up a cell. Both of these quantities can influence how interpolation weights are calculated.
! As would be expected, this argument is only applicable when {\tt srcField} and {\tt dstField} are
! built on grids which lie on the surface of a sphere. Section~\ref{opt:lineType} shows a
! list of valid options for this argument. If not specified, the default depends on the
! regrid method. Section~\ref{opt:lineType} has the defaults by line type. Figure~\ref{line_type_support} shows
! which line types are supported for each regrid method as well as showing the default line type by regrid method.
! \item [{[normType]}]
! This argument controls the type of normalization used when generating conservative weights. This option
! only applies to weights generated with {\tt regridmethod=ESMF\_REGRIDMETHOD\_CONSERVE}. Please see
! Section~\ref{opt:normType} for a
! list of valid options. If not specified {\tt normType} defaults to {\tt ESMF\_NORMTYPE\_DSTAREA}.
! \item [{[extrapMethod]}]
! The type of extrapolation. Please see Section~\ref{opt:extrapmethod}
! for a list of valid options. If not specified, defaults to
! {\tt ESMF\_EXTRAPMETHOD\_NONE}.
! \item [{[extrapNumSrcPnts]}]
! The number of source points to use for the extrapolation methods that use more than one source point
! (e.g. {\tt ESMF\_EXTRAPMETHOD\_NEAREST\_IDAVG}). If not specified, defaults to 8.
! \item [{[extrapDistExponent]}]
! The exponent to raise the distance to when calculating weights for
! the {\tt ESMF\_EXTRAPMETHOD\_NEAREST\_IDAVG} extrapolation method. A higher value reduces the influence
! of more distant points. If not specified, defaults to 2.0.
! \item [{[extrapNumLevels]}]
! The number of levels to output for the extrapolation methods that fill levels
! (e.g. {\tt ESMF\_EXTRAPMETHOD\_CREEP}). When a method is used that requires this, then an error will be returned, if it
! is not specified.
! \item [{[unmappedaction]}]
! Specifies what should happen if there are destination points that
! can not be mapped to a source cell. Please see Section~\ref{const:unmappedaction} for a
! list of valid options. If not specified, {\tt unmappedaction} defaults to {\tt ESMF\_UNMAPPEDACTION\_ERROR}.
! \item [{[ignoreDegenerate]}]
! Ignore degenerate cells when checking the input Grids or Meshes for errors. If this is set to true, then the
! regridding proceeds, but degenerate cells will be skipped. If set to false, a degenerate cell produces an error.
! If not specified, {\tt ignoreDegenerate} defaults to false.
!
! \item [{[srcTermProcessing]}]
! The {\tt srcTermProcessing} parameter controls how many source terms,
! located on the same PET and summing into the same destination element,
! are summed into partial sums on the source PET before being transferred
! to the destination PET. A value of 0 indicates that the entire arithmetic
! is done on the destination PET; source elements are neither multiplied
! by their factors nor added into partial sums before being sent off by the
! source PET. A value of 1 indicates that source elements are multiplied
! by their factors on the source side before being sent to the destination
! PET. Larger values of {\tt srcTermProcessing} indicate the maximum number
! of terms in the partial sums on the source side.
!
! Note that partial sums may lead to bit-for-bit differences in the results.
! See section \ref{RH:bfb} for an in-depth discussion of {\em all}
! bit-for-bit reproducibility aspects related to route-based communication
! methods.
!
! \begin{sloppypar}
! The {\tt ESMF\_FieldRegridStore()} method implements an auto-tuning scheme
! for the {\tt srcTermProcessing} parameter. The intent on the
! {\tt srcTermProcessing} argument is "{\tt inout}" in order to
! support both overriding and accessing the auto-tuning parameter.
! If an argument $>= 0$ is specified, it is used for the
! {\tt srcTermProcessing} parameter, and the auto-tuning phase is skipped.
! In this case the {\tt srcTermProcessing} argument is not modified on
! return. If the provided argument is $< 0$, the {\tt srcTermProcessing}
! parameter is determined internally using the auto-tuning scheme. In this
! case the {\tt srcTermProcessing} argument is re-set to the internally
! determined value on return. Auto-tuning is also used if the optional
! {\tt srcTermProcessing} argument is omitted.
! \end{sloppypar}
!
! \item [{[pipelineDepth]}]
! The {\tt pipelineDepth} parameter controls how many messages a PET
! may have outstanding during a sparse matrix exchange. Larger values
! of {\tt pipelineDepth} typically lead to better performance. However,
! on some systems too large a value may lead to performance degradation,
! or runtime errors.
!
! Note that the pipeline depth has no effect on the bit-for-bit
! reproducibility of the results. However, it may affect the performance
! reproducibility of the exchange.
!
! The {\tt ESMF\_FieldRegridStore()} method implements an auto-tuning scheme
! for the {\tt pipelineDepth} parameter. The intent on the
! {\tt pipelineDepth} argument is "{\tt inout}" in order to
! support both overriding and accessing the auto-tuning parameter.
! If an argument $>= 0$ is specified, it is used for the
! {\tt pipelineDepth} parameter, and the auto-tuning phase is skipped.
! In this case the {\tt pipelineDepth} argument is not modified on
! return. If the provided argument is $< 0$, the {\tt pipelineDepth}
! parameter is determined internally using the auto-tuning scheme. In this
! case the {\tt pipelineDepth} argument is re-set to the internally
! determined value on return. Auto-tuning is also used if the optional
! {\tt pipelineDepth} argument is omitted.
!
! \item [{[routehandle]}]
! Handle to the precomputed Route.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc, stat, i
    integer :: count
    type(ESMF_Field), allocatable :: srcFields(:), dstFields(:)
    integer :: rraShift, vectorLengthShift
    type(ESMF_RHL), pointer :: rhList, rhListE
    type(ESMF_Field), pointer :: srcFieldsPtr(:), dstFieldsPtr(:)
    logical :: verbosityFlag


!TODO: Remove code that is under all the -else branches after
!TODO: plenty of testing across systems, and no issues with the ESMF level
!TODO: implementation have been found or reported. I estimate this should happen
!TODO: right before the ESMF 8.6.0 release.
    verbosityFlag = .false.




    ! Initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! check variables
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

    ! consistency check counts
    if(srcFieldBundle%this%is_packed) then
      count = 1
    else
      call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=count, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(dstFieldBundle%this%is_packed) then
      i = 1
    else
      call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=i, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (i /= count) then
      call ESMF_LogSetError(ESMF_RC_ARG_BAD, &
        msg="Number of Fields in srcFieldBundle/dstFieldBundle must match!", &
        ESMF_CONTEXT, rcToReturn=rc)
      return ! bail out
    endif

    ! access the fields in the add order
    allocate(srcFields(count), dstFields(count), stat=stat)
    if (ESMF_LogFoundAllocError(statusToCheck=stat, &
      msg="Allocation of srcFields and dstFields.", &
      ESMF_CONTEXT, rcToReturn=rc)) &
      return ! bail out
    if(srcFieldBundle%this%is_packed) then
      allocate(srcFieldsPtr(1))
      call ESMF_ContainerGet(srcFieldBundle%this%container, &
        itemList=srcFieldsPtr, itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
      srcFields = srcFieldsPtr
      deallocate(srcFieldsPtr)
    else
      call ESMF_FieldBundleGet(srcFieldBundle, fieldList=srcFields, &
        itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
    endif
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(dstFieldBundle%this%is_packed) then
      allocate(dstFieldsPtr(1))
      call ESMF_ContainerGet(dstFieldBundle%this%container, &
        itemList=dstFieldsPtr, itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
      dstFields = dstFieldsPtr
      deallocate(dstFieldsPtr)
    else
      call ESMF_FieldBundleGet(dstFieldBundle, fieldList=dstFields, &
        itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
    endif
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! prepare Routehandle
    if (present(routehandle)) then
      ! create Routehandle
      routehandle = ESMF_RouteHandleCreate(rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
      ! set the type for bundle execution
      call ESMF_RouteHandlePrepXXE(routehandle, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! prepare auxiliary variables
    rraShift = 0 ! reset
    vectorLengthShift = 0 ! reset

    ! prepare rhList linked list
    nullify(rhList)

    ! loop over all fields
    do i=1, count

      call ESMF_FieldBundleRegridStorePair(srcFields(i), dstFields(i), &
        srcMaskValues, dstMaskValues, regridmethod, polemethod, &
        regridPoleNPnts, lineType, normType, extrapMethod, extrapNumSrcPnts, &
        extrapDistExponent, extrapNumLevels, unmappedaction, ignoreDegenerate, &
        srcTermProcessing, pipelineDepth, rhList, routehandle, rraShift, &
        vectorLengthShift, verbosityFlag, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    enddo

    ! take down rhList and destroy rh objects
    do while (associated(rhList))
      rhListE=>rhList
      rhList=>rhList%prev
      call ESMF_RouteHandleDestroy(rhListE%rh, noGarbage=.true., rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
      if (rhListE%factorAllocFlag) then
        deallocate(rhListE%factorIndexList)
        deallocate(rhListE%factorList)
      endif
      deallocate(rhListE%srcGridToFieldMap, rhListE%dstGridToFieldMap)
      deallocate(rhListE%srcUngriddedLBound, rhListE%srcUngriddedUBound)
      deallocate(rhListE%dstUngriddedLBound, rhListE%dstUngriddedUBound)
      deallocate(rhListE)
    enddo

    ! garbage collection
    deallocate(srcFields, dstFields)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleRegridStore
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRegridStorePair()"
!BOPI
! !IROUTINE: ESMF_FieldBundleRegridStorePair - Precompute a FieldBundle regrid operation for one pair
! !INTERFACE:
  subroutine ESMF_FieldBundleRegridStorePair(srcField, dstField, &
    srcMaskValues, dstMaskValues, regridmethod, polemethod, regridPoleNPnts, &
    lineType, normType, extrapMethod, extrapNumSrcPnts, extrapDistExponent, &
    extrapNumLevels, unmappedaction, ignoreDegenerate, srcTermProcessing, &
    pipelineDepth, rhList, routehandle, rraShift, vectorLengthShift, &
    verbosityFlag, rc)
!
! !ARGUMENTS:
    type(ESMF_Field), intent(in) :: srcField
    type(ESMF_Field), intent(inout) :: dstField
    integer(ESMF_KIND_I4), target, intent(in), optional :: srcMaskValues(:)
    integer(ESMF_KIND_I4), target, intent(in), optional :: dstMaskValues(:)
    type(ESMF_RegridMethod_Flag), intent(in), optional :: regridmethod
    type(ESMF_PoleMethod_Flag), intent(in), optional :: polemethod
    integer, intent(in), optional :: regridPoleNPnts
    type(ESMF_LineType_Flag), intent(in), optional :: lineType
    type(ESMF_NormType_Flag), intent(in), optional :: normType
    type(ESMF_ExtrapMethod_Flag), intent(in), optional :: extrapMethod
    integer, intent(in), optional :: extrapNumSrcPnts
    real, intent(in), optional :: extrapDistExponent
    integer, intent(in), optional :: extrapNumLevels
    type(ESMF_UnmappedAction_Flag), intent(in), optional :: unmappedaction
    logical, intent(in), optional :: ignoreDegenerate
    integer, intent(inout), optional :: srcTermProcessing
    integer, intent(inout), optional :: pipelineDepth
    type(ESMF_RHL), pointer :: rhList
    type(ESMF_RouteHandle), intent(inout) :: routehandle
    integer, intent(inout) :: rraShift
    integer, intent(inout) :: vectorLengthShift
    logical, intent(in), optional :: verbosityFlag
    integer, intent(out), optional :: rc
!EOPI
!------------------------------------------------------------------------------
    integer :: localrc
    integer :: j, localDeCount
    integer(ESMF_KIND_I4), pointer :: factorIndexList(:,:)
    real(ESMF_KIND_R8), pointer :: factorList(:)
    type(ESMF_GeomType_Flag) :: srcGeomtype, dstGeomtype
    type(ESMF_Grid) :: srcGrid, dstGrid
    type(ESMF_Mesh) :: srcMesh, dstMesh
    type(ESMF_ArraySpec) :: srcArraySpec, dstArraySpec
    type(ESMF_StaggerLoc) :: srcStaggerLoc, dstStaggerLoc
    type(ESMF_MeshLoc) :: srcMeshLoc, dstMeshLoc
    integer, pointer :: srcGridToFieldMap(:)
    integer, pointer :: dstGridToFieldMap(:)
    integer, pointer :: srcUngriddedLBound(:)
    integer, pointer :: srcUngriddedUBound(:)
    integer, pointer :: dstUngriddedLBound(:)
    integer, pointer :: dstUngriddedUBound(:)
    integer :: fieldDimCount, geomDimCount
    logical :: rhReuse
    type(ESMF_RHL), pointer :: rhListE, rhListG
    logical :: rhListMatch
    type(ESMF_RouteHandle) :: rhh
    logical :: verbosityFlagIntl
    character(ESMF_MAXSTR) :: fName
    integer :: dimCount, rank
    character(800) :: msgString

    verbosityFlagIntl = .false. ! default
    if (present(verbosityFlag)) verbosityFlagIntl = verbosityFlag

    call ESMF_FieldGet(srcField, geomtype=srcGeomtype, arrayspec=srcArraySpec, &
      dimCount=fieldDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return
    call ESMF_FieldGet(dstField, geomtype=dstGeomtype, arrayspec=dstArraySpec, &
      rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if (verbosityFlagIntl) then
      call ESMF_FieldGet(srcField, name=fName, dimCount=dimCount, rank=rank, &
        rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
      write(msgString,'(4A,I2,A,I2)') ESMF_METHOD, ": srcField: ", trim(fName), &
        " dimCount=", dimCount, " rank=", rank
      call ESMF_LogWrite(msgString, ESMF_LOGMSG_DEBUG, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
      call ESMF_FieldGet(dstField, name=fName, dimCount=dimCount, rank=rank, &
        rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
      write(msgString,'(4A,I2,A,I2)') ESMF_METHOD, ": dstField: ", trim(fName), &
        " dimCount=", dimCount, " rank=", rank
      call ESMF_LogWrite(msgString, ESMF_LOGMSG_DEBUG, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    rhReuse = ((srcGeomtype==ESMF_GEOMTYPE_GRID) .or. &
               (srcGeomtype==ESMF_GEOMTYPE_MESH))
    rhReuse = rhReuse .and. ((dstGeomtype==ESMF_GEOMTYPE_GRID) .or. &
                             (dstGeomtype==ESMF_GEOMTYPE_MESH))

    rhListMatch = .false.
    rhListG=>NULL()

    if (rhReuse) then

      if (verbosityFlagIntl) then
        call ESMF_LogWrite(ESMF_StringConcat(ESMF_METHOD, &
          ": Inside 'rhReuse' branch!"), ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      ! access src side information
      if (srcGeomtype==ESMF_GEOMTYPE_GRID) then
        call ESMF_FieldGet(srcField, grid=srcGrid, staggerLoc=srcStaggerLoc, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_GridGet(srcGrid, dimCount=geomDimCount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        if (verbosityFlagIntl) then
          write(msgString,'(2A,I2)') ESMF_METHOD, &
            ": srcGeomtype==ESMF_GEOMTYPE_GRID dimCount=", geomDimCount
          call ESMF_LogWrite(msgString, ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
      else if (srcGeomtype==ESMF_GEOMTYPE_MESH) then
        call ESMF_FieldGet(srcField, mesh=srcMesh, meshLoc=srcMeshLoc, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        geomDimCount = 1 ! Mesh always stored on 1D DistGrid
        if (verbosityFlagIntl) then
          write(msgString,'(2A,I2)') ESMF_METHOD, &
            ": srcGeomtype==ESMF_GEOMTYPE_MESH dimCount=", geomDimCount
          call ESMF_LogWrite(msgString, ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
      endif
      allocate(srcGridToFieldMap(geomDimCount))
      allocate(srcUngriddedLBound(fieldDimCount-geomDimCount), &
        srcUngriddedUBound(fieldDimCount-geomDimCount))
      call ESMF_FieldGet(srcField, gridToFieldMap=srcGridToFieldMap, &
        ungriddedLBound=srcUngriddedLBound, &
        ungriddedUBound=srcUngriddedUBound,rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! access dst side information
      if (dstGeomtype==ESMF_GEOMTYPE_GRID) then
        call ESMF_FieldGet(dstField, grid=dstGrid, staggerLoc=dstStaggerLoc, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_GridGet(dstGrid, dimCount=geomDimCount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        if (verbosityFlagIntl) then
          write(msgString,'(2A,I2)') ESMF_METHOD, &
            ": dstGeomtype==ESMF_GEOMTYPE_GRID dimCount=", geomDimCount
          call ESMF_LogWrite(msgString, ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
      else if (dstGeomtype==ESMF_GEOMTYPE_MESH) then
        call ESMF_FieldGet(dstField, mesh=dstMesh, meshLoc=dstMeshLoc, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        geomDimCount = 1 ! Mesh always stored on 1D DistGrid
        if (verbosityFlagIntl) then
          write(msgString,'(2A,I2)') ESMF_METHOD, &
            ": dstGeomtype==ESMF_GEOMTYPE_MESH dimCount=", geomDimCount
          call ESMF_LogWrite(msgString, ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
      endif
      allocate(dstGridToFieldMap(geomDimCount))
      allocate(dstUngriddedLBound(fieldDimCount-geomDimCount), &
        dstUngriddedUBound(fieldDimCount-geomDimCount))
      call ESMF_FieldGet(dstField, gridToFieldMap=dstGridToFieldMap, &
        ungriddedLBound=dstUngriddedLBound, &
        ungriddedUBound=dstUngriddedUBound,rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! search for a match
      rhListE=>rhList
      do while (associated(rhListE))
        ! test src geomtype match
        rhListMatch = (rhListE%srcGeomtype==srcGeomtype)
        if (.not.rhListMatch) goto 123
        if (srcGeomtype==ESMF_GEOMTYPE_GRID) then
          ! test src grid match
          rhListMatch = ESMF_GridMatch(rhListE%srcGrid, srcGrid, &
            globalflag=.true., rc=localrc) >= ESMF_GRIDMATCH_EXACT
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
          if (.not.rhListMatch) goto 123
          ! test src staggerLoc match
          rhListMatch = (rhListE%srcStaggerLoc==srcStaggerLoc)
          if (.not.rhListMatch) goto 123
        else if (srcGeomtype==ESMF_GEOMTYPE_MESH) then
          ! test src mesh match
          rhListMatch = (rhListE%srcMesh==srcMesh) ! simple alias check only
          if (.not.rhListMatch) goto 123
          ! test src meshLoc match
          rhListMatch = (rhListE%srcMeshLoc==srcMeshLoc)
          if (.not.rhListMatch) goto 123
        endif
        ! test dst geomtype match
        rhListMatch = (rhListE%dstGeomtype==dstGeomtype)
        if (.not.rhListMatch) goto 123
        if (dstGeomtype==ESMF_GEOMTYPE_GRID) then
          ! test dst grid match
          rhListMatch = ESMF_GridMatch(rhListE%dstGrid, dstGrid, &
            globalflag=.true., rc=localrc) >= ESMF_GRIDMATCH_EXACT
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
          if (.not.rhListMatch) goto 123
          ! test dst staggerLoc match
          rhListMatch = (rhListE%dstStaggerLoc==dstStaggerLoc)
          if (.not.rhListMatch) goto 123
        else if (dstGeomtype==ESMF_GEOMTYPE_MESH) then
          ! test dst mesh match
          rhListMatch = (rhListE%dstMesh==dstMesh) ! simple alias check only
          if (.not.rhListMatch) goto 123
          ! test dst meshLoc match
          rhListMatch = (rhListE%dstMeshLoc==dstMeshLoc)
          if (.not.rhListMatch) goto 123
        endif
        ! keep record of entry for which geom objects and loc are matching
        rhListG=>rhListE
        ! test src arrayspec match
        rhListMatch = (rhListE%srcArraySpec==srcArraySpec)
        if (.not.rhListMatch) goto 123
        ! test dst arrayspec match
        rhListMatch = (rhListE%dstArraySpec==dstArraySpec)
        if (.not.rhListMatch) goto 123
        ! test srcGridToFieldMap
        rhListMatch = &
          (size(rhListE%srcGridToFieldMap)==size(srcGridToFieldMap))
        if (.not.rhListMatch) goto 123
        do j=1, size(srcGridToFieldMap)
          rhListMatch = (rhListE%srcGridToFieldMap(j)==srcGridToFieldMap(j))
          if (.not.rhListMatch) goto 123
        enddo
        ! test dstGridToFieldMap
        rhListMatch = &
          (size(rhListE%dstGridToFieldMap)==size(dstGridToFieldMap))
        if (.not.rhListMatch) goto 123
        do j=1, size(dstGridToFieldMap)
          rhListMatch = (rhListE%dstGridToFieldMap(j)==dstGridToFieldMap(j))
          if (.not.rhListMatch) goto 123
        enddo
        ! test srcUngriddedLBound
        rhListMatch = &
          (size(rhListE%srcUngriddedLBound)==size(srcUngriddedLBound))
        if (.not.rhListMatch) goto 123
        do j=1, size(srcUngriddedLBound)
          rhListMatch = (rhListE%srcUngriddedLBound(j)==srcUngriddedLBound(j))
          if (.not.rhListMatch) goto 123
        enddo
        ! test srcUngriddedUBound
        rhListMatch = &
          (size(rhListE%srcUngriddedUBound)==size(srcUngriddedUBound))
        if (.not.rhListMatch) goto 123
        do j=1, size(srcUngriddedUBound)
          rhListMatch = (rhListE%srcUngriddedUBound(j)==srcUngriddedUBound(j))
          if (.not.rhListMatch) goto 123
        enddo
        ! test dstUngriddedLBound
        rhListMatch = &
          (size(rhListE%dstUngriddedLBound)==size(dstUngriddedLBound))
        if (.not.rhListMatch) goto 123
        do j=1, size(dstUngriddedLBound)
          rhListMatch = (rhListE%dstUngriddedLBound(j)==dstUngriddedLBound(j))
          if (.not.rhListMatch) goto 123
        enddo
        ! test dstUngriddedUBound
        rhListMatch = &
          (size(rhListE%dstUngriddedUBound)==size(dstUngriddedUBound))
        if (.not.rhListMatch) goto 123
        do j=1, size(dstUngriddedUBound)
          rhListMatch = (rhListE%dstUngriddedUBound(j)==dstUngriddedUBound(j))
          if (.not.rhListMatch) goto 123
        enddo
        ! test regridmethod
        if (rhListE%guard_regridmethod .and. present(regridmethod)) then
          rhListMatch = (rhListE%regridmethod==regridmethod)
          if (.not.rhListMatch) goto 123
        endif
        ! test extrapMethod
        if (rhListE%guard_extrapMethod .and. present(extrapMethod)) then
          rhListMatch = (rhListE%extrapMethod==extrapMethod)
          if (.not.rhListMatch) goto 123
        endif
        ! test extrapNumSrcPnts
        if (rhListE%guard_extrapNumSrcPnts .and. present(extrapNumSrcPnts)) then
          rhListMatch = (rhListE%extrapNumSrcPnts==extrapNumSrcPnts)
          if (.not.rhListMatch) goto 123
        endif
        ! test extrapDistExponent
        if (rhListE%guard_extrapDistExponent .and. present(extrapDistExponent)) then
          rhListMatch = (rhListE%extrapDistExponent==extrapDistExponent)
          if (.not.rhListMatch) goto 123
        endif
        ! test extrapNumLevels
        if (rhListE%guard_extrapNumLevels .and. present(extrapNumLevels)) then
          rhListMatch = (rhListE%extrapNumLevels==extrapNumLevels)
          if (.not.rhListMatch) goto 123
        endif
        ! test ignoreDegenerate
        if (rhListE%guard_ignoreDegenerate .and. present(ignoreDegenerate)) then
          rhListMatch = (rhListE%ignoreDegenerate.eqv.ignoreDegenerate)
          if (.not.rhListMatch) goto 123
        endif
        ! test srcMaskValues
        if (rhListE%guard_srcMaskValues .and. present(srcMaskValues)) then
          rhListMatch = (size(rhListE%srcMaskValues)==size(srcMaskValues))
          if (.not.rhListMatch) goto 123
          do j=1, size(srcMaskValues)
            rhListMatch = (rhListE%srcMaskValues(j)==srcMaskValues(j))
            if (.not.rhListMatch) goto 123
          enddo
        endif
        ! test dstMaskValues
        if (rhListE%guard_dstMaskValues .and. present(dstMaskValues)) then
          rhListMatch = (size(rhListE%dstMaskValues)==size(dstMaskValues))
          if (.not.rhListMatch) goto 123
          do j=1, size(dstMaskValues)
            rhListMatch = (rhListE%dstMaskValues(j)==dstMaskValues(j))
            if (.not.rhListMatch) goto 123
          enddo
        endif
        ! test polemethod
        if (rhListE%guard_polemethod .and. present(polemethod)) then
          rhListMatch = (rhListE%polemethod==polemethod)
          if (.not.rhListMatch) goto 123
        endif
        ! test regridPoleNPnts
        if (rhListE%guard_regridPoleNPnts .and. present(regridPoleNPnts)) then
          rhListMatch = (rhListE%regridPoleNPnts==regridPoleNPnts)
          if (.not.rhListMatch) goto 123
        endif
        ! test unmappedaction
        if (rhListE%guard_unmappedaction .and. present(unmappedaction)) then
          rhListMatch = (rhListE%unmappedaction==unmappedaction)
          if (.not.rhListMatch) goto 123
        endif
        ! completed search
        exit ! break out
        ! continue search with previous element
123 continue
        rhListE=>rhListE%prev ! previous element
      enddo

    endif

    if (.not.rhListMatch) then
      ! No match found: precompute new RH for this field pair
      if (verbosityFlagIntl) then
        call ESMF_LogWrite(ESMF_StringConcat(ESMF_METHOD, &
          ": no rhListMatch -> pre-compute new remapping!"), &
          ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif
      if (rhReuse) then
        ! add a new rhList element
        allocate(rhListE)
        rhListE%prev=>rhList ! link new element to previous list head
        rhList=>rhListE ! list head now pointing to new element
      endif

      if (associated(rhListG)) then
        ! able to reuse already precomputed weight matrix
        if (verbosityFlagIntl) then
          call ESMF_LogWrite(ESMF_StringConcat(ESMF_METHOD, &
            ": able to reuse already precomputed weight matrix!"), &
            ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        factorList=>rhListG%factorList
        factorIndexList=>rhListG%factorIndexList
        call ESMF_FieldSMMStore(srcField=srcField, &
          dstField=dstField, &
          routehandle=rhh, &
          factorList=factorList, factorIndexList=factorIndexList, &
          srcTermProcessing=srcTermProcessing, pipelineDepth=pipelineDepth, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      else
        ! must precompute full regridding
        if (verbosityFlagIntl) then
          call ESMF_LogWrite(ESMF_StringConcat(ESMF_METHOD, &
            ": must precompute full regridding!"), &
            ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        factorList=>NULL()
        factorIndexList=>NULL()
        call ESMF_FieldRegridStore(srcField=srcField, &
          dstField=dstField, &
          srcMaskValues=srcMaskValues, dstMaskValues=dstMaskValues, &
          regridmethod=regridmethod, &
          polemethod=polemethod, regridPoleNPnts=regridPoleNPnts, &
          lineType=lineType, normType=normType, &
          extrapMethod=extrapMethod, extrapNumSrcPnts=extrapNumSrcPnts, &
          extrapDistExponent=extrapDistExponent, &
          extrapNumLevels=extrapNumLevels, &
          unmappedaction=unmappedaction, ignoreDegenerate=ignoreDegenerate, &
          srcTermProcessing=srcTermProcessing, pipelineDepth=pipelineDepth, &
          routehandle=rhh, &
          factorIndexList=factorIndexList, factorList=factorList, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      if (rhReuse) then
        ! store info in the new rhList element
        rhListE%srcGeomtype = srcGeomtype
        rhListE%dstGeomtype = dstGeomtype
        rhListE%srcMesh = srcMesh
        rhListE%dstMesh = dstMesh
        rhListE%srcGrid = srcGrid
        rhListE%dstGrid = dstGrid
        rhListE%srcArraySpec = srcArraySpec
        rhListE%dstArraySpec = dstArraySpec
        rhListE%srcStaggerLoc = srcStaggerLoc
        rhListE%dstStaggerLoc = dstStaggerLoc
        rhListE%srcMeshLoc = srcMeshLoc
        rhListE%dstMeshLoc = dstMeshLoc
        rhListE%srcGridToFieldMap => srcGridToFieldMap
        rhListE%dstGridToFieldMap => dstGridToFieldMap
        rhListE%srcUngriddedLBound => srcUngriddedLBound
        rhListE%srcUngriddedUBound => srcUngriddedUBound
        rhListE%dstUngriddedLBound => dstUngriddedLBound
        rhListE%dstUngriddedUBound => dstUngriddedUBound
        rhListE%factorAllocFlag = .not.associated(rhListG)
        ! store guard information
        rhListE%guard_srcMaskValues = present(srcMaskValues)
        rhListE%guard_dstMaskValues = present(dstMaskValues)
        rhListE%guard_regridmethod = present(regridmethod)
        rhListE%guard_polemethod = present(polemethod)
        rhListE%guard_regridPoleNPnts = present(regridPoleNPnts)
        rhListE%guard_lineType = present(lineType)
        rhListE%guard_normType = present(normType)
        rhListE%guard_extrapMethod = present(extrapMethod)
        rhListE%guard_extrapNumSrcPnts = present(extrapNumSrcPnts)
        rhListE%guard_extrapDistExponent = present(extrapDistExponent)
        rhListE%guard_extrapNumLevels = present(extrapNumLevels)
        rhListE%guard_unmappedaction = present(unmappedaction)
        rhListE%guard_ignoreDegenerate = present(ignoreDegenerate)
        rhListE%guard_srcTermProcessing = present(srcTermProcessing)
        rhListE%guard_pipelineDepth = present(pipelineDepth)
        ! store actual regrid options
        if (rhListE%guard_srcMaskValues) &
          rhListE%srcMaskValues => srcMaskValues
        if (rhListE%guard_dstMaskValues) &
          rhListE%dstMaskValues => dstMaskValues
        if (rhListE%guard_regridmethod) &
          rhListE%regridmethod = regridmethod
        if (rhListE%guard_polemethod) &
          rhListE%polemethod = polemethod
        if (rhListE%guard_regridPoleNPnts) &
          rhListE%regridPoleNPnts = regridPoleNPnts
        if (rhListE%guard_lineType) &
          rhListE%lineType = lineType
        if (rhListE%guard_normType) &
          rhListE%normType = normType
        if (rhListE%guard_extrapMethod) &
          rhListE%extrapMethod = extrapMethod
        if (rhListE%guard_extrapNumSrcPnts) &
          rhListE%extrapNumSrcPnts = extrapNumSrcPnts
        if (rhListE%guard_extrapDistExponent) &
          rhListE%extrapDistExponent = extrapDistExponent
        if (rhListE%guard_extrapNumLevels) &
          rhListE%extrapNumLevels = extrapNumLevels
        if (rhListE%guard_unmappedaction) &
          rhListE%unmappedaction = unmappedaction
        if (rhListE%guard_ignoreDegenerate) &
          rhListE%ignoreDegenerate = ignoreDegenerate
        if (rhListE%guard_srcTermProcessing) &
          rhListE%srcTermProcessing = srcTermProcessing
        if (rhListE%guard_pipelineDepth) &
          rhListE%pipelineDepth = pipelineDepth
        ! misc info
        rhListE%rh = rhh
        rhListE%factorIndexList => factorIndexList
        rhListE%factorList => factorList
      endif
    else
      ! Match found: reuse previous RH for this field pair
      if (verbosityFlagIntl) then
        call ESMF_LogWrite(ESMF_StringConcat(ESMF_METHOD, &
          ": found rhListMatch -> reuse routehandle!"), &
          ESMF_LOGMSG_DEBUG, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
      endif
      ! pull out the routehandle from the matching rhList element
      rhh = rhListE%rh
      factorIndexList => rhListE%factorIndexList
      factorList => rhListE%factorList
      ! deallocate temporary grid/field info
      deallocate(srcGridToFieldMap, dstGridToFieldMap)
      deallocate(srcUngriddedLBound, srcUngriddedUBound)
      deallocate(dstUngriddedLBound, dstUngriddedUBound)
    endif

    ! append rhh to routehandle and clear rhh
    call ESMF_RouteHandleAppend(routehandle, appendRoutehandle=rhh, &
      rraShift=rraShift, vectorLengthShift=vectorLengthShift, &
      transferflag=.not.rhListMatch, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! adjust rraShift and vectorLengthShift
    call ESMF_FieldGet(srcField, localDeCount=localDeCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return
    rraShift = rraShift + localDeCount
    call ESMF_FieldGet(dstField, localDeCount=localDeCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return
    rraShift = rraShift + localDeCount
    vectorLengthShift = vectorLengthShift + 1
    ! local garbage collection
    if (.not.rhReuse) then
      ! for rhReuse factorIndexList and factorList allocs managed by rhList
      if (associated(factorIndexList)) deallocate(factorIndexList)
      if (associated(factorList)) deallocate(factorList)
    endif

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleRegridStorePair
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRemove()"
!BOP
! !IROUTINE: ESMF_FieldBundleRemove - Remove Fields from FieldBundle
!
! !INTERFACE:
  subroutine ESMF_FieldBundleRemove(fieldbundle, fieldNameList, &
    keywordEnforcer, multiflag, relaxedflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    character(len=*), intent(in) :: fieldNameList(:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: multiflag
    logical, intent(in), optional :: relaxedflag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Remove field(s) by name from FieldBundle. In the relaxed setting it is
! {\em not} an error if {\tt fieldNameList} contains names that are not
! found in {\tt fieldbundle}.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} from which to remove items.
! \item [fieldNameList]
! List of items to remove.
! \item [{[multiflag]}]
! A setting of {\tt .true.} allows multiple Fields with the same name
! to be removed from {\tt fieldbundle}. For {\tt .false.}, items to be
! removed must have unique names. The default setting is {\tt .false.}.
! \item [{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "remove"
! where it is {\em not} an error if {\tt fieldNameList} contains item
! names that are not found in {\tt fieldbundle}. For {\tt .false.} this is
! treated as an error condition.
! Further, in {\tt multiflag=.false.} mode, the relaxed definition of
! "remove" also covers the case where there are multiple items in
! {\tt fieldbundle} that match a single entry in {\tt fieldNameList}.
! For {\tt relaxedflag=.false.} this is treated as an error condition.
! The default setting is {\tt .false.}.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: fieldCount, i, fcount
    type(ESMF_Field), pointer :: garbageList(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    nullify(garbageList)

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    call ESMF_ContainerGarbageOn(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageClear(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Call into ContainerRemove
    call ESMF_ContainerRemove(fieldbundle%this%container, fieldNameList, &
      multiflag=multiflag, relaxedflag=relaxedflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageGet(fieldbundle%this%container, garbageList=garbageList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageOff(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    deallocate(garbageList)

    ! Check if the fieldbundle is empty
    call ESMF_FieldBundleGet(fieldbundle, fieldCount=fcount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(fcount == 0) then
      call ESMF_FieldBundleRemoveGeom(fieldbundle, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleRemove
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleReplace()"
!BOP
! !IROUTINE: ESMF_FieldBundleReplace - Replace Fields in FieldBundle
!
! !INTERFACE:
  subroutine ESMF_FieldBundleReplace(fieldbundle, fieldList, &
    keywordEnforcer, multiflag, relaxedflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_Field), intent(in) :: fieldList(:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: multiflag
    logical, intent(in), optional :: relaxedflag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Replace field(s) by name in FieldBundle. In the relaxed setting it is not
! an error if {\tt fieldList} contains Fields that do not match by name any
! item in {\tt fieldbundle}. These Fields are simply ignored in this case.
!
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} in which to replace items.
! \item [fieldList]
! List of items to replace.
! \item [{[multiflag]}]
! A setting of {\tt .true.} allows multiple items with the same name
! to be replaced in {\tt fieldbundle}. For {\tt .false.}, items to be
! replaced must have unique names. The default setting is {\tt .false.}.
! \item [{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "replace"
! where it is {\em not} an error if {\tt fieldList} contains items with
! names that are not found in {\tt fieldbundle}. These items in
! {\tt fieldList} are ignored in the relaxed mode. For {\tt .false.} this
! is treated as an error condition.
! Further, in {\tt multiflag=.false.} mode, the relaxed definition of
! "replace" also covers the case where there are multiple items in
! {\tt fieldbundle} that match a single entry by name in {\tt fieldList}.
! For {\tt relaxedflag=.false.} this is treated as an error condition.
! The default setting is {\tt .false.}.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    integer :: fieldCount, i
    type(ESMF_Field), pointer :: garbageList(:)

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    nullify(garbageList)

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    call ESMF_ContainerGarbageOn(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageClear(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Call into ContainerReplace
    call ESMF_ContainerReplace(fieldbundle%this%container, fieldList, &
      multiflag=multiflag, relaxedflag=relaxedflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageGet(fieldbundle%this%container, garbageList=garbageList, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_ContainerGarbageOff(fieldbundle%this%container, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Attribute link
    fieldCount = size(fieldList)
    if(fieldCount .ge. 1 .and. (fieldbundle%this%status /= ESMF_FBSTATUS_GRIDSET) ) then
      ! setgeom links grid geom automatically
      call ESMF_FieldBundleSetGeom(fieldbundle, fieldList(1), rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif ! non-empty fieldlist

    deallocate(garbageList)

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleReplace
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSetGrid"
!BOP
! !IROUTINE: ESMF_FieldBundleSet - Associate a Grid with an empty FieldBundle
!
! !INTERFACE:
      ! Private name; call using ESMF_FieldBundleSet()
      subroutine ESMF_FieldBundleSetGrid(fieldbundle, grid, keywordEnforcer, rc)
!
! !ARGUMENTS:
      type(ESMF_FieldBundle), intent(inout) :: fieldbundle
      type(ESMF_Grid), intent(in) :: grid
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! \begin{sloppypar}
! Sets the {\tt grid} for a {\tt fieldbundle}.
! \end{sloppypar}
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! An {\tt ESMF\_FieldBundle} object.
! \item [grid]
! The {\tt ESMF\_Grid} which all {\tt ESMF\_Field}s added to this
! {\tt ESMF\_FieldBundle} must have.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!
!EOP


      integer :: status ! Error status
      type(ESMF_FieldBundleType), pointer :: btype ! internal data

      ! Initialize return code; assume routine not implemented
      status = ESMF_RC_NOT_IMPL
      if (present(rc)) rc = ESMF_RC_NOT_IMPL

      ! check variables
      ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)
      ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)

      btype => fieldbundle%this

      ! here we will only let someone associate a grid with a fieldbundle
      ! if there is not one already associated with it.
      if (btype%status == ESMF_FBSTATUS_GRIDSET) then
        if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
          msg="FieldBundle is already associated with a Geom", &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      ! Create the geom around the grid, use the center stagger as a generic stagger here,
      ! because the stagger wont really matter in this case
      btype%geom=ESMF_GeomCreate(grid,ESMF_STAGGERLOC_CENTER,rc=status)
      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                   ESMF_CONTEXT, rcToReturn=rc)) return

      ! Set Status to containing a Grid
      btype%status = ESMF_FBSTATUS_GRIDSET

      if (present(rc)) rc = ESMF_SUCCESS

      end subroutine ESMF_FieldBundleSetGrid
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSetMesh"
!BOP
! !IROUTINE: ESMF_FieldBundleSet - Associate a Mesh with an empty FieldBundle
!
! !INTERFACE:
      ! Private name; call using ESMF_FieldBundleSet()
      subroutine ESMF_FieldBundleSetMesh(fieldbundle, mesh, keywordEnforcer, rc)
!
! !ARGUMENTS:
      type(ESMF_FieldBundle), intent(inout) :: fieldbundle
      type(ESMF_Mesh), intent(in) :: mesh
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! \begin{sloppypar}
! Sets the {\tt mesh} for a {\tt fieldbundle}.
! \end{sloppypar}
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! An {\tt ESMF\_FieldBundle} object.
! \item [mesh]
! The {\tt ESMF\_Mesh} which all {\tt ESMF\_Field}s added to this
! {\tt ESMF\_FieldBundle} must have.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!
!EOP


      integer :: status ! Error status
      type(ESMF_FieldBundleType), pointer :: btype ! internal data

      ! Initialize return code; assume routine not implemented
      status = ESMF_RC_NOT_IMPL
      if (present(rc)) rc = ESMF_RC_NOT_IMPL

      ! check variables
      ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)
      ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)

      btype => fieldbundle%this

      ! here we will only let someone associate a grid with a fieldbundle
      ! if there is not one already associated with it.
      if (btype%status == ESMF_FBSTATUS_GRIDSET) then
        if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
          msg="FieldBundle is already associated with a Geom", &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      ! Create the geom around the grid, use the center stagger as a generic stagger here,
      ! because the stagger wont really matter in this case
      btype%geom=ESMF_GeomCreate(mesh,rc=status)
      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                   ESMF_CONTEXT, rcToReturn=rc)) return

      ! Set Status to containing a Geom
      btype%status = ESMF_FBSTATUS_GRIDSET

      if (present(rc)) rc = ESMF_SUCCESS

      end subroutine ESMF_FieldBundleSetMesh
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSetLS"
!BOP
! !IROUTINE: ESMF_FieldBundleSet - Associate a LocStream with an empty FieldBundle
!
! !INTERFACE:
      ! Private name; call using ESMF_FieldBundleSet()
      subroutine ESMF_FieldBundleSetLS(fieldbundle, locstream, &
        keywordEnforcer, rc)
!
! !ARGUMENTS:
      type(ESMF_FieldBundle), intent(inout) :: fieldbundle
      type(ESMF_LocStream), intent(in) :: locstream
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! \begin{sloppypar}
! Sets the {\tt locstream} for a {\tt fieldbundle}.
! \end{sloppypar}
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! An {\tt ESMF\_FieldBundle} object.
! \item [locstream]
! The {\tt ESMF\_LocStream} which all {\tt ESMF\_Field}s added to this
! {\tt ESMF\_FieldBundle} must have.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!
!EOP


      integer :: status ! Error status
      type(ESMF_FieldBundleType), pointer :: btype ! internal data

      ! Initialize return code; assume routine not implemented
      status = ESMF_RC_NOT_IMPL
      if (present(rc)) rc = ESMF_RC_NOT_IMPL

      ! check variables
      ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)
      ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)

      btype => fieldbundle%this

      ! here we will only let someone associate a grid with a fieldbundle
      ! if there is not one already associated with it.
      if (btype%status == ESMF_FBSTATUS_GRIDSET) then
        if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
          msg="FieldBundle is already associated with a Geom", &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

       ! Create the geom around the locstream
       btype%geom=ESMF_GeomCreate(locstream, rc=status)
       if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                    ESMF_CONTEXT, rcToReturn=rc)) return

      ! Set Status to containing a Geom
      btype%status = ESMF_FBSTATUS_GRIDSET

      if (present(rc)) rc = ESMF_SUCCESS

      end subroutine ESMF_FieldBundleSetLS
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSetXGrid"
!BOP
! !IROUTINE: ESMF_FieldBundleSet - Associate a XGrid with an empty FieldBundle
!
! !INTERFACE:
      ! Private name; call using ESMF_FieldBundleSet()
      subroutine ESMF_FieldBundleSetXGrid(fieldbundle, xgrid, &
        keywordEnforcer, rc)
!
! !ARGUMENTS:
      type(ESMF_FieldBundle), intent(inout) :: fieldbundle
      type(ESMF_XGrid), intent(in) :: xgrid
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! \begin{sloppypar}
! Sets the {\tt xgrid} for a {\tt fieldbundle}
! \end{sloppypar}
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! An {\tt ESMF\_FieldBundle} object.
! \item [xgrid]
! The {\tt ESMF\_XGrid} which all {\tt ESMF\_Field}s added to this
! {\tt ESMF\_FieldBundle} must have.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!
!EOP

      integer :: status ! Error status
      type(ESMF_FieldBundleType), pointer :: btype ! internal data

      ! Initialize return code; assume routine not implemented
      status = ESMF_RC_NOT_IMPL
      if (present(rc)) rc = ESMF_RC_NOT_IMPL

      ! check variables
      ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)
      ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)

      btype => fieldbundle%this

      ! here we will only let someone associate a grid with a fieldbundle
      ! if there is not one already associated with it.
      if (btype%status == ESMF_FBSTATUS_GRIDSET) then
        if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
          msg="FieldBundle is already associated with a Geom", &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

       ! Create the geom around the locstream
       btype%geom=ESMF_GeomCreate(xgrid, rc=status)
       if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                    ESMF_CONTEXT, rcToReturn=rc)) return

      ! Set Status to containing a Geom
      btype%status = ESMF_FBSTATUS_GRIDSET

      if (present(rc)) rc = ESMF_SUCCESS

      end subroutine ESMF_FieldBundleSetXGrid
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMM()"
!BOP
! !IROUTINE: ESMF_FieldBundleSMM - Execute a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  subroutine ESMF_FieldBundleSMM(srcFieldBundle, dstFieldBundle, &
    routehandle, keywordEnforcer, &
    zeroregion, & ! DEPRECATED ARGUMENT
    zeroregionflag, termorderflag, checkflag, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in), optional :: srcFieldBundle
    type(ESMF_FieldBundle), intent(inout), optional :: dstFieldBundle
    type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Region_Flag), intent(in), optional :: zeroregion ! DEPRECATED ARGUMENT
    type(ESMF_Region_Flag), intent(in), target, optional :: zeroregionflag(:)
    type(ESMF_TermOrder_Flag), intent(in), optional :: termorderflag(:)
    logical, intent(in), optional :: checkflag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[7.0.0] Added argument {\tt termorderflag}.
! The new argument gives the user control over the order in which
! the src terms are summed up.
! \item[8.1.0] Added argument {\tt zeroregionflag}, and deprecated
! {\tt zeroregion}. The new argument allows greater flexibility
! in setting the zero region for individual FieldBundle members.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Execute a precomputed sparse matrix multiplication from {\tt srcFieldBundle}
! to {\tt dstFieldBundle}.
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} must match the
! respective FieldBundles used during {\tt ESMF\_FieldBundleRedistStore()}
! in {\em type}, {\em kind}, and memory layout of the {\em gridded}
! dimensions. However, the size, number,
! and index order of {\em ungridded} dimensions may be different. See section
! \ref{RH:Reusability} for a more detailed discussion of RouteHandle
! reusability.
!
! The {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments are optional in support of
! the situation where {\tt srcFieldBundle} and/or {\tt dstFieldBundle} are not defined on
! all PETs. The {\tt srcFieldBundle} and {\tt dstFieldBundle} must be specified on those
! PETs that hold source or destination DEs, respectively, but may be omitted
! on all other PETs. PETs that hold neither source nor destination DEs may
! omit both arguments.
!
! It is erroneous to specify the identical FieldBundle object for {\tt srcFieldBundle} and
! {\tt dstFieldBundle} arguments.
!
! See {\tt ESMF\_FieldBundleSMMStore()} on how to precompute
! {\tt routehandle}.
!
! This call is {\em collective} across the current VM.
!
! For examples and associated documentation regarding this method see Section
! \ref{sec:fieldbundle:usage:smm_1dptr}.
!
! \begin{description}
! \item [{[srcFieldBundle]}]
! {\tt ESMF\_FieldBundle} with source data.
! \item [{[dstFieldBundle]}]
! {\tt ESMF\_FieldBundle} with destination data.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[zeroregion]}]
! If set to {\tt ESMF\_REGION\_TOTAL} {\em (default)} the total regions of
! all DEs in all Fields in {\tt dstFieldBundle} will be initialized to zero
! before updating the elements with the results of the sparse matrix
! multiplication. If set to {\tt ESMF\_REGION\_EMPTY} the elements in the
! Fields in {\tt dstFieldBundle} will not be modified prior to the sparse
! matrix multiplication and results will be added to the incoming element
! values. Setting {\tt zeroregion} to {\tt ESMF\_REGION\_SELECT} will only
! zero out those elements in the destination Fields that will be updated
! by the sparse matrix multiplication. See section \ref{const:region}
! \item [{[zeroregionflag]}]
! If set to {\tt ESMF\_REGION\_TOTAL} {\em (default)} the total regions of
! all DEs in the destination Field will be initialized to zero
! before updating the elements with the results of the sparse matrix
! multiplication. If set to {\tt ESMF\_REGION\_EMPTY} the elements in the
! destination Field will not be modified prior to the sparse
! matrix multiplication and results will be added to the incoming element
! values. A setting of {\tt ESMF\_REGION\_SELECT} will only
! zero out those elements in the destination Field that will be updated
! by the sparse matrix multiplication. See section \ref{const:region}
! for a complete list of valid settings.
! The size of this array argument must either be 1 or equal the number of
! Fields in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the zero region for each Field SMM operation is
! indicated separately. If only one zero region element is specified, it is
! used for {\em all} Field pairs.
! \item [{[termorderflag]}]
! Specifies the order of the source side terms in all of the destination
! sums. The {\tt termorderflag} only affects the order of terms during
! the execution of the RouteHandle. See the \ref{RH:bfb} section for an
! in-depth discussion of {\em all} bit-for-bit reproducibility
! aspects related to route-based communication methods.
! See \ref{const:termorderflag} for a full list of options.
! The size of this array argument must either be 1 or equal the number of
! Fields in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the term order for each Field SMM operation is
! indicated separately. If only one term order element is specified, it is
! used for {\em all} Field pairs.
! The default is {\tt (/ESMF\_TERMORDER\_FREE/)}, allowing maximum
! flexibility in the order of terms for optimum performance.
! \item [{[checkflag]}]
! If set to {\tt .TRUE.} the input FieldBundle pair will be checked for
! consistency with the precomputed operation provided by {\tt routehandle}.
! If set to {\tt .FALSE.} {\em (default)} only a very basic input check
! will be performed, leaving many inconsistencies undetected. Set
! {\tt checkflag} to {\tt .FALSE.} to achieve highest performance.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! local internal variables
        logical :: src_bundle
        logical :: dst_bundle
        integer :: fcount, i

        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments, deal with optional FieldBundle args
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

        src_bundle = .true.
        if (present(srcFieldBundle)) then
          srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        else
          src_bundle = .false.
        endif

        dst_bundle = .true.
        if (present(dstFieldBundle)) then
          dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        else
          dst_bundle = .false.
        endif

        ! perform FieldBundle SMM
        if(src_bundle .and. dst_bundle) then
          call ESMF_ArrayBundleSMM(srcab, dstab, routehandle, &
            zeroregion=zeroregion, zeroregionflag=zeroregionflag, &
            termorderflag=termorderflag, checkflag=checkflag, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        else if(src_bundle .and. .not. dst_bundle) then
          call ESMF_ArrayBundleSMM(srcArrayBundle=srcab, &
            routehandle=routehandle, &
            zeroregion=zeroregion, zeroregionflag=zeroregionflag, &
            termorderflag=termorderflag, checkflag=checkflag, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        else if(.not. src_bundle .and. dst_bundle) then
          call ESMF_ArrayBundleSMM(dstArrayBundle=dstab, &
            routehandle=routehandle, &
            zeroregion=zeroregion, zeroregionflag=zeroregionflag, &
            termorderflag=termorderflag, checkflag=checkflag, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        else if(.not. src_bundle .and. .not. dst_bundle) then
          call ESMF_ArrayBundleSMM(routehandle=routehandle, &
            zeroregion=zeroregion, zeroregionflag=zeroregionflag, &
            termorderflag=termorderflag, checkflag=checkflag, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif

        ! garbage collection
        if (present(srcFieldBundle)) then
          call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        if (present(dstFieldBundle)) then
          call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMM

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMRelease()"
!BOP
! !IROUTINE: ESMF_FieldBundleSMMRelease - Release resources associated with a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  subroutine ESMF_FieldBundleSMMRelease(routehandle, keywordEnforcer, &
    noGarbage, rc)
!
! !ARGUMENTS:
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: noGarbage
        integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.0.0] Added argument {\tt noGarbage}.
! The argument provides a mechanism to override the default garbage collection
! mechanism when destroying an ESMF object.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Release resources associated with a FieldBundle sparse matrix multiplication. After this call
! {\tt routehandle} becomes invalid.
!
! \begin{description}
! \item [routehandle]
! Handle to the precomputed Route.
! \item[{[noGarbage]}]
! If set to {\tt .TRUE.} the object will be fully destroyed and removed
! from the ESMF garbage collection system. Note however that under this
! condition ESMF cannot protect against accessing the destroyed object
! through dangling aliases -- a situation which may lead to hard to debug
! application crashes.
!
! It is generally recommended to leave the {\tt noGarbage} argument
! set to {\tt .FALSE.} (the default), and to take advantage of the ESMF
! garbage collection system which will prevent problems with dangling
! aliases or incorrect sequences of destroy calls. However this level of
! support requires that a small remnant of the object is kept in memory
! past the destroy call. This can lead to an unexpected increase in memory
! consumption over the course of execution in applications that use
! temporary ESMF objects. For situations where the repeated creation and
! destruction of temporary objects leads to memory issues, it is
! recommended to call with {\tt noGarbage} set to {\tt .TRUE.}, fully
! removing the entire temporary object from memory.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
        integer :: localrc ! local return code

        ! initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! Check init status of arguments
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_RouteHandleGetInit, routehandle, rc)

        ! Call into the RouteHandle code
        call ESMF_RouteHandleRelease(routehandle, noGarbage=noGarbage, &
          rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMRelease

!----------------------------------------------------------------------------
!BOP
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute a FieldBundle sparse matrix multiplication with local factors
!
! !INTERFACE:
! ! Private name; call using ESMF_FieldBundleSMMStore()
! subroutine ESMF_FieldBundleSMMStore<type><kind>(srcFieldBundle, &
! dstFieldBundle, routehandle, factorList, factorIndexList, &
! keywordEnforcer, ignoreUnmatchedIndicesFlag, srcTermProcessing, rc)
!
! !ARGUMENTS:
! type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
! type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
! type(ESMF_RouteHandle), intent(inout) :: routehandle
! <type>(ESMF_KIND_<kind>), intent(in) :: factorList(:)
! integer, intent(in), :: factorIndexList(:,:)
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
! integer, intent(inout), optional :: srcTermProcessing(:)
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[7.1.0r] Added argument {\tt srcTermProcessing}.
! The new argument gives the user access to the tuning parameter
! affecting the sparse matrix execution and bit-wise
! reproducibility.
! \item[8.1.0] Added argument {\tt ignoreUnmatchedIndicesFlag} to support cases
! where the sparse matrix includes terms with source or destination sequence
! indices not present in the source or destination field.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
!
! \begin{sloppypar}
! Store a FieldBundle sparse matrix multiplication operation from {\tt srcFieldBundle}
! to {\tt dstFieldBundle}. PETs that specify non-zero matrix coefficients must use
! the <type><kind> overloaded interface and provide the {\tt factorList} and
! {\tt factorIndexList} arguments. Providing {\tt factorList} and
! {\tt factorIndexList} arguments with
! {\tt size(factorList) = (/0/)} and
! {\tt size(factorIndexList) = (/2,0/)} or {\tt (/4,0/)} indicates that a
! PET does not provide matrix elements. Alternatively, PETs that do not
! provide matrix elements may also call into the overloaded interface
! {\em without} {\tt factorList} and {\tt factorIndexList} arguments.
! \end{sloppypar}
!
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} are interpreted as sequentialized
! vectors. The
! sequence is defined by the order of DistGrid dimensions and the order of
! tiles within the DistGrid or by user-supplied arbitrary sequence indices. See
! section \ref{Array:SparseMatMul} for details on the definition of {\em sequence indices}.
! SMM corresponds to an identity mapping of the source FieldBundle vector to
! the destination FieldBundle vector.
!
! Source and destination Fields may be of different <type><kind>. Further source
! and destination Fields may differ in shape, however, the number of elements
! must match.
!
! It is erroneous to specify the identical FieldBundle object for srcFieldBundle
! and dstFieldBundle arguments.
!
! The routine returns an {\tt ESMF\_RouteHandle} that can be used to call
! {\tt ESMF\_FieldBundleSMM()} on any pair of FieldBundles that matches
! {\tt srcFieldBundle} and {\tt dstFieldBundle} in {\em type}, {\em kind},
! and memory layout of the {\em gridded} dimensions. However, the size,
! number, and index order of {\em ungridded} dimensions may be different.
! See section \ref{RH:Reusability} for a more detailed discussion of
! RouteHandle reusability.
!
! This method is overloaded for:\newline
! {\tt ESMF\_TYPEKIND\_I4}, {\tt ESMF\_TYPEKIND\_I8},\newline
! {\tt ESMF\_TYPEKIND\_R4}, {\tt ESMF\_TYPEKIND\_R8}.
! \newline
!
! This call is collective across the current VM.
!
! For examples and associated documentation regarding this method see Section
! \ref{sec:fieldbundle:usage:smm_1dptr}.
!
! The arguments are:
! \begin{description}
! \item [srcFieldBundle]
! {\tt ESMF\_FieldBundle} with source data.
! \item [dstFieldBundle]
! {\tt ESMF\_FieldBundle} with destination data. The data in this
! FieldBundle may be destroyed by this call.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [factorList]
! List of non-zero coefficients.
! \item [factorIndexList]
! Pairs of sequence indices for the factors stored in {\tt factorList}.
!
! \begin{sloppypar}
! The second dimension of {\tt factorIndexList} steps through the list of
! pairs, i.e. {\tt size(factorIndexList,2) == size(factorList)}. The first
! dimension of {\tt factorIndexList} is either of size 2 or size 4.
! \end{sloppypar}
!
! In the {\em size 2 format} {\tt factorIndexList(1,:)} specifies the
! sequence index of the source element in the {\tt srcFieldBundle} while
! {\tt factorIndexList(2,:)} specifies the sequence index of the
! destination element in {\tt dstFieldBundle}. For this format to be a valid
! option source and destination FieldBundles must have matching number of
! tensor elements (the product of the sizes of all Field tensor dimensions).
! Under this condition an identity matrix can be applied within the space of
! tensor elements for each sparse matrix factor.
!
! The {\em size 4 format} is more general and does not require a matching
! tensor element count. Here the
!
! {\tt factorIndexList(1,:)} specifies the
! sequence index while {\tt factorIndexList(2,:)} specifies the tensor
! sequence index of the source element in the {\tt srcFieldBundle}. Further
! {\tt factorIndexList(3,:)} specifies the sequence index and
! {\tt factorIndexList(4,:)} specifies the tensor sequence index of the
! destination element in the {\tt dstFieldBundle}.
!
! See section \ref{Array:SparseMatMul} for details on the definition of
! {\em sequence indices} and {\em tensor sequence indices}.
!
! \item [{[ignoreUnmatchedIndicesFlag]}]
! If set to {.false.}, the {\em default}, source and destination side must
! cover all of the squence indices defined in the sparse matrix. An error
! will be returned if a sequence index in the sparse matrix does not match
! on either the source or destination side.
! If set to {.true.}, mismatching sequence indices are silently ignored.
! The size of this array argument must either be 1 or equal the number of
! Fieldss in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the handling of unmatched indices is specified for each
! Field pair separately. If only one element is specified, it is
! used for {\em all} Field pairs.
!
! \item [{[srcTermProcessing]}]
! Source term summing options for route handle creation. See
! {\tt ESMF\_FieldRegridStore} documentation for a full parameter description.
! Two forms may be provided. If a single element list is provided, this
! integer value is applied across all bundle members. Otherwise, the list must
! contain as many elements as there are bundle members. For the special case
! of accessing the auto-tuned parameter (providing a negative integer value),
! the list length must equal the bundle member count.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!----------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMStoreI4"
!BOPI
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleSMMStore()
    subroutine ESMF_FieldBundleSMMStoreI4(srcFieldBundle, dstFieldBundle, &
        routehandle, factorList, factorIndexList, keywordEnforcer, &
        ignoreUnmatchedIndicesFlag, srcTermProcessing, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        integer(ESMF_KIND_I4), intent(in) :: factorList(:)
        integer, intent(in) :: factorIndexList(:,:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(inout), optional :: srcTermProcessing(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleSMMStore(srcab, dstab, routehandle, factorList, &
            factorIndexList, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcTermProcessing=srcTermProcessing, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMStoreI4
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMStoreI8"
!BOPI
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleSMMStore()
    subroutine ESMF_FieldBundleSMMStoreI8(srcFieldBundle, dstFieldBundle, &
      routehandle, factorList, factorIndexList, keywordEnforcer, &
      ignoreUnmatchedIndicesFlag, srcTermProcessing, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        integer(ESMF_KIND_I8), intent(in) :: factorList(:)
        integer, intent(in) :: factorIndexList(:,:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(inout), optional :: srcTermProcessing(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleSMMStore(srcab, dstab, routehandle, factorList, &
            factorIndexList, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcTermProcessing=srcTermProcessing, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMStoreI8
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMStoreR4"
!BOPI
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleSMMStore()
    subroutine ESMF_FieldBundleSMMStoreR4(srcFieldBundle, dstFieldBundle, &
      routehandle, factorList, factorIndexList, keywordEnforcer, &
      ignoreUnmatchedIndicesFlag, srcTermProcessing, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        real(ESMF_KIND_R4), intent(in) :: factorList(:)
        integer, intent(in) :: factorIndexList(:,:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(inout), optional :: srcTermProcessing(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleSMMStore(srcab, dstab, routehandle, factorList, &
            factorIndexList, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcTermProcessing=srcTermProcessing, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMStoreR4
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMStoreR8"
!BOPI
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleSMMStore()
    subroutine ESMF_FieldBundleSMMStoreR8(srcFieldBundle, dstFieldBundle, &
      routehandle, factorList, factorIndexList, keywordEnforcer, &
      ignoreUnmatchedIndicesFlag, srcTermProcessing, rc)

        ! input arguments
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
        real(ESMF_KIND_R8), intent(in) :: factorList(:)
        integer, intent(in) :: factorIndexList(:,:)
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(inout), optional :: srcTermProcessing(:)
        integer, intent(out), optional :: rc

!EOPI
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleSMMStore(srcab, dstab, routehandle, factorList, &
            factorIndexList, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcTermProcessing=srcTermProcessing, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMStoreR8

!----------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMStoreNF"
!BOP
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute a FieldBundle sparse matrix multiplication
!
! !INTERFACE:
  ! Private name; call using ESMF_FieldBundleSMMStore()
    subroutine ESMF_FieldBundleSMMStoreNF(srcFieldBundle, dstFieldBundle, &
        routehandle, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
        srcTermProcessing, rc)

! !ARGUMENTS:
        type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
        type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
        type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
        logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
        integer, intent(inout), optional :: srcTermProcessing(:)
        integer, intent(out), optional :: rc

! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[7.1.0r] Added argument {\tt srcTermProcessing}.
! The new argument gives the user access to the tuning parameter
! affecting the sparse matrix execution and bit-wise
! reproducibility.
! \item[8.1.0] Added argument {\tt ignoreUnmatchedIndicesFlag} to support cases
! where the sparse matrix includes terms with source or destination sequence
! indices not present in the source or destination field.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
!
! \begin{sloppypar}
! Store a FieldBundle sparse matrix multiplication operation from {\tt srcFieldBundle}
! to {\tt dstFieldBundle}. PETs that specify non-zero matrix coefficients must use
! the <type><kind> overloaded interface and provide the {\tt factorList} and
! {\tt factorIndexList} arguments. Providing {\tt factorList} and
! {\tt factorIndexList} arguments with {\tt size(factorList) = (/0/)} and
! {\tt size(factorIndexList) = (/2,0/)} or {\tt (/4,0/)} indicates that a
! PET does not provide matrix elements. Alternatively, PETs that do not
! provide matrix elements may also call into the overloaded interface
! {\em without} {\tt factorList} and {\tt factorIndexList} arguments.
! \end{sloppypar}
!
! Both {\tt srcFieldBundle} and {\tt dstFieldBundle} are interpreted as sequentialized
! vectors. The
! sequence is defined by the order of DistGrid dimensions and the order of
! tiles within the DistGrid or by user-supplied arbitrary sequence indices. See
! section \ref{Array:SparseMatMul} for details on the definition of {\em sequence indices}.
! SMM corresponds to an identity mapping of the source FieldBundle vector to
! the destination FieldBundle vector.
!
! Source and destination Fields may be of different <type><kind>. Further source
! and destination Fields may differ in shape, however, the number of elements
! must match.
!
! It is erroneous to specify the identical FieldBundle object for srcFieldBundle and dstFieldBundle
! arguments.
!
! The routine returns an {\tt ESMF\_RouteHandle} that can be used to call
! {\tt ESMF\_FieldBundleSMM()} on any pair of FieldBundles that matches
! {\tt srcFieldBundle} and {\tt dstFieldBundle} in {\em type}, {\em kind},
! and memory layout of the {\em gridded} dimensions. However, the size,
! number, and index order of {\em ungridded} dimensions may be different.
! See section \ref{RH:Reusability} for a more detailed discussion of
! RouteHandle reusability.
!
! \begin{sloppypar}
! This method is overloaded for
! {\tt ESMF\_TYPEKIND\_I4}, {\tt ESMF\_TYPEKIND\_I8},
! {\tt ESMF\_TYPEKIND\_R4}, {\tt ESMF\_TYPEKIND\_R8}.
! \end{sloppypar}
!
! This call is collective across the current VM.
!
! For examples and associated documentation regarding this method see Section
! \ref{sec:fieldbundle:usage:smm_1dptr}.
!
! The arguments are:
! \begin{description}
! \item [srcFieldBundle]
! {\tt ESMF\_FieldBundle} with source data.
! \item [dstFieldBundle]
! {\tt ESMF\_FieldBundle} with destination data. The data in this
! FieldBundle may be destroyed by this call.
! \item [routehandle]
! Handle to the precomputed Route.
! \item [{[ignoreUnmatchedIndicesFlag]}]
! If set to {.false.}, the {\em default}, source and destination side must
! cover all of the squence indices defined in the sparse matrix. An error
! will be returned if a sequence index in the sparse matrix does not match
! on either the source or destination side.
! If set to {.true.}, mismatching sequence indices are silently ignored.
! The size of this array argument must either be 1 or equal the number of
! Fieldss in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the handling of unmatched indices is specified for each
! Field pair separately. If only one element is specified, it is
! used for {\em all} Field pairs.
! \item [{[srcTermProcessing]}]
! Source term summing options for route handle creation. See
! {\tt ESMF\_FieldRegridStore} documentation for a full parameter description.
! Two forms may be provided. If a single element list is provided, this
! integer value is applied across all bundle members. Otherwise, the list must
! contain as many elements as there are bundle members. For the special case
! of accessing the auto-tuned parameter (providing a negative integer value),
! the list length must equal the bundle member count.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
        ! local variables as temporary input/output arguments

        ! internal local variables
        integer :: localrc, sfcount, dfcount, i
        type(ESMF_ArrayBundle) :: srcab, dstab
        type(ESMF_Array), allocatable :: srca(:), dsta(:)

        ! Initialize return code; assume routine not implemented
        localrc = ESMF_RC_NOT_IMPL
        if(present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check variables
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, srcFieldBundle, rc)
        ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, dstFieldBundle, rc)

        ! loop over source and destination fields.
        ! verify src and dst FieldBundles can communicate
        ! field_count match
        call ESMF_FieldBundleGet(srcFieldBundle, fieldCount=sfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleGet(dstFieldBundle, fieldCount=dfcount, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if(sfcount /= dfcount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="src and dst FieldBundle must have same number of fields", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif

        ! TODO:
        ! internal grids match
        !if(ESMF_GridMatch(srcFieldBundle%btypep%grid, dstFieldBundle%btypep%grid) then
        ! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
        ! "src and dst FieldBundle must have matching grid", &
        ! ESMF_CONTEXT, rcToReturn=rc)
        ! return
        !endif

        srcab = ESMF_FieldBundleToAB(srcFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        dstab = ESMF_FieldBundleToAB(dstFieldBundle, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_ArrayBundleSMMStore(srcab, dstab, routehandle, &
            ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
            srcTermProcessing=srcTermProcessing, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! garbage collection
        call ESMF_ArrayBundleDestroy(srcab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_ArrayBundleDestroy(dstab, noGarbage=.true., rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

        ! return successfully
        if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMStoreNF
! ----------------------------------------------------------------------------

! ----------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSMMStoreFromFile"

!BOP
! !IROUTINE: ESMF_FieldBundleSMMStore - Precompute field bundle sparse matrix multiplication using factors read from file
!
! !INTERFACE:
! ! Private name; call using ESMF_FieldBundleSMMStore()
    subroutine ESMF_FieldBundleSMMStoreFromFile(srcFieldBundle, dstFieldBundle, &
      filename, routehandle, keywordEnforcer, ignoreUnmatchedIndicesFlag, &
      srcTermProcessing, rc)

! ! ARGUMENTS:
      type(ESMF_FieldBundle), intent(in) :: srcFieldBundle
      type(ESMF_FieldBundle), intent(inout) :: dstFieldBundle
      character(len=*), intent(in) :: filename
      type(ESMF_RouteHandle), intent(inout) :: routehandle
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      logical, intent(in), optional :: ignoreUnmatchedIndicesFlag(:)
      integer, intent(inout), optional :: srcTermProcessing(:)
      integer, intent(out), optional :: rc

!
! !DESCRIPTION:
!
! Compute an {\tt ESMF\_RouteHandle} using factors read from file.
!
! The arguments are:
!
! \begin{description}
!
! \item [srcFieldBundle]
! {\tt ESMF\_FieldBundle} with source data.
!
! \item [dstFieldBundle]
! {\tt ESMF\_FieldBundle} with destination data. The data in this field
! bundle may be destroyed by this call.
!
! \item [filename]
! Path to the file containing weights for creating an {\tt ESMF\_RouteHandle}.
! See ~(\ref{sec:weightfileformat}) for a description of the SCRIP weight
! file format. Only "row", "col", and "S" variables are required. They
! must be one-dimensionsal with dimension "n\_s".
!
! \item [routehandle]
! Handle to the {\tt ESMF\_RouteHandle}.
!
! \item [{[ignoreUnmatchedIndicesFlag]}]
! If set to {.false.}, the {\em default}, source and destination side must
! cover all of the squence indices defined in the sparse matrix. An error
! will be returned if a sequence index in the sparse matrix does not match
! on either the source or destination side.
! If set to {.true.}, mismatching sequence indices are silently ignored.
! The size of this array argument must either be 1 or equal the number of
! Fieldss in the {\tt srcFieldBundle} and {\tt dstFieldBundle} arguments. In
! the latter case, the handling of unmatched indices is specified for each
! Field pair separately. If only one element is specified, it is
! used for {\em all} Field pairs.
!
! \item [{[srcTermProcessing]}]
! Source term summing options for route handle creation. See
! {\tt ESMF\_FieldRegridStore} documentation for a full parameter description.
! Two forms may be provided. If a single element list is provided, this
! integer value is applied across all bundle members. Otherwise, the list must
! contain as many elements as there are bundle members. For the special case
! of accessing the auto-tuned parameter (providing a negative integer value),
! the list length must equal the bundle member count.
!
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
!
! \end{description}
!
!EOP
! ----------------------------------------------------------------------------------

      ! LOCAL VARIABLES:
      real(ESMF_KIND_R8), dimension(:), allocatable :: factorList
      integer, dimension(:, :), allocatable :: factorIndexList
      integer :: localrc

      ! Initialize return code; assume routine not implemented
      localrc = ESMF_RC_NOT_IMPL
      if (present(rc)) rc = ESMF_RC_NOT_IMPL

      ! Fill the factorList and factorIndexList.
      call ESMF_FactorRead(filename, &
                           factorList, &
                           factorIndexList, &
                           rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! Generate routeHandle from factorList and factorIndexList
      call ESMF_FieldBundleSMMStore(srcFieldBundle, dstFieldBundle, routehandle, &
        factorList, factorIndexList, &
        ignoreUnmatchedIndicesFlag=ignoreUnmatchedIndicesFlag, &
        srcTermProcessing=srcTermProcessing, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      deallocate(factorList)
      deallocate(factorIndexList)

      if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSMMStoreFromFile

! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleValidate()"
!BOP
! !IROUTINE: ESMF_FieldBundleValidate - Validate fieldbundle internals

! !INTERFACE:
  subroutine ESMF_FieldBundleValidate(fieldbundle, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
    integer, intent(out), optional :: rc
!
!
! !DESCRIPTION:
! Validates that the {\tt fieldbundle} is internally consistent.
! The method returns an error code if problems are found.
!
! The arguments are:
! \begin{description}
! \item[fieldbundle]
! Specified {\tt ESMF\_FieldBundle} object.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! Initialize return code
    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    localrc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    ! Call into the C++ interface layer
    !todo: call c_ESMC_FieldBundleValidate(fieldbundle, localrc)
    localrc = ESMF_SUCCESS ! remove when todo is done.

    ! Use LogErr to handle return code
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleValidate
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleWrite()"
!BOP
! !IROUTINE: ESMF_FieldBundleWrite - Write the Fields into a file
! \label{api:FieldBundleWrite}

! !INTERFACE:
  subroutine ESMF_FieldBundleWrite(fieldbundle, fileName, keywordEnforcer, &
      convention, purpose, singleFile, overwrite, status, timeslice, iofmt, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
    character(*), intent(in) :: fileName
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords for the below
    character(*), intent(in), optional :: convention
    character(*), intent(in), optional :: purpose
    logical, intent(in), optional :: singleFile
    logical , intent(in), optional :: overwrite
    type(ESMF_FileStatus_Flag), intent(in), optional :: status
    integer, intent(in), optional :: timeslice
    type(ESMF_IOFmt_Flag), intent(in), optional :: iofmt
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Write the Fields into a file. For this API to be functional,
! the environment variable {\tt ESMF\_PIO} should be set to either "internal" or "external"
! when the ESMF library is built. Please see the section on
! Data I/O,~\ref{io:dataio}.
!
! When {\tt convention} and {\tt purpose} arguments are specified, NetCDF dimension
! labels and variable attributes are written from each Field in the FieldBundle
! from the corresponding Attribute package. Additionally, Attributes may be
! set on the FieldBundle level under the same Attribute package. This allows
! the specification of global attributes within the file.
! As with individual Fields, the value associated with each name may be either
! a scalar character string, or a scalar or array of type integer, real, or
! double precision.
!
! Limitations:
! \begin{itemize}
! \item For multi-tile Fields, all Fields in the FieldBundle must contain
! the same number of tiles.
! \end{itemize}
!
! The arguments are:
! \begin{description}
! \item[fieldbundle]
! An {\tt ESMF\_FieldBundle} object.
! \item[fileName]
! The name of the output file to which field bundle data is written.
! If the FieldBundle contains multi-tile Arrays, then fileName must contain
! exactly one instance of "*"; this is a placeholder that will be replaced
! by the tile number, with each tile being written to a separate file. (For
! example, for a fileName of "myfile*.nc", tile 1 will be written to
! "myfile1.nc", tile 2 to "myfile2.nc", etc.)
! (This handling of the fileName for multi-tile I/O is subject to change.)
! \item[{[convention]}]
! Specifies an Attribute package associated with the FieldBundle, and the
! contained Fields, used to create NetCDF dimension labels and attributes
! in the file. When this argument is present, the {\tt purpose}
! argument must also be present. Use this argument only with a NetCDF
! I/O format. If binary format is used, ESMF will return an error code.
! \item[{[purpose]}]
! Specifies an Attribute package associated with the FieldBundle, and the
! contained Fields, used to create NetCDF dimension labels and attributes
! in the file. When this argument is present, the {\tt convention}
! argument must also be present. Use this argument only with a NetCDF
! I/O format. If binary format is used, ESMF will return an error code.
! \item[{[singleFile]}]
! A logical flag, the default is .true., i.e., all fields in the bundle
! are written in one single file. If .false., each field will be written
! in separate files; these files are numbered with the name based on the
! argument "file". That is, a set of files are named: [file\_name]001,
! [file\_name]002, [file\_name]003,...
! \item[{[overwrite]}]
! \begin{sloppypar}
! A logical flag, the default is .false., i.e., existing field data may
! {\em not} be overwritten. If .true., only the
! data corresponding to the fields name will be
! be overwritten. If the {\tt timeslice} option is given, only data for
! the given timeslice will be overwritten.
! Note that it is always an error to attempt to overwrite a NetCDF
! variable with data which has a different shape.
! \end{sloppypar}
! \item[{[status]}]
! \begin{sloppypar}
! The file status. Please see Section~\ref{const:filestatusflag} for
! the list of options. If not present, defaults to
! {\tt ESMF\_FILESTATUS\_UNKNOWN}.
! \end{sloppypar}
! \item[{[timeslice]}]
! \begin{sloppypar}
! Some I/O formats (e.g. NetCDF) support the output of data in form of
! time slices. The {\tt timeslice} argument provides access to this
! capability. {\tt timeslice} must be positive. The behavior of this
! option may depend on the setting of the {\tt overwrite} flag:
! \begin{description}
! \item[{\tt overwrite = .false.}:]\ If the timeslice value is
! less than the maximum time already in the file, the write will fail.
! \item[{\tt overwrite = .true.}:]\ Any positive timeslice value is valid.
! \end{description}
! By default, i.e. by omitting the {\tt timeslice} argument, no
! provisions for time slicing are made in the output file,
! however, if the file already contains a time axis for the variable,
! a timeslice one greater than the maximum will be written.
! \end{sloppypar}
! \item[{[iofmt]}]
! \begin{sloppypar}
! The I/O format. Please see Section~\ref{opt:iofmtflag} for the list
! of options. If not present, defaults to {\tt ESMF\_IOFMT\_NETCDF}.
! \end{sloppypar}
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    character(len=ESMF_MAXSTR) :: name
    integer :: fieldCount
    integer :: i
    type(ESMF_Field), allocatable :: fieldList(:)
    logical :: singlef
    character(len=3) :: cnum
    character(len=len (fileName) + 3) :: filename_num ! len (file) + len (cnum)
    type(ESMF_Array) :: array
    type(ESMF_FieldType), pointer :: fp
    type(ESMF_Grid) :: grid
    logical :: opt_overwriteflag ! helper variable
    type(ESMF_FileStatus_Flag) :: opt_status ! helper variable
    type(ESMF_IOFmt_Flag) :: opt_iofmt
    type(ESMF_IO) :: io ! The I/O object
    logical :: errorFound ! True if err. cond.
    integer :: file_ext_p

#ifdef ESMF_PIO
    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    ! Check options
    singlef = .true.
    if (present(singleFile)) singlef = singleFile

    opt_overwriteflag = .false.
    if (present(overwrite)) opt_overwriteflag = overwrite

    opt_status = ESMF_FILESTATUS_UNKNOWN
    if (present(status)) opt_status = status

    ! Set iofmt based on file name extension (if present)
    if (present (iofmt)) then
      opt_iofmt = iofmt
    else
      opt_iofmt = ESMF_IOFMT_NETCDF
    end if

    if (present (convention) .neqv. present (purpose)) then
      if (ESMF_LogFoundError (ESMF_RC_ARG_WRONG, &
          msg='Both convention and purpose must be specified', &
          ESMF_CONTEXT, rcToReturn=rc)) return
    end if

    call ESMF_FieldBundleGet(fieldbundle, fieldCount=fieldCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    allocate (fieldList(fieldCount))
    call ESMF_FieldBundleGet(fieldbundle, fieldList=fieldList, &
!TODO: gjt thinks this should be doine in ADDORDER below. However, currently
!TODO: this causes an ESMF_FieldBundleIOUTest failure. Needs to be fixed.
! itemorderflag=ESMF_ITEMORDER_ADDORDER, &
      rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create an I/O object



    io = ESMF_IOCreate(rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return




    ! From here on out, we need to clean up so no returning on error
    if (singlef) then
      ! Get and read the fields in the Bundle
      do i=1,fieldCount
        call ESMF_FieldGet(fieldList(i), array=array, name=name, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        fp => fieldList(i)%ftypep
        if (present (convention)) then
          call ESMF_FieldGet (fieldList(i), grid=grid, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return
        end if

        call c_esmc_fieldioaddarray (io, fp%base, array, grid, name, &
            fieldbundle%this%base, convention, purpose, &
            localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
      enddo




      call ESMF_IOWrite(io, trim(fileName), overwrite=opt_overwriteflag, &
          status=opt_status, timeslice=timeslice, iofmt=opt_iofmt, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return




    else
      do i=1,fieldCount
        ! Clear the IO object (only need to do this for i > 1)
        if (i .gt. 1) call ESMF_IOClear(io)
        write(cnum,"(i3.3)") i
        filename_num = ESMF_StringConcat(trim (fileName) , cnum)
        call ESMF_FieldGet(fieldList(i), array=array, name=name, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        fp => fieldList(i)%ftypep
        if (present (convention)) then
          call ESMF_FieldGet (fieldList(i), grid=grid, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return
        end if

        call c_esmc_fieldioaddarray (io, fp%base, array, grid, name, &
            fieldbundle%this%base, convention, purpose, &
            localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        call ESMF_IOWrite(io, trim(filename_num), &
             overwrite=opt_overwriteflag, status=opt_status, &
             timeslice=timeslice, iofmt=opt_iofmt, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
      enddo
    endif

    ! Set rc here in case we had an error but destroy succeeds
    if (present(rc)) rc = localrc

    call ESMF_IODestroy(io, rc=localrc)
    ! Log error but donot reset rc
    errorFound = ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=localrc)

    ! Last chance to return an error code (IODestroy failed)
    if (present(rc)) then
      if (rc == ESMF_SUCCESS) rc = localrc
    end if

#else
    ! Return indicating PIO not present
    if (present(rc)) rc = ESMF_RC_LIB_NOT_PRESENT
#endif

  end subroutine ESMF_FieldBundleWrite
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSerialize"

!BOPI
! !IROUTINE: ESMF_FieldBundleSerialize - Serialize fieldbundle info into a byte stream
!
! !INTERFACE:
    subroutine ESMF_FieldBundleSerialize(fieldbundle, buffer, length, offset, &
                                          attreconflag, inquireflag, rc)
!
! !ARGUMENTS:
      type(ESMF_FieldBundle), intent(inout) :: fieldbundle
      character, pointer, dimension(:) :: buffer
      integer, intent(inout) :: length
      integer, intent(inout) :: offset
      type(ESMF_AttReconcileFlag), intent(in), optional :: attreconflag
      type(ESMF_InquireFlag), intent(in), optional :: inquireflag
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Takes an {\tt ESMF\_FieldBundle} object and adds all the information needed
! to save the information to a file or recreate the object based on this
! information. Expected to be used by {\tt ESMF\_StateReconcile()} and
! by {\tt ESMF\_FieldBundleWrite()} and {\tt ESMF\_FieldBundleRead()}.
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! {\tt ESMF\_FieldBundle} object to be serialized.
! \item [buffer]
! Data buffer which will hold the serialized information.
! \item [length]
! Current length of buffer, in bytes. If the serialization
! process needs more space it will allocate it and update
! this length.
! \item [offset]
! Current write offset in the current buffer. This will be
! updated by this routine and return pointing to the next
! available byte in the buffer.
! \item[{[attreconflag]}]
! Flag to tell if Attribute serialization is to be done
! \item[{[inquireflag]}]
! Flag to tell if serialization is to be done (ESMF_NOINQUIRE)
! or if this is simply a size inquiry (ESMF_INQUIREONLY)
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI

      integer :: localrc ! Error status
      integer :: i, fieldCount, packedVal, encodeNameLength
      type(ESMF_FieldBundleType), pointer :: bp ! fieldbundle type
      type(ESMF_AttReconcileFlag) :: lattreconflag
      type(ESMF_InquireFlag) :: linquireflag
      type(ESMF_Field), pointer :: l_fieldList(:)

      logical, parameter :: debug = .false.
      character(len=ESMF_MAXSTR) :: logmsg

      ! Initialize return code; assume routine not implemented
      if (present(rc)) rc = ESMF_RC_NOT_IMPL
      localrc = ESMF_RC_NOT_IMPL

      ! check inputs
      ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)

      nullify(l_fieldList)

      ! deal with optional attreconflag and inquireflag
      if (present(attreconflag)) then
        lattreconflag = attreconflag
      else
        lattreconflag = ESMF_ATTRECONCILE_OFF
      endif

      if (present (inquireflag)) then
        linquireflag = inquireflag
      else
        linquireflag = ESMF_NOINQUIRE
      end if

      ! shortcut to internals
      bp => fieldbundle%this

      call ESMF_ContainerGet(bp%container, itemCount=fieldCount, &
        itemList=l_fieldList, itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      if (debug) then
        call ESMF_LogWrite(&
          ESMF_StringConcat(ESMF_METHOD , ": Calling ESMF_BaseSerialize"), &
          ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
        write(logmsg, *) ESMF_StringConcat(ESMF_METHOD , ": offset (before)= "), offset
        call ESMF_LogWrite(TRIM(logmsg), ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
      end if

      call ESMF_BaseSerialize (bp%base, buffer, offset, &
          lattreconflag, linquireflag, &
          rc=localrc)
      if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

      if (debug) then
        write(logmsg, *) ESMF_StringConcat(ESMF_METHOD , ": offset (after)= "), offset
        call ESMF_LogWrite(TRIM(logmsg), ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
         logmsg = ESMF_StringConcat(ESMF_METHOD , ": Exited ESMF_BaseSerialize")
         call ESMF_LogWrite(logmsg, ESMF_LOGMSG_DEBUG, rc=localrc)
         if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
       end if

      call c_ESMC_FieldBundleSerialize(bp%status, fieldCount, &
                                 buffer, length, offset, linquireflag, localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      packedVal = 0
      if(bp%is_packed) packedVal = 1

      call c_ESMC_FieldBundleSerialize(bp%status, packedVal, &
                                 buffer, length, offset, linquireflag, localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      if(bp%status == ESMF_FBSTATUS_GRIDSET) then
        ! NOTE: A small optimization for StateReconcile is possible by removing
        ! this extra Geom serialization/deserialization. According to Fei,
        ! this just references the first Field.
        call ESMF_GeomSerialize(bp%geom, buffer, length, offset, &
                                 lattreconflag, linquireflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      do i = 1, fieldCount
          call ESMF_FieldSerialize(l_fieldList(i), buffer, length, offset, &
                                  attreconflag=lattreconflag, &
                                  inquireflag=linquireflag, rc=localrc)
          if (ESMF_LogFoundError(localrc, &
             ESMF_ERR_PASSTHRU, &
             ESMF_CONTEXT, rcToReturn=rc)) return
      enddo

      deallocate(l_fieldList)

      if(bp%is_packed) then
        encodeNameLength = len(bp%encodeName)
        call c_ESMC_FieldBundleSerializeEncodeName(encodeNameLength, bp%encodeName, &
                                   buffer, length, offset, linquireflag, localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_FieldBundleSerialize

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleDeserialize"

!BOPI
! !IROUTINE: ESMF_FieldBundleDeserialize - Deserialize a byte stream into a FieldBundle
!
! !INTERFACE:
    function ESMF_FieldBundleDeserialize(buffer, offset, &
                                          attreconflag, rc)
!
! !RETURN VALUE:
      type(ESMF_FieldBundle) :: ESMF_FieldBundleDeserialize
!
! !ARGUMENTS:
      character, pointer, dimension(:) :: buffer
      integer, intent(inout) :: offset
      type(ESMF_AttReconcileFlag), optional :: attreconflag
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Takes a byte-stream buffer and reads the information needed to
! recreate a FieldBundle object. Recursively calls the deserialize routines
! needed to recreate the subobjects.
! Expected to be used by {\tt ESMF\_StateReconcile()} and
! by {\tt ESMF\_FieldBundleWrite()} and {\tt ESMF\_FieldBundleRead()}.
!
! The arguments are:
! \begin{description}
! \item [buffer]
! Data buffer which holds the serialized information.
! \item [offset]
! Current read offset in the current buffer. This will be
! updated by this routine and return pointing to the next
! unread byte in the buffer.
! \item[{[attreconflag]}]
! Flag to tell if Attribute deserialization is to be done
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI

      integer :: localrc, status ! Error status, allocation status
      integer :: i, fieldCount, packedVal, encodeNameLength
      type(ESMF_FieldBundleType), pointer :: bp ! fieldbundle type
      type(ESMF_AttReconcileFlag) :: lattreconflag
      type(ESMF_Grid) :: grid
      type(ESMF_LocStream) :: locstream
      type(ESMF_GeomType_Flag) :: geomtype
      type(ESMF_Field), pointer :: flist(:)

      logical, parameter :: debug = .false.
      character(len=ESMF_MAXSTR) :: logmsg

      ! Initialize return code; assume routine not implemented
      if (present(rc)) rc = ESMF_RC_NOT_IMPL
      localrc = ESMF_RC_NOT_IMPL
      status = ESMF_RC_NOT_IMPL

      nullify(flist)

      ! deal with optional attreconflag
      if (present(attreconflag)) then
        lattreconflag = attreconflag
      else
        lattreconflag = ESMF_ATTRECONCILE_OFF
      endif

      ! in case of error, make sure this is invalid.
      nullify(bp)
      nullify(ESMF_FieldBundleDeserialize%this)

      ! shortcut to internals
      allocate(bp, stat=status)
      if (ESMF_LogFoundAllocError(status, &
        msg="space for new FieldBundle object", &
        ESMF_CONTEXT, rcToReturn=rc)) return

      if (debug) then
        call ESMF_LogWrite(&
           ESMF_StringConcat(ESMF_METHOD , ": Calling ESMF_BaseDeserialize"), &
           ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
        write(logmsg, *) ESMF_StringConcat(ESMF_METHOD , ": buffer length= "), LEN(buffer)
        call ESMF_LogWrite(TRIM(logmsg), ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
        write(logmsg, *) ESMF_StringConcat(ESMF_METHOD , ": offset= "), offset
        call ESMF_LogWrite(TRIM(logmsg), ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
      end if

      ! Deserialize Base
      bp%base = ESMF_BaseDeserialize (buffer, offset=offset, &
          attreconflag=lattreconflag, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      if (debug) then
        logmsg = ESMF_StringConcat(ESMF_METHOD , ": Exited ESMF_BaseDeserialize")
        call ESMF_LogWrite(logmsg, ESMF_LOGMSG_DEBUG, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
      end if

      call ESMF_BaseSetInitCreated(bp%base, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! Deserialize other FieldBundle members

      call c_ESMC_FieldBundleDeserialize(bp%status, fieldCount, &
                                 buffer, offset, localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      call c_ESMC_FieldBundleDeserialize(bp%status, packedVal, &
                                 buffer, offset, localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      bp%is_packed = .false.
      if(packedVal == 1) bp%is_packed = .true.

      if(bp%status == ESMF_FBSTATUS_GRIDSET) then
        bp%geom = ESMF_GeomDeserialize(buffer, offset, &
          attreconflag=attreconflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      ! TODO: decide if these need to be sent before or after
      allocate(flist(fieldCount), stat=localrc)
      if (ESMF_LogFoundAllocError(localrc, &
        msg = " - Field list", &
        ESMF_CONTEXT, rcToReturn=rc)) return

      do i = 1, fieldCount
        flist(i) = ESMF_FieldDeserialize(buffer, offset, &
                                    attreconflag=lattreconflag, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) then
          deallocate(flist)
          return
        endif
      enddo

      bp%container = ESMF_ContainerCreate(rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      bp%is_proxy = .true.

      ESMF_FieldBundleDeserialize%this => bp

      ! Add reference to this object into ESMF garbage collection table
      ! Only call this in those Create() methods that call Construct()
      call c_ESMC_VMAddFObject(ESMF_FieldBundleDeserialize, &
        ESMF_ID_FIELDBUNDLE%objectID)

      ! Set as created
      ESMF_INIT_SET_CREATED(ESMF_FieldBundleDeserialize)

      call ESMF_FieldBundleAddListInternal(ESMF_FieldBundleDeserialize, &
        flist, multiflag=.true., rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      deallocate(flist)

      if(bp%is_packed) then
        call c_ESMC_FieldBundleDeserializeInt(encodeNameLength, &
                                   buffer, offset, localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        allocate(character(encodeNameLength) :: bp%encodeName)
        call c_ESMC_FieldBundleDeSerializeEncodeName(encodeNameLength, bp%encodeName, &
                                   buffer, offset, localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      if (present(rc)) rc = ESMF_SUCCESS

    end function ESMF_FieldBundleDeserialize

! -------------------------- ESMF-internal method -----------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleSetGeom()"
!BOPI
! !IROUTINE: ESMF_FieldBundleSetGeom - Set a Geom in FieldBundle
!
! !INTERFACE:
  subroutine ESMF_FieldBundleSetGeom(fieldbundle, field, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    type(ESMF_Field), intent(in) :: field
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Set a geom in FieldBundle, if the geom is a Grid, attribute
! linking is done inside FieldBundleSetGrid.
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! fieldbundle object.
! \item [field]
! field object.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI

    integer :: localrc
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    type(ESMF_XGrid) :: xgrid
    type(ESMF_Mesh) :: mesh
    type(ESMF_LocStream) :: locstream
    type(ESMF_FieldStatus_Flag) :: fstatus

    localrc = ESMF_RC_NOT_IMPL
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! check input arguments
    ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)
    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field, rc)

    ! Donot do anything if fieldbundle already has a geom
    ! TODO: we will check matchness here in the future between field%geom
    ! and fieldbundle%geom
    if(fieldbundle%this%status == ESMF_FBSTATUS_GRIDSET) then
      if(present(rc)) rc = ESMF_SUCCESS
      return
    endif

    call ESMF_FieldGet(field, status=fstatus, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    if(fstatus == ESMF_FIELDSTATUS_GRIDSET .or. &
       fstatus == ESMF_FIELDSTATUS_COMPLETE) then
      call ESMF_FieldGet(field, geomtype=geomtype, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
      if(geomtype == ESMF_GEOMTYPE_GRID) then
        call ESMF_FieldGet(field, grid=grid, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        ! this call takes care of attribute linking of Grid
        call ESMF_FieldBundleSet(fieldbundle, grid, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      else if(geomtype == ESMF_GEOMTYPE_XGRID) then
        call ESMF_FieldGet(field, xgrid=xgrid, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleSet(fieldbundle, xgrid, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      else if(geomtype == ESMF_GEOMTYPE_MESH) then
        call ESMF_FieldGet(field, mesh=mesh, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleSet(fieldbundle, mesh, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      else if(geomtype == ESMF_GEOMTYPE_LOCSTREAM) then
        call ESMF_FieldGet(field, locstream=locstream, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_FieldBundleSet(fieldbundle, locstream, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif
      fieldbundle%this%status = ESMF_FBSTATUS_GRIDSET
    endif

    if(present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleSetGeom

! -----------------------------------------------------------------------------


! -------------------------- ESMF-internal method -----------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleRemoveGeom()"
!BOPI
! !IROUTINE: ESMF_FieldBundleRemoveGeom - Remove the Geom in FieldBundle
!
! !INTERFACE:
  subroutine ESMF_FieldBundleRemoveGeom(fieldbundle, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(inout) :: fieldbundle
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Remove the geom in FieldBundle, if the geom is a Grid, attribute
! linking is removed as well. Called when the fieldbundle is emptied.
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! fieldbundle object.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI

    integer :: localrc
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid

    localrc = ESMF_RC_NOT_IMPL
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! check input arguments
    ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc)

    ! should never call this method when it is not GRIDSET
    if(fieldbundle%this%status /= ESMF_FBSTATUS_GRIDSET) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
        msg = " - cannot remove a gemobase from a fieldbundle that is empty", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    call ESMF_GeomGet(fieldbundle%this%geom, geomtype=geomtype, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_GeomDestroy(fieldbundle%this%geom, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    fieldbundle%this%status = ESMF_FBSTATUS_EMPTY

    if(present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldBundleRemoveGeom
! -----------------------------------------------------------------------------

! -------------------------- ESMF-internal method -----------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleToAB()"
!BOPI
! !IROUTINE: ESMF_FieldBundleToAB - Create an ArrayBundle from a FieldBundle
!
! !INTERFACE:
  function ESMF_FieldBundleToAB(fieldbundle, rc)

!
! !RETURN VALUE:
    type(ESMF_ArrayBundle) :: ESMF_FieldBundleToAB

!
! !ARGUMENTS:
    type(ESMF_FieldBundle), intent(in) :: fieldbundle
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an ArrayBundle from a FieldBundle.
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! fieldbundle object.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI

    integer :: fcount, i, localrc
    type(ESMF_Field), pointer :: flist(:)
    type(ESMF_Array), pointer :: alist(:)
    character(800) :: name, msgString

    localrc = ESMF_RC_NOT_IMPL
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldBundleGetInit, fieldbundle, rc)

    if(.not. fieldbundle%this%is_packed) then

      call ESMF_FieldBundleGet(fieldbundle, fieldCount=fcount, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      allocate(flist(fcount), stat=localrc)
      if (ESMF_LogFoundAllocError(localrc, msg= "allocating flist", &
        ESMF_CONTEXT, rcToReturn=rc)) return ! bail out

      call ESMF_FieldBundleGet(fieldbundle, fieldList=flist, &
        itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    else
      fcount = 1

      allocate(flist(fcount), stat=localrc)
      if (ESMF_LogFoundAllocError(localrc, msg= "allocating flist", &
        ESMF_CONTEXT, rcToReturn=rc)) return ! bail out

      call ESMF_ContainerGet(fieldbundle%this%container, &
        itemList=flist, itemorderflag=ESMF_ITEMORDER_ADDORDER, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    allocate(alist(fcount), stat=localrc)
    if (ESMF_LogFoundAllocError(localrc, msg= "allocating alist", &
      ESMF_CONTEXT, rcToReturn=rc)) return ! bail out

    do i = 1, fcount
        call ESMF_FieldGet(flist(i), array=alist(i), rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return







    enddo

    ESMF_FieldBundleToAB = ESMF_ArrayBundleCreate(arrayList=alist, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    deallocate(alist, flist)

    if(present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldBundleToAB
! -----------------------------------------------------------------------------

! -------------------------- ESMF-internal method -----------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldBundleGetInit()"
!BOPI
! !IROUTINE: ESMF_FieldBundleGetInit - Internal access routine for init code
!
! !INTERFACE:
      function ESMF_FieldBundleGetInit(fieldbundle)
!
! !RETURN VALUE:
      ESMF_INIT_TYPE :: ESMF_FieldBundleGetInit
!
! !ARGUMENTS:
      type(ESMF_FieldBundle), intent(in), optional :: fieldbundle
!
! !DESCRIPTION:
! Access deep object init code.
!
! The arguments are:
! \begin{description}
! \item [fieldbundle]
! fieldbundle object.
! \end{description}
!
!EOPI

    if (present(fieldbundle)) then
      ESMF_FieldBundleGetInit = ESMF_INIT_GET(fieldbundle)
    else
      ESMF_FieldBundleGetInit = ESMF_INIT_CREATED
    endif

    end function ESMF_FieldBundleGetInit
!------------------------------------------------------------------------------

    function ESMF_FieldBundleStatusEQ(sf1, sf2)
     logical ESMF_FieldBundleStatusEQ
     type(ESMF_FieldBundleStatus), intent(in) :: sf1, sf2

     ESMF_FieldBundleStatusEQ = (sf1%status == sf2%status)
    end function

    function ESMF_FieldBundleStatusNE(sf1, sf2)
     logical ESMF_FieldBundleStatusNE
     type(ESMF_FieldBundleStatus), intent(in) :: sf1, sf2

     ESMF_FieldBundleStatusNE = (sf1%status /= sf2%status)
    end function

    subroutine decodeName(string, delimiter, decodeCount, strArray)
      ! splitstring splits a string to an array of
      ! substrings based on a selected delimiter
      ! note any facing space/blank in substrings will be removed
      ! Credit to Mohammad from comp.lang.fortran with slight modification

      character(len=*), intent(in) :: string
      character, intent(in) :: delimiter
      integer, intent(out):: decodeCount
      character(len=ESMF_MAXSTR), pointer :: strarray(:)

      ! Local Variables
      integer :: n, m, p
      integer :: i, idx
      character(len=:), allocatable:: strtmp


      ! 0. remove initial blanks if any
      strtmp=trim (adjustl(string) )

      ! 1. count the number substrings separated by delimiter
      i = 1
      n = count( [ (strtmp(i:i), i=1, len_trim(strtmp)) ] == delimiter)
      ! 4. allocate the output string array
      allocate(strarray(n+1))
      decodeCount = n+1

      m=1
      do i=1, n
          idx=index(strtmp(m:),delimiter)
          strarray(i) = adjustl( strtmp(m:m+idx-2) )
          m = m + idx
      end do
      strarray(n+1)=adjustl(strtmp(m:) )

      ! 6. pring the results
      !print*, '"'
      !print *, 'nvalues=', n+1
      !print '(a)', strarray(1:n+1)
      !print*,'----------------------'

    end subroutine decodeName

    ! Have to work around macro expansion not allowing
    ! We will want to inline this for performance
    !dir$ attributes forceinline :: frob

    function concatName(str, delim, append)
      character(len=*), intent(in) :: str
      character(len=*), intent(in) :: delim
      character(len=*), intent(in) :: append

      character(len=len(str)+len(delim)+len(append)) :: concatName

      concatName(:len(str)) = str
      concatName(len(str)+1:len(str)+len(delim)) = delim
      concatName(len(str)+len(delim)+1:) = append
    end function

!------------------------------------------------------------------------------

  subroutine ESMF_FieldBundleTypeDeepCopy(in, out)
    type(ESMF_FieldBundleType) :: in, out
    out%base = in%base
    out%geom = in%geom
    out%container = in%container
    out%status = in%status
    out%is_proxy = in%is_proxy
    out%is_packed = in%is_packed
    call move_alloc(in%encodeName, out%encodeName) ! this seems critical for Intel 21.7.1 and up!
  end subroutine

end module ESMF_FieldBundleMod
