! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
#define ESMF_FILENAME "ESMF_FieldCreate.F90"
!==============================================================================
!
! ESMF FieldCreate module
module ESMF_FieldCreateMod
!
!==============================================================================
!
! This file contains the FieldCreate() methods.
!
!------------------------------------------------------------------------------
! INCLUDES
! < ignore blank lines below. they are created by the files which
! define various macros. >
#include "ESMF.h"

!------------------------------------------------------------------------------
! !USES:
  use ESMF_UtilTypesMod
  use ESMF_BaseMod
  use ESMF_InitMacrosMod
  use ESMF_LogErrMod
  use ESMF_ArraySpecMod
  use ESMF_LocalArrayMod
  use ESMF_DELayoutMod
  use ESMF_StaggerLocMod
  use ESMF_GridMod
  use ESMF_XGridMod
  use ESMF_XGridCreateMod
  use ESMF_XGridGetMod
  use ESMF_XGridGeomBaseMod
  use ESMF_DistGridMod
  use ESMF_MeshMod
  use ESMF_LocStreamMod
  use ESMF_GeomMod
  use ESMF_ArrayMod
  use ESMF_ArrayGetMod
  use ESMF_ArrayCreateMod
  use ESMF_VMMod

  use ESMF_FieldMod
  use ESMF_FieldGetMod
  use ESMF_FieldEmptyMod

  implicit none

!------------------------------------------------------------------------------
! !PRIVATE TYPES:
  private

!------------------------------------------------------------------------------
! !PUBLIC MEMBER FUNCTIONS:

  public operator(==)
  public operator(/=)

  public ESMF_FieldCopy
  public ESMF_FieldCreate
  public ESMF_FieldDestroy ! Destroy a Field
  public ESMF_FieldDestruct ! For internal ESMF use only


!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
  character(*), parameter, private :: version = &
    '$Id$'

!==============================================================================
!
! INTERFACE BLOCKS
!
!==============================================================================


! -------------------------- ESMF-public method -------------------------------
!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_FieldCreate - Create a new Field with data
!
! !INTERFACE:
  interface ESMF_FieldCreate

! !PRIVATE MEMBER FUNCTIONS:
!

    module procedure ESMF_FieldCreateGBTKR
    module procedure ESMF_FieldCreateGBArraySpec
    module procedure ESMF_FieldCreateGBArray
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateGBData1DI1 
 module procedure ESMF_FieldCreateGBData2DI1 
 module procedure ESMF_FieldCreateGBData3DI1 
 module procedure ESMF_FieldCreateGBData4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGBData5DI1 
 module procedure ESMF_FieldCreateGBData6DI1 
 module procedure ESMF_FieldCreateGBData7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateGBData1DI2 
 module procedure ESMF_FieldCreateGBData2DI2 
 module procedure ESMF_FieldCreateGBData3DI2 
 module procedure ESMF_FieldCreateGBData4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGBData5DI2 
 module procedure ESMF_FieldCreateGBData6DI2 
 module procedure ESMF_FieldCreateGBData7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateGBData1DI4 
 module procedure ESMF_FieldCreateGBData1DI8 
 module procedure ESMF_FieldCreateGBData1DR4 
 module procedure ESMF_FieldCreateGBData1DR8 
 module procedure ESMF_FieldCreateGBData2DI4 
 module procedure ESMF_FieldCreateGBData2DI8 
 module procedure ESMF_FieldCreateGBData2DR4 
 module procedure ESMF_FieldCreateGBData2DR8 
 module procedure ESMF_FieldCreateGBData3DI4 
 module procedure ESMF_FieldCreateGBData3DI8 
 module procedure ESMF_FieldCreateGBData3DR4 
 module procedure ESMF_FieldCreateGBData3DR8 
 module procedure ESMF_FieldCreateGBData4DI4 
 module procedure ESMF_FieldCreateGBData4DI8 
 module procedure ESMF_FieldCreateGBData4DR4 
 module procedure ESMF_FieldCreateGBData4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGBData5DI4 
 module procedure ESMF_FieldCreateGBData5DI8 
 module procedure ESMF_FieldCreateGBData5DR4 
 module procedure ESMF_FieldCreateGBData5DR8 
 module procedure ESMF_FieldCreateGBData6DI4 
 module procedure ESMF_FieldCreateGBData6DI8 
 module procedure ESMF_FieldCreateGBData6DR4 
 module procedure ESMF_FieldCreateGBData6DR8 
 module procedure ESMF_FieldCreateGBData7DI4 
 module procedure ESMF_FieldCreateGBData7DI8 
 module procedure ESMF_FieldCreateGBData7DR4 
 module procedure ESMF_FieldCreateGBData7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateGBDataPtr1DI1 
 module procedure ESMF_FieldCreateGBDataPtr2DI1 
 module procedure ESMF_FieldCreateGBDataPtr3DI1 
 module procedure ESMF_FieldCreateGBDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGBDataPtr5DI1 
 module procedure ESMF_FieldCreateGBDataPtr6DI1 
 module procedure ESMF_FieldCreateGBDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateGBDataPtr1DI2 
 module procedure ESMF_FieldCreateGBDataPtr2DI2 
 module procedure ESMF_FieldCreateGBDataPtr3DI2 
 module procedure ESMF_FieldCreateGBDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGBDataPtr5DI2 
 module procedure ESMF_FieldCreateGBDataPtr6DI2 
 module procedure ESMF_FieldCreateGBDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateGBDataPtr1DI4 
 module procedure ESMF_FieldCreateGBDataPtr1DI8 
 module procedure ESMF_FieldCreateGBDataPtr1DR4 
 module procedure ESMF_FieldCreateGBDataPtr1DR8 
 module procedure ESMF_FieldCreateGBDataPtr2DI4 
 module procedure ESMF_FieldCreateGBDataPtr2DI8 
 module procedure ESMF_FieldCreateGBDataPtr2DR4 
 module procedure ESMF_FieldCreateGBDataPtr2DR8 
 module procedure ESMF_FieldCreateGBDataPtr3DI4 
 module procedure ESMF_FieldCreateGBDataPtr3DI8 
 module procedure ESMF_FieldCreateGBDataPtr3DR4 
 module procedure ESMF_FieldCreateGBDataPtr3DR8 
 module procedure ESMF_FieldCreateGBDataPtr4DI4 
 module procedure ESMF_FieldCreateGBDataPtr4DI8 
 module procedure ESMF_FieldCreateGBDataPtr4DR4 
 module procedure ESMF_FieldCreateGBDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGBDataPtr5DI4 
 module procedure ESMF_FieldCreateGBDataPtr5DI8 
 module procedure ESMF_FieldCreateGBDataPtr5DR4 
 module procedure ESMF_FieldCreateGBDataPtr5DR8 
 module procedure ESMF_FieldCreateGBDataPtr6DI4 
 module procedure ESMF_FieldCreateGBDataPtr6DI8 
 module procedure ESMF_FieldCreateGBDataPtr6DR4 
 module procedure ESMF_FieldCreateGBDataPtr6DR8 
 module procedure ESMF_FieldCreateGBDataPtr7DI4 
 module procedure ESMF_FieldCreateGBDataPtr7DI8 
 module procedure ESMF_FieldCreateGBDataPtr7DR4 
 module procedure ESMF_FieldCreateGBDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


    module procedure ESMF_FieldCreateGridTKR
    module procedure ESMF_FieldCreateGridArraySpec
    module procedure ESMF_FieldCreateGridArray
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateGridData1DI1 
 module procedure ESMF_FieldCreateGridData2DI1 
 module procedure ESMF_FieldCreateGridData3DI1 
 module procedure ESMF_FieldCreateGridData4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGridData5DI1 
 module procedure ESMF_FieldCreateGridData6DI1 
 module procedure ESMF_FieldCreateGridData7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateGridData1DI2 
 module procedure ESMF_FieldCreateGridData2DI2 
 module procedure ESMF_FieldCreateGridData3DI2 
 module procedure ESMF_FieldCreateGridData4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGridData5DI2 
 module procedure ESMF_FieldCreateGridData6DI2 
 module procedure ESMF_FieldCreateGridData7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateGridData1DI4 
 module procedure ESMF_FieldCreateGridData1DI8 
 module procedure ESMF_FieldCreateGridData1DR4 
 module procedure ESMF_FieldCreateGridData1DR8 
 module procedure ESMF_FieldCreateGridData2DI4 
 module procedure ESMF_FieldCreateGridData2DI8 
 module procedure ESMF_FieldCreateGridData2DR4 
 module procedure ESMF_FieldCreateGridData2DR8 
 module procedure ESMF_FieldCreateGridData3DI4 
 module procedure ESMF_FieldCreateGridData3DI8 
 module procedure ESMF_FieldCreateGridData3DR4 
 module procedure ESMF_FieldCreateGridData3DR8 
 module procedure ESMF_FieldCreateGridData4DI4 
 module procedure ESMF_FieldCreateGridData4DI8 
 module procedure ESMF_FieldCreateGridData4DR4 
 module procedure ESMF_FieldCreateGridData4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGridData5DI4 
 module procedure ESMF_FieldCreateGridData5DI8 
 module procedure ESMF_FieldCreateGridData5DR4 
 module procedure ESMF_FieldCreateGridData5DR8 
 module procedure ESMF_FieldCreateGridData6DI4 
 module procedure ESMF_FieldCreateGridData6DI8 
 module procedure ESMF_FieldCreateGridData6DR4 
 module procedure ESMF_FieldCreateGridData6DR8 
 module procedure ESMF_FieldCreateGridData7DI4 
 module procedure ESMF_FieldCreateGridData7DI8 
 module procedure ESMF_FieldCreateGridData7DR4 
 module procedure ESMF_FieldCreateGridData7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateGridDataPtr1DI1 
 module procedure ESMF_FieldCreateGridDataPtr2DI1 
 module procedure ESMF_FieldCreateGridDataPtr3DI1 
 module procedure ESMF_FieldCreateGridDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGridDataPtr5DI1 
 module procedure ESMF_FieldCreateGridDataPtr6DI1 
 module procedure ESMF_FieldCreateGridDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateGridDataPtr1DI2 
 module procedure ESMF_FieldCreateGridDataPtr2DI2 
 module procedure ESMF_FieldCreateGridDataPtr3DI2 
 module procedure ESMF_FieldCreateGridDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGridDataPtr5DI2 
 module procedure ESMF_FieldCreateGridDataPtr6DI2 
 module procedure ESMF_FieldCreateGridDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateGridDataPtr1DI4 
 module procedure ESMF_FieldCreateGridDataPtr1DI8 
 module procedure ESMF_FieldCreateGridDataPtr1DR4 
 module procedure ESMF_FieldCreateGridDataPtr1DR8 
 module procedure ESMF_FieldCreateGridDataPtr2DI4 
 module procedure ESMF_FieldCreateGridDataPtr2DI8 
 module procedure ESMF_FieldCreateGridDataPtr2DR4 
 module procedure ESMF_FieldCreateGridDataPtr2DR8 
 module procedure ESMF_FieldCreateGridDataPtr3DI4 
 module procedure ESMF_FieldCreateGridDataPtr3DI8 
 module procedure ESMF_FieldCreateGridDataPtr3DR4 
 module procedure ESMF_FieldCreateGridDataPtr3DR8 
 module procedure ESMF_FieldCreateGridDataPtr4DI4 
 module procedure ESMF_FieldCreateGridDataPtr4DI8 
 module procedure ESMF_FieldCreateGridDataPtr4DR4 
 module procedure ESMF_FieldCreateGridDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateGridDataPtr5DI4 
 module procedure ESMF_FieldCreateGridDataPtr5DI8 
 module procedure ESMF_FieldCreateGridDataPtr5DR4 
 module procedure ESMF_FieldCreateGridDataPtr5DR8 
 module procedure ESMF_FieldCreateGridDataPtr6DI4 
 module procedure ESMF_FieldCreateGridDataPtr6DI8 
 module procedure ESMF_FieldCreateGridDataPtr6DR4 
 module procedure ESMF_FieldCreateGridDataPtr6DR8 
 module procedure ESMF_FieldCreateGridDataPtr7DI4 
 module procedure ESMF_FieldCreateGridDataPtr7DI8 
 module procedure ESMF_FieldCreateGridDataPtr7DR4 
 module procedure ESMF_FieldCreateGridDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


    module procedure ESMF_FieldCreateLSTKR
    module procedure ESMF_FieldCreateLSArraySpec
    module procedure ESMF_FieldCreateLSArray
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateLSData1DI1 
 module procedure ESMF_FieldCreateLSData2DI1 
 module procedure ESMF_FieldCreateLSData3DI1 
 module procedure ESMF_FieldCreateLSData4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateLSData5DI1 
 module procedure ESMF_FieldCreateLSData6DI1 
 module procedure ESMF_FieldCreateLSData7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateLSData1DI2 
 module procedure ESMF_FieldCreateLSData2DI2 
 module procedure ESMF_FieldCreateLSData3DI2 
 module procedure ESMF_FieldCreateLSData4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateLSData5DI2 
 module procedure ESMF_FieldCreateLSData6DI2 
 module procedure ESMF_FieldCreateLSData7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateLSData1DI4 
 module procedure ESMF_FieldCreateLSData1DI8 
 module procedure ESMF_FieldCreateLSData1DR4 
 module procedure ESMF_FieldCreateLSData1DR8 
 module procedure ESMF_FieldCreateLSData2DI4 
 module procedure ESMF_FieldCreateLSData2DI8 
 module procedure ESMF_FieldCreateLSData2DR4 
 module procedure ESMF_FieldCreateLSData2DR8 
 module procedure ESMF_FieldCreateLSData3DI4 
 module procedure ESMF_FieldCreateLSData3DI8 
 module procedure ESMF_FieldCreateLSData3DR4 
 module procedure ESMF_FieldCreateLSData3DR8 
 module procedure ESMF_FieldCreateLSData4DI4 
 module procedure ESMF_FieldCreateLSData4DI8 
 module procedure ESMF_FieldCreateLSData4DR4 
 module procedure ESMF_FieldCreateLSData4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateLSData5DI4 
 module procedure ESMF_FieldCreateLSData5DI8 
 module procedure ESMF_FieldCreateLSData5DR4 
 module procedure ESMF_FieldCreateLSData5DR8 
 module procedure ESMF_FieldCreateLSData6DI4 
 module procedure ESMF_FieldCreateLSData6DI8 
 module procedure ESMF_FieldCreateLSData6DR4 
 module procedure ESMF_FieldCreateLSData6DR8 
 module procedure ESMF_FieldCreateLSData7DI4 
 module procedure ESMF_FieldCreateLSData7DI8 
 module procedure ESMF_FieldCreateLSData7DR4 
 module procedure ESMF_FieldCreateLSData7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateLSDataPtr1DI1 
 module procedure ESMF_FieldCreateLSDataPtr2DI1 
 module procedure ESMF_FieldCreateLSDataPtr3DI1 
 module procedure ESMF_FieldCreateLSDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateLSDataPtr5DI1 
 module procedure ESMF_FieldCreateLSDataPtr6DI1 
 module procedure ESMF_FieldCreateLSDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateLSDataPtr1DI2 
 module procedure ESMF_FieldCreateLSDataPtr2DI2 
 module procedure ESMF_FieldCreateLSDataPtr3DI2 
 module procedure ESMF_FieldCreateLSDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateLSDataPtr5DI2 
 module procedure ESMF_FieldCreateLSDataPtr6DI2 
 module procedure ESMF_FieldCreateLSDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateLSDataPtr1DI4 
 module procedure ESMF_FieldCreateLSDataPtr1DI8 
 module procedure ESMF_FieldCreateLSDataPtr1DR4 
 module procedure ESMF_FieldCreateLSDataPtr1DR8 
 module procedure ESMF_FieldCreateLSDataPtr2DI4 
 module procedure ESMF_FieldCreateLSDataPtr2DI8 
 module procedure ESMF_FieldCreateLSDataPtr2DR4 
 module procedure ESMF_FieldCreateLSDataPtr2DR8 
 module procedure ESMF_FieldCreateLSDataPtr3DI4 
 module procedure ESMF_FieldCreateLSDataPtr3DI8 
 module procedure ESMF_FieldCreateLSDataPtr3DR4 
 module procedure ESMF_FieldCreateLSDataPtr3DR8 
 module procedure ESMF_FieldCreateLSDataPtr4DI4 
 module procedure ESMF_FieldCreateLSDataPtr4DI8 
 module procedure ESMF_FieldCreateLSDataPtr4DR4 
 module procedure ESMF_FieldCreateLSDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateLSDataPtr5DI4 
 module procedure ESMF_FieldCreateLSDataPtr5DI8 
 module procedure ESMF_FieldCreateLSDataPtr5DR4 
 module procedure ESMF_FieldCreateLSDataPtr5DR8 
 module procedure ESMF_FieldCreateLSDataPtr6DI4 
 module procedure ESMF_FieldCreateLSDataPtr6DI8 
 module procedure ESMF_FieldCreateLSDataPtr6DR4 
 module procedure ESMF_FieldCreateLSDataPtr6DR8 
 module procedure ESMF_FieldCreateLSDataPtr7DI4 
 module procedure ESMF_FieldCreateLSDataPtr7DI8 
 module procedure ESMF_FieldCreateLSDataPtr7DR4 
 module procedure ESMF_FieldCreateLSDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


    module procedure ESMF_FieldCreateMeshTKR
    module procedure ESMF_FieldCreateMeshArraySpec
    module procedure ESMF_FieldCreateMeshArray
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateMeshData1DI1 
 module procedure ESMF_FieldCreateMeshData2DI1 
 module procedure ESMF_FieldCreateMeshData3DI1 
 module procedure ESMF_FieldCreateMeshData4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateMeshData5DI1 
 module procedure ESMF_FieldCreateMeshData6DI1 
 module procedure ESMF_FieldCreateMeshData7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateMeshData1DI2 
 module procedure ESMF_FieldCreateMeshData2DI2 
 module procedure ESMF_FieldCreateMeshData3DI2 
 module procedure ESMF_FieldCreateMeshData4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateMeshData5DI2 
 module procedure ESMF_FieldCreateMeshData6DI2 
 module procedure ESMF_FieldCreateMeshData7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateMeshData1DI4 
 module procedure ESMF_FieldCreateMeshData1DI8 
 module procedure ESMF_FieldCreateMeshData1DR4 
 module procedure ESMF_FieldCreateMeshData1DR8 
 module procedure ESMF_FieldCreateMeshData2DI4 
 module procedure ESMF_FieldCreateMeshData2DI8 
 module procedure ESMF_FieldCreateMeshData2DR4 
 module procedure ESMF_FieldCreateMeshData2DR8 
 module procedure ESMF_FieldCreateMeshData3DI4 
 module procedure ESMF_FieldCreateMeshData3DI8 
 module procedure ESMF_FieldCreateMeshData3DR4 
 module procedure ESMF_FieldCreateMeshData3DR8 
 module procedure ESMF_FieldCreateMeshData4DI4 
 module procedure ESMF_FieldCreateMeshData4DI8 
 module procedure ESMF_FieldCreateMeshData4DR4 
 module procedure ESMF_FieldCreateMeshData4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateMeshData5DI4 
 module procedure ESMF_FieldCreateMeshData5DI8 
 module procedure ESMF_FieldCreateMeshData5DR4 
 module procedure ESMF_FieldCreateMeshData5DR8 
 module procedure ESMF_FieldCreateMeshData6DI4 
 module procedure ESMF_FieldCreateMeshData6DI8 
 module procedure ESMF_FieldCreateMeshData6DR4 
 module procedure ESMF_FieldCreateMeshData6DR8 
 module procedure ESMF_FieldCreateMeshData7DI4 
 module procedure ESMF_FieldCreateMeshData7DI8 
 module procedure ESMF_FieldCreateMeshData7DR4 
 module procedure ESMF_FieldCreateMeshData7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateMeshDataPtr1DI1 
 module procedure ESMF_FieldCreateMeshDataPtr2DI1 
 module procedure ESMF_FieldCreateMeshDataPtr3DI1 
 module procedure ESMF_FieldCreateMeshDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateMeshDataPtr5DI1 
 module procedure ESMF_FieldCreateMeshDataPtr6DI1 
 module procedure ESMF_FieldCreateMeshDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateMeshDataPtr1DI2 
 module procedure ESMF_FieldCreateMeshDataPtr2DI2 
 module procedure ESMF_FieldCreateMeshDataPtr3DI2 
 module procedure ESMF_FieldCreateMeshDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateMeshDataPtr5DI2 
 module procedure ESMF_FieldCreateMeshDataPtr6DI2 
 module procedure ESMF_FieldCreateMeshDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateMeshDataPtr1DI4 
 module procedure ESMF_FieldCreateMeshDataPtr1DI8 
 module procedure ESMF_FieldCreateMeshDataPtr1DR4 
 module procedure ESMF_FieldCreateMeshDataPtr1DR8 
 module procedure ESMF_FieldCreateMeshDataPtr2DI4 
 module procedure ESMF_FieldCreateMeshDataPtr2DI8 
 module procedure ESMF_FieldCreateMeshDataPtr2DR4 
 module procedure ESMF_FieldCreateMeshDataPtr2DR8 
 module procedure ESMF_FieldCreateMeshDataPtr3DI4 
 module procedure ESMF_FieldCreateMeshDataPtr3DI8 
 module procedure ESMF_FieldCreateMeshDataPtr3DR4 
 module procedure ESMF_FieldCreateMeshDataPtr3DR8 
 module procedure ESMF_FieldCreateMeshDataPtr4DI4 
 module procedure ESMF_FieldCreateMeshDataPtr4DI8 
 module procedure ESMF_FieldCreateMeshDataPtr4DR4 
 module procedure ESMF_FieldCreateMeshDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateMeshDataPtr5DI4 
 module procedure ESMF_FieldCreateMeshDataPtr5DI8 
 module procedure ESMF_FieldCreateMeshDataPtr5DR4 
 module procedure ESMF_FieldCreateMeshDataPtr5DR8 
 module procedure ESMF_FieldCreateMeshDataPtr6DI4 
 module procedure ESMF_FieldCreateMeshDataPtr6DI8 
 module procedure ESMF_FieldCreateMeshDataPtr6DR4 
 module procedure ESMF_FieldCreateMeshDataPtr6DR8 
 module procedure ESMF_FieldCreateMeshDataPtr7DI4 
 module procedure ESMF_FieldCreateMeshDataPtr7DI8 
 module procedure ESMF_FieldCreateMeshDataPtr7DR4 
 module procedure ESMF_FieldCreateMeshDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


    module procedure ESMF_FieldCreateXGTKR
    module procedure ESMF_FieldCreateXGArraySpec
    module procedure ESMF_FieldCreateXGArray
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateXGData1DI1 
 module procedure ESMF_FieldCreateXGData2DI1 
 module procedure ESMF_FieldCreateXGData3DI1 
 module procedure ESMF_FieldCreateXGData4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateXGData5DI1 
 module procedure ESMF_FieldCreateXGData6DI1 
 module procedure ESMF_FieldCreateXGData7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateXGData1DI2 
 module procedure ESMF_FieldCreateXGData2DI2 
 module procedure ESMF_FieldCreateXGData3DI2 
 module procedure ESMF_FieldCreateXGData4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateXGData5DI2 
 module procedure ESMF_FieldCreateXGData6DI2 
 module procedure ESMF_FieldCreateXGData7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateXGData1DI4 
 module procedure ESMF_FieldCreateXGData1DI8 
 module procedure ESMF_FieldCreateXGData1DR4 
 module procedure ESMF_FieldCreateXGData1DR8 
 module procedure ESMF_FieldCreateXGData2DI4 
 module procedure ESMF_FieldCreateXGData2DI8 
 module procedure ESMF_FieldCreateXGData2DR4 
 module procedure ESMF_FieldCreateXGData2DR8 
 module procedure ESMF_FieldCreateXGData3DI4 
 module procedure ESMF_FieldCreateXGData3DI8 
 module procedure ESMF_FieldCreateXGData3DR4 
 module procedure ESMF_FieldCreateXGData3DR8 
 module procedure ESMF_FieldCreateXGData4DI4 
 module procedure ESMF_FieldCreateXGData4DI8 
 module procedure ESMF_FieldCreateXGData4DR4 
 module procedure ESMF_FieldCreateXGData4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateXGData5DI4 
 module procedure ESMF_FieldCreateXGData5DI8 
 module procedure ESMF_FieldCreateXGData5DR4 
 module procedure ESMF_FieldCreateXGData5DR8 
 module procedure ESMF_FieldCreateXGData6DI4 
 module procedure ESMF_FieldCreateXGData6DI8 
 module procedure ESMF_FieldCreateXGData6DR4 
 module procedure ESMF_FieldCreateXGData6DR8 
 module procedure ESMF_FieldCreateXGData7DI4 
 module procedure ESMF_FieldCreateXGData7DI8 
 module procedure ESMF_FieldCreateXGData7DR4 
 module procedure ESMF_FieldCreateXGData7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldCreateXGDataPtr1DI1 
 module procedure ESMF_FieldCreateXGDataPtr2DI1 
 module procedure ESMF_FieldCreateXGDataPtr3DI1 
 module procedure ESMF_FieldCreateXGDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateXGDataPtr5DI1 
 module procedure ESMF_FieldCreateXGDataPtr6DI1 
 module procedure ESMF_FieldCreateXGDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldCreateXGDataPtr1DI2 
 module procedure ESMF_FieldCreateXGDataPtr2DI2 
 module procedure ESMF_FieldCreateXGDataPtr3DI2 
 module procedure ESMF_FieldCreateXGDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateXGDataPtr5DI2 
 module procedure ESMF_FieldCreateXGDataPtr6DI2 
 module procedure ESMF_FieldCreateXGDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldCreateXGDataPtr1DI4 
 module procedure ESMF_FieldCreateXGDataPtr1DI8 
 module procedure ESMF_FieldCreateXGDataPtr1DR4 
 module procedure ESMF_FieldCreateXGDataPtr1DR8 
 module procedure ESMF_FieldCreateXGDataPtr2DI4 
 module procedure ESMF_FieldCreateXGDataPtr2DI8 
 module procedure ESMF_FieldCreateXGDataPtr2DR4 
 module procedure ESMF_FieldCreateXGDataPtr2DR8 
 module procedure ESMF_FieldCreateXGDataPtr3DI4 
 module procedure ESMF_FieldCreateXGDataPtr3DI8 
 module procedure ESMF_FieldCreateXGDataPtr3DR4 
 module procedure ESMF_FieldCreateXGDataPtr3DR8 
 module procedure ESMF_FieldCreateXGDataPtr4DI4 
 module procedure ESMF_FieldCreateXGDataPtr4DI8 
 module procedure ESMF_FieldCreateXGDataPtr4DR4 
 module procedure ESMF_FieldCreateXGDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldCreateXGDataPtr5DI4 
 module procedure ESMF_FieldCreateXGDataPtr5DI8 
 module procedure ESMF_FieldCreateXGDataPtr5DR4 
 module procedure ESMF_FieldCreateXGDataPtr5DR8 
 module procedure ESMF_FieldCreateXGDataPtr6DI4 
 module procedure ESMF_FieldCreateXGDataPtr6DI8 
 module procedure ESMF_FieldCreateXGDataPtr6DR4 
 module procedure ESMF_FieldCreateXGDataPtr6DR8 
 module procedure ESMF_FieldCreateXGDataPtr7DI4 
 module procedure ESMF_FieldCreateXGDataPtr7DI8 
 module procedure ESMF_FieldCreateXGDataPtr7DR4 
 module procedure ESMF_FieldCreateXGDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


    module procedure ESMF_FieldCreateFromField

! !DESCRIPTION:
! This interface provides an entry point for methods that create a complete
! {\tt ESMF\_Field}. These method all contain an {\tt ESMF\_Grid} and
! {\tt ESMF\_Data}. The variations allow the user to specify the data
! using either a Fortran array or an {\tt ESMF\_Array}.
  end interface
!EOPI
!------------------------------------------------------------------------------

!===============================================================================
! FieldOperator() interfaces
!===============================================================================

! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_FieldAssignment(=) - Field assignment
!
! !INTERFACE:
! interface assignment(=)
! field1 = field2
!
! !ARGUMENTS:
! type(ESMF_Field) :: field1
! type(ESMF_Field) :: field2
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Assign field1 as an alias to the same ESMF Field object in memory
! as field2. If field2 is invalid, then field1 will be equally invalid after
! the assignment.
!
! The arguments are:
! \begin{description}
! \item[field1]
! The {\tt ESMF\_Field} object on the left hand side of the assignment.
! \item[field2]
! The {\tt ESMF\_Field} object on the right hand side of the assignment.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_FieldOperator(==) - Field equality operator
!
! !INTERFACE:
  interface operator(==)
! if (field1 == field2) then ... endif
! OR
! result = (field1 == field2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_Field), intent(in) :: field1
! type(ESMF_Field), intent(in) :: field2
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Test whether field1 and field2 are valid aliases to the same ESMF
! Field object in memory. For a more general comparison of two ESMF Fields,
! going beyond the simple alias test, the ESMF\_FieldMatch() function (not yet
! implemented) must be used.
!
! The arguments are:
! \begin{description}
! \item[field1]
! The {\tt ESMF\_Field} object on the left hand side of the equality
! operation.
! \item[field2]
! The {\tt ESMF\_Field} object on the right hand side of the equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_FieldEQ

  end interface
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_FieldOperator(/=) - Field not equal operator
!
! !INTERFACE:
  interface operator(/=)
! if (field1 /= field2) then ... endif
! OR
! result = (field1 /= field2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_Field), intent(in) :: field1
! type(ESMF_Field), intent(in) :: field2
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Test whether field1 and field2 are {\it not} valid aliases to the
! same ESMF Field object in memory. For a more general comparison of two ESMF
! Fields, going beyond the simple alias test, the ESMF\_FieldMatch() function
! (not yet implemented) must be used.
!
! The arguments are:
! \begin{description}
! \item[field1]
! The {\tt ESMF\_Field} object on the left hand side of the non-equality
! operation.
! \item[field2]
! The {\tt ESMF\_Field} object on the right hand side of the non-equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_FieldNE

  end interface
!------------------------------------------------------------------------------


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

contains

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldEQ()"
!BOPI
! !IROUTINE: ESMF_FieldEQ - Compare two Fields for equality
!
! !INTERFACE:
  impure elemental function ESMF_FieldEQ(field1, field2)
!
! !RETURN VALUE:
    logical :: ESMF_FieldEQ

! !ARGUMENTS:
    type(ESMF_Field), intent(in) :: field1
    type(ESMF_Field), intent(in) :: field2

! !DESCRIPTION:
! Test if both {\tt field1} and {\tt field2} alias the same ESMF Field
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE finit1, finit2
    integer :: localrc1, localrc2
    logical :: lval1, lval2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ! TODO: Consider moving this logic to C++: use Base class? status?
    ! Or replicate logic for C interface also.

    ! check inputs
    finit1 = ESMF_FieldGetInit(field1)
    finit2 = ESMF_FieldGetInit(field2)

    ! TODO: this line must remain split in two for SunOS f90 8.3 127000-03
    if (finit1 .eq. ESMF_INIT_CREATED .and. &
      finit2 .eq. ESMF_INIT_CREATED) then
      ESMF_FieldEQ = associated(field1%ftypep,field2%ftypep)
    else
      ESMF_FieldEQ = ESMF_FALSE
    endif

  end function ESMF_FieldEQ
!-------------------------------------------------------------------------------


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldNE()"
!BOPI
! !IROUTINE: ESMF_FieldNE - Compare two Fields for non-equality
!
! !INTERFACE:
  impure elemental function ESMF_FieldNE(field1, field2)
!
! !RETURN VALUE:
    logical :: ESMF_FieldNE

! !ARGUMENTS:
    type(ESMF_Field), intent(in) :: field1
    type(ESMF_Field), intent(in) :: field2

! !DESCRIPTION:
! Test if both {\tt field1} and {\tt field2} alias the same ESMF Field
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE finit1, finit2
    integer :: localrc1, localrc2
    logical :: lval1, lval2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ESMF_FieldNE = .not.ESMF_FieldEQ(field1, field2)

  end function ESMF_FieldNE
!-------------------------------------------------------------------------------

! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCopy()"
!BOP
! !IROUTINE: ESMF_FieldCopy - Copy data from one Field object to another

! !INTERFACE:
  subroutine ESMF_FieldCopy(fieldOut, fieldIn, keywordEnforcer, rc)
!
! !ARGUMENTS:
       type(ESMF_Field), intent(inout) :: fieldOut
       type(ESMF_Field), intent(in) :: fieldIn
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
       integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Copy data from one {\tt ESMF\_Field} object to another.
!
! The arguments are:
! \begin{description}
! \item[fieldOut]
! {\tt ESMF\_Field} object into which to copy the data. The incoming
! {\tt fieldOut} must already references a matching memory allocation.
! \item[fieldIn]
! {\tt ESMF\_Field} object that holds the data to be copied.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code

    ! Initialize return code; assume failure until success is certain
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldGetInit, fieldIn, rc)
    ESMF_INIT_CHECK_DEEP_SHORT(ESMF_FieldGetInit, fieldOut, rc)

    ! Call into the Array method
    call ESMF_ArrayCopy(fieldOut%ftypep%array, fieldIn%ftypep%array, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldCopy
!------------------------------------------------------------------------------

! ESMF_FieldCreateGe

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateGBTKR"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Geom and typekind

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
function ESMF_FieldCreateGBTKR(geom, typekind, keywordEnforcer, &
     gridToFieldMap, ungriddedLBound, ungriddedUBound, &
     totalLWidth, totalUWidth, pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateGBTKR
!
! !ARGUMENTS:
    type(ESMF_Geom), intent(in) :: geom
    type(ESMF_TypeKind_Flag),intent(in) :: typekind
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_tkr}.
!
! The arguments are:
! \begin{description}
! \item [geom]
! {\tt ESMF\_Geom} object.
! \item[typekind]
! The typekind of the Field. See section \ref{const:typekind}
! for a list of valid typekind options.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Grid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Grid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_ArraySpec) :: arrayspec
    integer :: rank, ungridded_dimcount, ungridded_ldimcount, ungridded_udimcount
    integer :: memDimCount, grid_repdimcount, i
    character(320) :: msgString

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Get ungridded dim information
    ungridded_dimcount = 0
    if(present(ungriddedLBound)) then
      ungridded_ldimcount = size(ungriddedLBound)
      ungridded_dimcount = ungridded_ldimcount
    endif
    if(present(ungriddedUBound)) then
      ungridded_udimcount = size(ungriddedUBound)
      ungridded_dimcount = ungridded_udimcount
    endif

    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif


    ! Get number of replicated dims
    grid_repdimcount = 0
    if (present(gridToFieldMap)) then
       do i = 1, size(gridToFieldMap)
           if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
       enddo
    endif

    ! Get rank of geom
    call ESMF_GeomGet(geom, rank=memDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return


    ! Calculate rank from dimension information
    rank = ungridded_dimcount + memDimCount - grid_repdimcount
    ! Set ArraySpec from rank and typekind input
    call ESMF_ArraySpecSet(arrayspec, rank=rank, typekind=typekind, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ESMF_FieldCreateGBTKR = ESMF_FieldCreate(geom, arrayspec, &
      gridToFieldMap=gridToFieldMap, ungriddedLBound=ungriddedLBound, &
      ungriddedUBound=ungriddedUBound, totalLWidth=totalLWidth, &
      totalUWidth=totalUWidth, pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

end function ESMF_FieldCreateGBTKR

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateGBArraySpec"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Geom and ArraySpec

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateGBArraySpec(geom, arrayspec, gridToFieldMap, &
    ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, &
    pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateGBArraySpec
!
! !ARGUMENTS:
    type(ESMF_Geom) :: geom
    type(ESMF_ArraySpec), intent(in) :: arrayspec
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_arrayspec}.
!
! The arguments are:
! \begin{description}
! \item [geom]
! {\tt ESMF\_Geom} object.
! \item [arrayspec]
! Data type and kind specification.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    type(ESMF_FieldType), pointer :: ftype ! Pointer to new field
    integer :: localrc ! Local error code
    logical :: rcpresent ! Return code present
    integer :: memDimCount, fieldDimCount, fieldUngriddedDimCount
    integer :: gridDimCount, gridDimCount_norep, grid_repdimcount
    integer :: fieldUndistDimCount
    integer :: i,j
    integer :: localGridToFieldMap(ESMF_MAXDIM)
    integer :: localMaxHaloLWidth (ESMF_MAXDIM)
    integer :: localMaxHaloUWidth (ESMF_MAXDIM)
    integer :: distgridToGridMap(ESMF_MAXDIM)
    logical :: flipflop(ESMF_MAXDIM)
    logical :: found
    type(ESMF_GridDecompType) :: decompType
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    integer :: arbdim
    character(320) :: msgString

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL
    rcpresent = .FALSE.
    nullify(ftype)
    nullify(ESMF_FieldCreateGBArraySpec%ftypep)

    ! Initialize return code
    if(present(rc)) then
      rcpresent=.TRUE.
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    ! Check if geom is a grid, if so, check if it is arbitrary
    decompType = ESMF_GRID_NONARBITRARY
    call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if (geomtype .eq. ESMF_GEOMTYPE_GRID) then
       call ESMF_GeomGet(geom, grid=grid, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
       call ESMF_GridGetDecompType(grid, decompType, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Get rank of proposed Field.
    call ESMF_ArraySpecGet(arrayspec, rank=memDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    ! Get number of grid dimensions, number
    ! of distributed grid dimensions, distgrid,
    ! number of ungridded Field dimensions,
    ! and number of undistributed Field Dimensions
    call ESMF_GeomGet(geom, dimCount=gridDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Error Check Input
    if (present(gridToFieldMap)) then
        if (size(gridToFieldMap) .ne. gridDimCount) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
             msg="- gridToFieldMap size must equal to grid_rank", &
             ESMF_CONTEXT, rcToReturn=rc)
            return
        endif
    endif

    grid_repdimcount = 0
    localGridToFieldMap = -1
    if (present(gridToFieldMap)) then
       localGridToFieldMap(1:gridDimCount) = &
         gridToFieldMap (1:gridDimCount)
       do i = 1, size(gridToFieldMap)
           if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
       enddo
    else
      do i = 1, gridDimCount
        localGridToFieldMap(i) = i
      enddo
    endif
    gridDimCount_norep = gridDimCount - grid_repdimcount

    if (decompType .eq. ESMF_GRID_NONARBITRARY) then
            fieldDimCount = memDimCount
    else
        call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, &
            rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        ! find out how many grid dimensions are arbitrarily distributed and calculate
        ! fieldDimCount accordingly
        arbdim = 0
        do i=1,gridDimCount
          if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1
        enddo
        fieldDimCount = memDimCount + arbdim - 1

        ! If there is any replicated dimension, check if any of the arb. dimensions are replicated.
        ! If one arb dimension is replicated, all the arb. dimensions have to be replicated
        if (grid_repdimcount .ne. 0) then
          do i = 1,gridDimCount
            if(localGridToFieldMap(i) == 0) then
                found = .false.
                do j=1,arbdim
                    if (distgridToGridMap(j) .eq. i) found = .true.
                enddo
                if (found) then
                  ! one arb.dimension is a replicated dimension, check if other arb dimensions are
                  ! also replicated
                  do j=1,arbdim
                    if (distgridToGridMap(j) .ne. i) then
                        if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then
                            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
  msg="- Arb. grid dimensions have to be either all replicated or not replicated", &
  ESMF_CONTEXT, rcToReturn=rc)
                            return
                        endif
                    endif
                  enddo
                  ! all arb. dimension are replication, jump out of the first do loop
                  ! fieldDimCount should be the same as the memDimCount
                  fieldDimCount = memDimCount
                  exit
                endif
             endif
           enddo
        endif
    endif

    if(fieldDimCount .lt. gridDimCount_norep) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
        msg="- Field rank must be greater than or equal to its gridded rank", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    if(fieldDimCount .gt. gridDimCount_norep) then
      if( (.not. present(ungriddedLBound)) .or. &
          (.not. present(ungriddedUBound)) ) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
          msg="- ungridded bounds must be present when Field has ungridded dimension(s)", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount
    fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount

    if (present(ungriddedLBound)) then
       if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then
          write(msgString,*) "ungriddedLBound does not match: ", &
            " fieldUngriddedDimCount=", fieldUngriddedDimCount, &
            " fieldDimCount=", fieldDimCount, &
            " gridDimCount=", gridDimCount, &
            " grid_repdimcount=", grid_repdimcount
          call ESMF_LogWrite(msgString, ESMF_LOGMSG_INFO, rc=rc)
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
               msg="- ungriddedLBound size must equal to array_rank-grid_rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    if (present(ungriddedUBound)) then
       if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
                 msg="- ungriddedUBound size must equal to array_rank-grid_rank", &
                 ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    if (present(totalLWidth)) then
        if (size(totalLWidth) .ne. gridDimCount_norep) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
              msg="- totalLWidth must equal to grid dimCount", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif
    endif

    if (present(totalUWidth)) then
       if (size(totalUWidth) .ne. gridDimCount_norep) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
              msg="- totalUWidth must equal to grid dimCount", &
                ESMF_CONTEXT, rcToReturn=rc)
           return
       endif
    endif

    ! gridToFieldMap elements must be in range 1...fieldRank and unique
    ! algorithm to check element uniqueness:
    ! run time: O(ESMF_MAXDIM)
    ! memory: O(2*ESMF_MAXDIM)
    ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector
    flipflop = .false.
    do i = 1, gridDimCount
       if(localGridToFieldMap(i) .lt. 0 .and. &
         localGridToFieldMap(i) .gt. fieldDimCount) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
             msg="- gridToFieldMap element must be within range 0...array rank", &
             ESMF_CONTEXT, rcToReturn=rc)
           return
       endif
       if(localGridToFieldMap(i) /= 0) then
           if(flipflop(localGridToFieldMap(i))) then
             call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="- gridToFieldMap element must be unique", &
               ESMF_CONTEXT, rcToReturn=rc)
             return
           endif
           flipflop(localGridToFieldMap(i)) = .true.
       endif
    enddo

     if(present(totalLWidth)) then
       localMaxHaloLWidth(1:gridDimCount_norep) = &
          totalLWidth (1:gridDimCount_norep)
     else
          localMaxHaloLWidth = 0
     endif

     if(present(totalUWidth)) then
       localMaxHaloUWidth(1:gridDimCount_norep) = &
          totalUWidth (1:gridDimCount_norep)
     else
          localMaxHaloUWidth = 0
     endif

    ! allocate Fieldtype
    allocate(ftype, stat=localrc)
    if (ESMF_LogFoundAllocError(localrc, &
      msg="Allocating Field Type", &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_FieldInitialize(ftype, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Call construction method to allocate and initialize field internals.
    if (decompType .eq. ESMF_GRID_NONARBITRARY) then
       call ESMF_FieldConstructIANew(ftype, geom, arrayspec, &
                                localGridToFieldMap(1:gridDimCount), ungriddedLBound, &
                                ungriddedUBound, localMaxHaloLWidth(1:gridDimCount_norep), &
                                localMaxHaloUWidth(1:gridDimCount_norep), &
                                pinflag=pinflag, name=name, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return
    else
          call ESMF_FieldConstructIANew(ftype, geom, arrayspec, &
                                localGridToFieldMap(1:gridDimCount), ungriddedLBound, &
                                ungriddedUBound, &
                                pinflag=pinflag, name=name, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Set return values.
    ESMF_FieldCreateGBArraySpec%ftypep => ftype

    ! Add reference to this object into ESMF garbage collection table
    ! Only call this in those Create() methods that call Construct()
    call c_ESMC_VMAddFObject(ESMF_FieldCreateGBArraySpec, &
      ESMF_ID_FIELD%objectID)

    ESMF_INIT_SET_CREATED(ESMF_FieldCreateGBArraySpec)

    call ESMF_FieldValidate(ESMF_FieldCreateGBArraySpec, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) then
      call ESMF_BaseSetStatus(ftype%base, ESMF_STATUS_INVALID, rc=localrc)
      return
    endif

    if(rcpresent) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateGBArraySpec
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateGBArray"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Geom and Array

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateGBArray(geom, array, datacopyflag, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, name, vm, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateGBArray
!
! !ARGUMENTS:
    type(ESMF_Geom), intent(in) :: geom
    type(ESMF_Array), intent(in) :: array
    type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    character (len = *), intent(in), optional :: name
    type(ESMF_VM), intent(in), optional :: vm
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field}. This version of creation
! assumes the data exists already and is being
! passed in through an {\tt ESMF\_Array}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_array}.
!
! The arguments are:
! \begin{description}
! \item [geom]
! {\tt ESMF\_Geom} object.
! \item [array]
! {\tt ESMF\_Array} object.
! \item [{[datacopyflag]}]
! Indicates whether to copy the contents of the {\tt array} or reference it directly.
! For valid values see \ref{const:datacopyflag}. The default is
! {\tt ESMF\_DATACOPY\_REFERENCE}.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[name]}]
! Field name.
! \item[{[vm]}]
! If present, the Field object is created on the specified
! {\tt ESMF\_VM} object. The default is to create on the VM of the
! current component context.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    type(ESMF_FieldType), pointer :: ftype ! Pointer to new field
    integer :: localrc ! Local error code
    logical :: rcpresent ! Return code present

    integer :: memDimCount, fieldDimCount, fieldUngriddedDimCount
    integer :: gridDimCount, gridDimCount_norep, grid_repdimcount
    integer :: fieldUndistDimCount
    integer :: i,j
    integer :: localGridToFieldMap(ESMF_MAXDIM)
    logical :: flipflop(ESMF_MAXDIM)
    type(ESMF_DataCopy_Flag) :: l_datacopyflag
    type (ESMF_Index_Flag) :: arrayIndexFlag, gridIndexFlag
    integer :: distgridToGridMap(ESMF_MAXDIM)
    logical :: found
    type(ESMF_GridDecompType) :: decompType
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    integer :: arbdim
    type(ESMF_Pointer) :: vmThis
    logical :: actualFlag

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL
    rcpresent = .FALSE.
    nullify(ftype)
    nullify(ESMF_FieldCreateGBArray%ftypep)

    ! Initialize return code
    if(present(rc)) then
      rcpresent = .TRUE.
      rc = ESMF_RC_NOT_IMPL
    endif

    ! Must make sure the local PET is associated with an actual member
    actualFlag = .true.
    if (present(vm)) then
      ESMF_INIT_CHECK_DEEP(ESMF_VMGetInit, vm, rc)
      call ESMF_VMGetThis(vm, vmThis)
      if (vmThis == ESMF_NULL_POINTER) then
        actualFlag = .false. ! local PET is not for an actual member
      endif
    endif

    if (actualFlag) then

      ! check init status of input parameters
      ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc)
      ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,array,rc)

      ! Check if geom is a grid, if so, check if it is arbitrary
      decompType = ESMF_GRID_NONARBITRARY
      call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      if (geomtype .eq. ESMF_GEOMTYPE_GRID) then
         call ESMF_GeomGet(geom, grid=grid, rc=localrc)
         if (ESMF_LogFoundError(localrc, &
             ESMF_ERR_PASSTHRU, &
             ESMF_CONTEXT, rcToReturn=rc)) return
         call ESMF_GridGetDecompType(grid, decompType, rc=localrc)
         if (ESMF_LogFoundError(localrc, &
             ESMF_ERR_PASSTHRU, &
             ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      ! Get rank of proposed Field.
      call ESMF_ArrayGet(array, rank=memDimCount, &
                         indexflag=arrayIndexFlag, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return

      ! Get number of grid dimensions, number
      ! of distributed grid dimensions, distgrid,
      ! number of ungridded Field dimensions,
      ! and number of undistributed Field Dimensions
      call ESMF_GeomGet(geom, dimCount=gridDimCount, &
                        indexflag=gridIndexFlag, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      ! Error Check Input
      if (present(gridToFieldMap)) then
          if (size(gridToFieldMap) .ne. gridDimCount) then
              call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
                msg="- gridToFieldMap size must equal to grid_rank", &
                ESMF_CONTEXT, rcToReturn=rc)
              return
          endif
      endif

      grid_repdimcount = 0
      localGridToFieldMap = -1
      if (present(gridToFieldMap)) then
         localGridToFieldMap(1:gridDimCount) = &
           gridToFieldMap (1:gridDimCount)
         do i = 1, size(gridToFieldMap)
             if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
         enddo
      else
        do i = 1, gridDimCount
          localGridToFieldMap(i) = i
        enddo
      endif
      gridDimCount_norep = gridDimCount - grid_repdimcount

      if (decompType .eq. ESMF_GRID_NONARBITRARY) then
              fieldDimCount = memDimCount
      else
          call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, &
              rc=localrc)
          if (ESMF_LogFoundError(localrc, &
              ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return
          ! find out how many grid dimensions are arbitrarily distributed and calculate
          ! fieldDimCount accordingly
          arbdim = 0
          do i=1,gridDimCount
            if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1
          enddo
          fieldDimCount = memDimCount + arbdim - 1

          ! If there is any replicated dimension, check if any of the arb. dimensions are replicated.
          ! If one arb dimension is replicated, all the arb. dimensions have to be replicated
          if (grid_repdimcount .ne. 0) then
            do i = 1,gridDimCount
              if(localGridToFieldMap(i) == 0) then
                  found = .false.
                  do j=1,arbdim
                      if (distgridToGridMap(j) .eq. i) found = .true.
                  enddo
                  if (found) then
                    ! one arb.dimension is a replicated dimension, check if other arb dimensions are
                    ! also replicated
                    do j=1,arbdim
                      if (distgridToGridMap(j) .ne. i) then
                          if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then
                              call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
    msg="- Arb. grid dimensions have to be either all replicated or not replicated", &
    ESMF_CONTEXT, rcToReturn=rc)
                              return
                          endif
                      endif
                    enddo
                    ! all arb. dimension are replication, jump out of the first do loop
                    ! fieldDimCount should be the same as the memDimCount
                    fieldDimCount = memDimCount
                    exit
                  endif
               endif
             enddo
          endif
      endif

      if(fieldDimCount .lt. gridDimCount_norep) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
            msg="- Field rank must be greater than or equal to its gridded rank", &
            ESMF_CONTEXT, rcToReturn=rc)
          return
      endif

      if(fieldDimCount .gt. gridDimCount_norep) then
        if( (.not. present(ungriddedLBound)) .or. &
            (.not. present(ungriddedUBound)) ) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
           msg="- ungridded bounds must be present when Field has ungridded dimension(s)", &
             ESMF_CONTEXT, rcToReturn=rc)
          return
        endif
      endif

      fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount
      fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount

      ! Error Check Input
      if (present(gridToFieldMap)) then
          if (size(gridToFieldMap) .ne. gridDimCount) then
              call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
                msg="- gridToFieldMap size must equal to grid_rank", &
                ESMF_CONTEXT, rcToReturn=rc)
              return
          endif
      endif

      if (present(ungriddedLBound)) then
         if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
              msg="- ungriddedLBound size must equal to array_rank-grid_rank", &
              ESMF_CONTEXT, rcToReturn=rc)
            return
         endif
      endif

      if (present(ungriddedUBound)) then
         if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
              msg="- ungriddedUBound size must equal to array_rank-grid_rank", &
              ESMF_CONTEXT, rcToReturn=rc)
            return
         endif
      endif

      if (present(totalLWidth)) then
          if (size(totalLWidth) .ne. gridDimCount_norep) then
              call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
                msg="- totalLWidth must equal to grid dimCount", &
                ESMF_CONTEXT, rcToReturn=rc)
              return
          endif
      endif

      if (present(totalUWidth)) then
         if (size(totalUWidth) .ne. gridDimCount_norep) then
             call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
                msg="- totalUWidth must equal to grid dimCount", &
                ESMF_CONTEXT, rcToReturn=rc)
             return
         endif
      endif

      if (.not.(arrayIndexFlag .eq. gridIndexFlag)) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
            msg="- Array indexflag must be the same as the Grid indexflag", &
            ESMF_CONTEXT, rcToReturn=rc)
          return
      endif

      if (present(datacopyflag)) then
          l_datacopyflag = datacopyflag
      else
          l_datacopyflag = ESMF_DATACOPY_REFERENCE
      endif

      ! gridToFieldMap elements must be in range 1...fieldRank and unique
      ! algorithm to check element uniqueness:
      ! run time: O(ESMF_MAXDIM)
      ! memory: O(2*ESMF_MAXDIM)
      ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector
      flipflop = .false.
      do i = 1, gridDimCount
         if(localGridToFieldMap(i) .lt. 0 .and. &
           localGridToFieldMap(i) .gt. fieldDimCount) then
             call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="- gridToFieldMap element must be within range 0...array rank", &
               ESMF_CONTEXT, rcToReturn=rc)
             return
         endif
         if(localGridToFieldMap(i) /= 0) then
             if(flipflop(localGridToFieldMap(i))) then
                 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
                   msg="- gridToFieldMap element must be unique", &
                   ESMF_CONTEXT, rcToReturn=rc)
                 return
             endif
             flipflop(localGridToFieldMap(i)) = .true.
         endif
      enddo

      allocate(ftype, stat=localrc)
      if (ESMF_LogFoundAllocError(localrc, msg="Allocating Field information", &
        ESMF_CONTEXT, rcToReturn=rc)) return
      call ESMF_FieldInitialize(ftype, rc=localrc)
      if (ESMF_LogFoundAllocError(localrc, msg="Default initialize Field", &
        ESMF_CONTEXT, rcToReturn=rc)) return

      if (decompType .eq. ESMF_GRID_NONARBITRARY) then
          ! Call construction method to allocate and initialize field internals.
          call ESMF_FieldConstructIANewArray(ftype, geom, array, l_datacopyflag, &
                                           localgridToFieldMap, &
                                           ungriddedLBound=ungriddedLBound, &
                                           ungriddedUBound=ungriddedUBound, &
                                           totalLWidth=totalLWidth, &
                                           totalUWidth=totalUWidth, &
                                           name=name, vm=vm, rc=localrc)
          if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
      else
              ! Call construction method to allocate and initialize field internals.
          call ESMF_FieldConstructIANewArray(ftype, geom, array, l_datacopyflag, &
                                           localgridToFieldMap, &
                                           ungriddedLBound=ungriddedLBound, &
                                           ungriddedUBound=ungriddedUBound, &
                                           name=name, vm=vm, rc=localrc)
          if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
      endif

    endif ! - actualFlag

    ! Set return values.
    ESMF_FieldCreateGBArray%ftypep => ftype
    ESMF_INIT_SET_CREATED(ESMF_FieldCreateGBArray)

    if (actualFlag) then
      ! Add reference to this object into ESMF garbage collection table
      ! Only call this in those Create() methods that call Construct()
      call c_ESMC_VMAddFObject(ESMF_FieldCreateGBArray, ESMF_ID_FIELD%objectID)

      call ESMF_FieldValidate(ESMF_FieldCreateGBArray, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) then
        call ESMF_BaseSetStatus(ftype%base, ESMF_STATUS_INVALID, rc=localrc)
        return
      endif
    endif

    if(rcpresent) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateGBArray
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from Geom and Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateGBData<rank><type><kind>(geom, & 
! farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
! ungriddedUBound, totalLWidth, totalUWidth, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateGBData<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_Geom) :: geom 
! <type> (ESMF_KIND_<kind>), dimension(<rank>), target :: farray 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data array and {\tt ESMF\_Geom}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! The arguments are: 
! \begin{description} 
! \item [geom] 
! {\tt ESMF\_Geom} object. The dimCount of the 
! Geom must be smaller than or equal to the rank of the {\tt farray}. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item [indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farray} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The default is to 
! map all of the {\tt grid}'s dimensions against the lowest dimensions of 
! the {\tt farray} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farray} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farray} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt grid}. Ungridded dimensions must be in the same order they are 
! stored in the {\t farray}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farray}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt farray}. That is, for each gridded dimension the 
! {\tt farray} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farray}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt farray}. That is, for each gridded dimension the 
! {\tt farray} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData1Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData1Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData1Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData1Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData1Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData2Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData2Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData2Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData2Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData2Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData3Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData3Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData3Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData3Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData3Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData4Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData4Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData4Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData4Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData4Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData5Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData5Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData5Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData5Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData5Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData6Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData6Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData6Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData6Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData6Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData7Di1(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData7Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData7Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData7Di1, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData7Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData1Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData1Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData1Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData1Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData1Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData2Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData2Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData2Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData2Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData2Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData3Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData3Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData3Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData3Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData3Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData4Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData4Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData4Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData4Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData4Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData5Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData5Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData5Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData5Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData5Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData6Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData6Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData6Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData6Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData6Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData7Di2(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData7Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData7Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData7Di2, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData7Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData1Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData1Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData1Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData1Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData1Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData1Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData1Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData1Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData1Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData1Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData1Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData1Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData1Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData1Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData1Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData1Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData1Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData1Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData1Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData1Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData2Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData2Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData2Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData2Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData2Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData2Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData2Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData2Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData2Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData2Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData2Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData2Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData2Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData2Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData2Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData2Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData2Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData2Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData2Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData2Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData3Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData3Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData3Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData3Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData3Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData3Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData3Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData3Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData3Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData3Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData3Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData3Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData3Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData3Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData3Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData3Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData3Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData3Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData3Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData3Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData4Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData4Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData4Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData4Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData4Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData4Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData4Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData4Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData4Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData4Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData4Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData4Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData4Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData4Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData4Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData4Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData4Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData4Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData4Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData4Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData5Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData5Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData5Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData5Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData5Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData5Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData5Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData5Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData5Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData5Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData5Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData5Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData5Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData5Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData5Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData5Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData5Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData5Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData5Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData5Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData6Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData6Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData6Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData6Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData6Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData6Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData6Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData6Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData6Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData6Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData6Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData6Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData6Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData6Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData6Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData6Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData6Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData6Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData6Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData6Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData7Di4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData7Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData7Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData7Di4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData7Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData7Di8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData7Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData7Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData7Di8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData7Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData7Dr4(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData7Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData7Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData7Dr4, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData7Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBData" 
 function ESMF_FieldCreateGBData7Dr8(geom, & 
 farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBData7Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBData7Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBData7Dr8, & 
 geom, farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBData7Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBData7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!------------------------------------------------------------------------------ 



!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from Geom and Fortran array pointer 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateGBDataPtr<rank><type><kind>(geom, & 
! farrayPtr, datacopyflag, gridToFieldMap, & 
! totalLWidth, totalUWidth, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_Geom) :: geom 
! <type> (ESMF_KIND_<kind>), dimension(<rank>), pointer :: farrayPtr 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data pointer and {\tt ESMF\_Geom}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried and deallocated when 
! datacopyflag is {\tt ESMF\_DATACOPY\_REFERENCE}. Note that the {\tt ESMF\_FieldDestroy} call does not 
! deallocate the Fortran data pointer in this case. This gives user more flexibility over memory management. 
! 
! 
! The arguments are: 
! \begin{description} 
! \item [geom] 
! {\tt ESMF\_Geom} object. The dimCount of the 
! Geom must be smaller than or equal to the rank of the {\tt farrayPtr}. 
! \item [farrayPtr] 
! Native Fortran data pointer to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farrayPtr}. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farrayPtr} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. The default is to 
! map all of the {\tt grid}'s dimensions against the lowest dimensions of 
! the {\tt farrayPtr} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farrayPtr} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farrayPtr} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt grid}. Ungridded dimensions must be in the same order they are 
! stored in the {\t farrayPtr}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farrayPtr}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt farrayPtr}. That is, for each gridded dimension the 
! {\tt farrayPtr} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farrayPtr}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt farrayPtr}. That is, for each gridded dimension the 
! {\tt farrayPtr} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr1Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr1Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr1Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr1Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr1Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr2Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr2Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr2Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr2Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr2Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr3Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr3Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr3Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr3Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr3Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr4Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr4Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr4Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr4Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr4Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr5Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr5Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr5Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr5Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr5Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr6Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr6Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr6Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr6Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr6Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr7Di1(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr7Di1 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr7Di1 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr7Di1, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr7Di1, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr1Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr1Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr1Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr1Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr1Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr2Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr2Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr2Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr2Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr2Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr3Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr3Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr3Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr3Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr3Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr4Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr4Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr4Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr4Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr4Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr5Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr5Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr5Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr5Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr5Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr6Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr6Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr6Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr6Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr6Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr7Di2(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr7Di2 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr7Di2 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr7Di2, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr7Di2, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr1Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr1Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr1Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr1Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr1Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr1Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr1Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr1Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr1Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr1Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr1Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr1Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr1Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr1Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr1Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr1Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr1Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr1Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr1Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr1Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr2Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr2Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr2Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr2Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr2Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr2Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr2Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr2Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr2Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr2Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr2Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr2Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr2Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr2Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr2Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr2Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr2Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr2Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr2Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr2Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr3Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr3Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr3Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr3Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr3Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr3Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr3Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr3Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr3Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr3Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr3Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr3Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr3Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr3Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr3Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr3Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr3Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr3Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr3Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr3Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr4Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr4Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr4Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr4Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr4Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr4Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr4Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr4Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr4Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr4Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr4Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr4Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr4Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr4Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr4Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr4Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr4Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr4Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr4Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr4Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr5Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr5Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr5Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr5Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr5Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr5Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr5Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr5Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr5Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr5Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr5Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr5Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr5Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr5Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr5Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr5Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr5Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr5Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr5Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr5Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr6Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr6Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr6Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr6Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr6Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr6Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr6Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr6Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr6Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr6Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr6Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr6Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr6Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr6Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr6Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr6Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr6Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr6Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr6Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr6Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr7Di4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr7Di4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr7Di4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr7Di4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr7Di4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr7Di8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr7Di8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr7Di8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr7Di8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr7Di8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr7Dr4(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr7Dr4 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr7Dr4 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr7Dr4, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr7Dr4, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGBDataPtr" 
 function ESMF_FieldCreateGBDataPtr7Dr8(geom, & 
 farrayPtr, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGBDataPtr7Dr8 

 ! input arguments 
 type(ESMF_Geom) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ESMF_FieldCreateGBDataPtr7Dr8 = & 
 ESMF_FieldEmptyCreate(name=name, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldEmptyComplete( & 
 ESMF_FieldCreateGBDataPtr7Dr8, & 
 geom, farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 call ESMF_FieldValidate(ESMF_FieldCreateGBDataPtr7Dr8, & 
 rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGBDataPtr7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!------------------------------------------------------------------------------

! ESMF_FieldCreateGr
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateGridTKR"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Grid and typekind

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateGridTKR(grid, typekind, keywordEnforcer, &
    indexflag, staggerloc, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateGridTKR
!
! !ARGUMENTS:
    type(ESMF_Grid), intent(in) :: grid
    type(ESMF_TypeKind_Flag),intent(in) :: typekind
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Index_Flag), intent(in), optional :: indexflag
    type(ESMF_StaggerLoc), intent(in), optional :: staggerloc
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt pinflag} to provide access to DE sharing
! between PETs.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_tkr}.
!
! The arguments are:
! \begin{description}
! \item [grid]
! {\tt ESMF\_Grid} object.
! \item[typekind]
! The typekind of the Field. See section \ref{const:typekind}
! for a list of valid typekind options.
! \item[{[indexflag]}]
! Indicate how DE-local indices are defined. By default each DE's
! exclusive region is placed to start at the local index space origin,
! i.e. (1, 1, ..., 1). Alternatively the DE-local index space can be
! aligned with the global index space, if a global index space is well
! defined by the associated Grid. See section \ref{const:indexflag}
! for a list of valid indexflag options. The default indexflag value is the
! one stored in then {\tt ESMF\_Grid} object. Currently it is
! erroneous to specify an indexflag
! different from the one stored in the {\tt ESMF\_Grid} object. The default
! value is {\tt ESMF\_INDEX\_DELOCAL}
! \item [{[staggerloc]}]
! Stagger location of data in grid cells. For valid
! predefined values see section \ref{const:staggerloc}.
! To create a custom stagger location see section
! \ref{sec:usage:staggerloc:adv}. The default
! value is {\tt ESMF\_STAGGERLOC\_CENTER}.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Grid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Grid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_ArraySpec) :: arrayspec
    integer :: rank, ungridded_dimcount, ungridded_ldimcount, ungridded_udimcount
    integer :: memDimCount, grid_repdimcount, i
    character(320) :: msgString

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Compute rank
    ungridded_dimcount = 0
    if(present(ungriddedLBound)) then
      ungridded_ldimcount = size(ungriddedLBound)
      ungridded_dimcount = ungridded_ldimcount
    endif
    if(present(ungriddedUBound)) then
      ungridded_udimcount = size(ungriddedUBound)
      ungridded_dimcount = ungridded_udimcount
    endif

    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    call ESMF_GridGet(grid, rank=memDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    grid_repdimcount = 0
    if (present(gridToFieldMap)) then
       do i = 1, size(gridToFieldMap)
           if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
       enddo
    endif

    rank = ungridded_dimcount + memDimCount - grid_repdimcount
    ! Set ArraySpec from rank and typekind input
    call ESMF_ArraySpecSet(arrayspec, rank=rank, typekind=typekind, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ESMF_FieldCreateGridTKR = ESMF_FieldCreate(grid, arrayspec, &
      indexflag=indexflag, staggerloc=staggerloc, &
      gridToFieldMap=gridToFieldMap, ungriddedLBound=ungriddedLBound, &
      ungriddedUBound=ungriddedUBound, totalLWidth=totalLWidth, &
      totalUWidth=totalUWidth, pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

end function ESMF_FieldCreateGridTKR

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateGridArraySpec"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Grid and ArraySpec

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateGridArraySpec(grid, arrayspec, keywordEnforcer, &
    indexflag, staggerloc, gridToFieldMap, ungriddedLBound, &
    ungriddedUBound, totalLWidth, totalUWidth, pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateGridArraySpec
!
! !ARGUMENTS:
    type(ESMF_Grid), intent(in) :: grid
    type(ESMF_ArraySpec), intent(in) :: arrayspec
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Index_Flag), intent(in), optional :: indexflag
    type(ESMF_StaggerLoc), intent(in), optional :: staggerloc
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt pinflag} to provide access to DE sharing
! between PETs.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_arrayspec}.
!
! The arguments are:
! \begin{description}
! \item [grid]
! {\tt ESMF\_Grid} object.
! \item [arrayspec]
! Data type and kind specification.
! \item[{[indexflag]}]
! Indicate how DE-local indices are defined. By default each DE's
! exclusive region is placed to start at the local index space origin,
! i.e. (1, 1, ..., 1). Alternatively the DE-local index space can be
! aligned with the global index space, if a global index space is well
! defined by the associated Grid. See section \ref{const:indexflag}
! for a list of valid indexflag options. The default indexflag value is the
! one stored in then {\tt ESMF\_Grid} object. Currently it is
! erroneous to specify an indexflag
! different from the one stored in the {\tt ESMF\_Grid} object.The default
! value is {\tt ESMF\_INDEX\_DELOCAL}
! \item [{[staggerloc]}]
! Stagger location of data in grid cells. For valid
! predefined values see section \ref{const:staggerloc}.
! To create a custom stagger location see section
! \ref{sec:usage:staggerloc:adv}. The default
! value is {\tt ESMF\_STAGGERLOC\_CENTER}.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Grid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Grid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom
    type(ESMF_Index_Flag) :: g_indexflag
    type(ESMF_GridDecompType) :: decompType

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    call ESMF_GridGetDecompType(grid, decompType, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if (decompType .eq. ESMF_GRID_ARBITRARY) then
        if ((present(indexflag)) .or. (present(totalLWidth)) .or. (present(totalUWidth))) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
                 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", &
                 ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif

    call ESMF_GridGet(grid, indexFlag=g_indexflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if (present(indexflag)) then
      if(.not. (g_indexflag .eq. indexflag)) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
             msg="- user specified indexflag must be identical with Grid indexflag", &
               ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    ! Error check staggerloc
    if (present(staggerloc)) then
       if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. &
            (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
               msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    ! Create Geom from Grid
    geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateGridArraySpec=ESMF_FieldCreateGBArraySpec(geom, &
        arrayspec, gridToFieldMap, &
        ungriddedLBound, ungriddedUBound, &
        totalLWidth, totalUWidth, pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateGridArraySpec%ftypep%geomb_internal = .true.

    if(present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateGridArraySpec
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateGridArray"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Grid and Array

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateGridArray(grid, array, keywordEnforcer, datacopyflag, &
    staggerloc, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, name, vm, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateGridArray
!
! !ARGUMENTS:
    type(ESMF_Grid), intent(in) :: grid
    type(ESMF_Array), intent(in) :: array
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag
    type(ESMF_StaggerLoc), intent(in), optional :: staggerloc
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    character (len = *), intent(in), optional :: name
    type(ESMF_VM), intent(in), optional :: vm
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.0.0] Added argument {\tt vm} to support object creation on a
! different VM than that of the current context.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field}. This version of creation
! assumes the data exists already and is being
! passed in through an {\tt ESMF\_Array}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_array}.
!
! The arguments are:
! \begin{description}
! \item [grid]
! {\tt ESMF\_Grid} object.
! \item [array]
! {\tt ESMF\_Array} object.
! \item [{[datacopyflag]}]
! Indicates whether to copy the contents of the {\tt array} or reference it directly.
! For valid values see \ref{const:datacopyflag}. The default is
! {\tt ESMF\_DATACOPY\_REFERENCE}.
! \item [{[staggerloc]}]
! Stagger location of data in grid cells. For valid
! predefined values see section \ref{const:staggerloc}.
! To create a custom stagger location see section
! \ref{sec:usage:staggerloc:adv}. The default
! value is {\tt ESMF\_STAGGERLOC\_CENTER}.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Grid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Grid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[name]}]
! Field name.
! \item[{[vm]}]
! If present, the Field object is constructed on the specified
! {\tt ESMF\_VM} object. The default is to construct on the VM of the
! current component context.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom
    type(ESMF_GridDecompType) :: decompType
    type(ESMF_Pointer) :: vmThis
    logical :: actualFlag

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if (present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! Must make sure the local PET is associated with an actual member
    actualFlag = .true.
    if (present(vm)) then
      ESMF_INIT_CHECK_DEEP(ESMF_VMGetInit, vm, rc)
      call ESMF_VMGetThis(vm, vmThis)
      if (vmThis == ESMF_NULL_POINTER) then
        actualFlag = .false. ! local PET is not for an actual member
      endif
    endif

    if (actualFlag) then

      ! check init status of input parameters
      ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)
      ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,array,rc)

      call ESMF_GridGetDecompType(grid, decompType, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      if (decompType .eq. ESMF_GRID_ARBITRARY) then
          if ((present(totalLWidth)) .or. (present(totalUWidth))) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
                   msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", &
                   ESMF_CONTEXT, rcToReturn=rc)
             return
          endif
      endif

      ! Error check staggerloc
      if (present(staggerloc)) then
         if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. &
              (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
                 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", &
                 ESMF_CONTEXT, rcToReturn=rc)
            return
         endif
      endif

      ! Create Geom from Grid
      geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return

   endif ! - actualFlag

    ! Create Field from Geom
    ESMF_FieldCreateGridArray=ESMF_FieldCreateGBArray(geom, array, datacopyflag, &
        gridToFieldMap, ungriddedLBound, ungriddedUBound, totalLWidth, &
        totalUWidth, name=name, vm=vm, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if (actualFlag) then

      ! remember that geom was created internally
      ESMF_FieldCreateGridArray%ftypep%geomb_internal = .true.

    endif ! - actualFlag

    if (present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateGridArray
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from Grid and Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateGridData<rank><type><kind>(grid, & 
! farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
! gridToFieldMap, ungriddedLBound, ungriddedUBound, & 
! totalLWidth, totalUWidth, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateGridData<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_Grid), intent(in) :: grid 
! <type> (ESMF_KIND_<kind>),intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data array and {\tt ESMF\_Grid}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! For examples and 
! associated documentation regarding this method see section 
! \ref{sec:field:usage:create_2darray}, 
! \ref{sec:field:usage:create_2dgrid_3dptr}, 
! \ref{sec:field:usage:create_2dgrid_3dptr_map}, 
! \ref{sec:field:usage:create_2dgrid_3dptr_map_halo}, and 
! \ref{sec:field:usage:create_5dgrid_7dptr_2dungridded}. 
! 
! The arguments are: 
! \begin{description} 
! \item [grid] 
! {\tt ESMF\_Grid} object. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item[indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! Currently it is erroneous to specify an indexflag 
! different from the one stored in the {\tt ESMF\_Grid} object. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farray} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[staggerloc]}] 
! Stagger location of data in grid cells. For valid 
! predefined values see section \ref{const:staggerloc}. 
! To create a custom stagger location see section 
! \ref{sec:usage:staggerloc:adv}. The default 
! value is {\tt ESMF\_STAGGERLOC\_CENTER}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The default is to 
! map all of the {\tt grid}'s dimensions against the lowest dimensions of 
! the {\tt farray} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farray} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farray} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt grid}. Ungridded dimensions must be in the same order they are 
! stored in the {\t farray}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the Grid dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Grid dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farray}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt farray}. That is, for each gridded dimension the 
! {\tt farray} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farray}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt farray}. That is, for each gridded dimension the 
! {\tt farray} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData1Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData1Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData1Di1 = & 
 ESMF_FieldCreateGBData1Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData2Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData2Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData2Di1 = & 
 ESMF_FieldCreateGBData2Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData3Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData3Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData3Di1 = & 
 ESMF_FieldCreateGBData3Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData4Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData4Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData4Di1 = & 
 ESMF_FieldCreateGBData4Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData5Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData5Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData5Di1 = & 
 ESMF_FieldCreateGBData5Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData6Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData6Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData6Di1 = & 
 ESMF_FieldCreateGBData6Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData7Di1(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData7Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData7Di1 = & 
 ESMF_FieldCreateGBData7Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData1Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData1Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData1Di2 = & 
 ESMF_FieldCreateGBData1Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData2Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData2Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData2Di2 = & 
 ESMF_FieldCreateGBData2Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData3Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData3Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData3Di2 = & 
 ESMF_FieldCreateGBData3Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData4Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData4Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData4Di2 = & 
 ESMF_FieldCreateGBData4Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData5Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData5Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData5Di2 = & 
 ESMF_FieldCreateGBData5Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData6Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData6Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData6Di2 = & 
 ESMF_FieldCreateGBData6Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData7Di2(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData7Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData7Di2 = & 
 ESMF_FieldCreateGBData7Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData1Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData1Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData1Di4 = & 
 ESMF_FieldCreateGBData1Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData1Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData1Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData1Di8 = & 
 ESMF_FieldCreateGBData1Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData1Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData1Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData1Dr4 = & 
 ESMF_FieldCreateGBData1Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData1Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData1Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData1Dr8 = & 
 ESMF_FieldCreateGBData1Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData2Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData2Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData2Di4 = & 
 ESMF_FieldCreateGBData2Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData2Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData2Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData2Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData2Di8 = & 
 ESMF_FieldCreateGBData2Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData2Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData2Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData2Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData2Dr4 = & 
 ESMF_FieldCreateGBData2Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData2Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData2Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData2Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData2Dr8 = & 
 ESMF_FieldCreateGBData2Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData2Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData3Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData3Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData3Di4 = & 
 ESMF_FieldCreateGBData3Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData3Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData3Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData3Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData3Di8 = & 
 ESMF_FieldCreateGBData3Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData3Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData3Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData3Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData3Dr4 = & 
 ESMF_FieldCreateGBData3Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData3Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData3Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData3Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData3Dr8 = & 
 ESMF_FieldCreateGBData3Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData3Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData4Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData4Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData4Di4 = & 
 ESMF_FieldCreateGBData4Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData4Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData4Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData4Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData4Di8 = & 
 ESMF_FieldCreateGBData4Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData4Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData4Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData4Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData4Dr4 = & 
 ESMF_FieldCreateGBData4Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData4Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData4Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData4Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData4Dr8 = & 
 ESMF_FieldCreateGBData4Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData4Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData5Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData5Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData5Di4 = & 
 ESMF_FieldCreateGBData5Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData5Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData5Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData5Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData5Di8 = & 
 ESMF_FieldCreateGBData5Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData5Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData5Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData5Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData5Dr4 = & 
 ESMF_FieldCreateGBData5Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData5Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData5Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData5Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData5Dr8 = & 
 ESMF_FieldCreateGBData5Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData5Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData6Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData6Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData6Di4 = & 
 ESMF_FieldCreateGBData6Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData6Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData6Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData6Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData6Di8 = & 
 ESMF_FieldCreateGBData6Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData6Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData6Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData6Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData6Dr4 = & 
 ESMF_FieldCreateGBData6Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData6Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData6Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData6Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData6Dr8 = & 
 ESMF_FieldCreateGBData6Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData6Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData7Di4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData7Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData7Di4 = & 
 ESMF_FieldCreateGBData7Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData7Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData7Di8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData7Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData7Di8 = & 
 ESMF_FieldCreateGBData7Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData7Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData7Dr4(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData7Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData7Dr4 = & 
 ESMF_FieldCreateGBData7Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData7Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridData" 
 function ESMF_FieldCreateGridData7Dr8(grid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, staggerloc, & 
 gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridData7Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag),intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridData7Dr8 = & 
 ESMF_FieldCreateGBData7Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridData7Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridData7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from Grid and Fortran array pointer 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateGridDataPtr<rank><type><kind>(grid, & 
! farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
! totalLWidth, totalUWidth, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_Grid), intent(in) :: grid 
! <type> (ESMF_KIND_<kind>), pointer :: farrayPtr(<rank>) 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! type(ESMF_StaggerLoc), intent(in), optional :: staggerloc 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data pointer and {\tt ESMF\_Grid}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried and deallocated when 
! datacopyflag is {\tt ESMF\_DATACOPY\_REFERENCE}. Note that the {\tt ESMF\_FieldDestroy} call does not 
! deallocate the Fortran data pointer in this case. This gives user more flexibility over memory management. 
! 
! For examples and 
! associated documentation regarding this method see section 
! \ref{sec:field:usage:create_2dptr}, 
! \ref{sec:field:usage:create_2dgrid_3dptr}, 
! \ref{sec:field:usage:create_2dgrid_3dptr_map}, 
! \ref{sec:field:usage:create_2dgrid_3dptr_map_halo}, and 
! \ref{sec:field:usage:create_5dgrid_7dptr_2dungridded}. 
! 
! The arguments are: 
! \begin{description} 
! \item [grid] 
! {\tt ESMF\_Grid} object. 
! \item [farrayPtr] 
! Native Fortran data pointer to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farrayPtr}. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farrayPtr} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[staggerloc]}] 
! Stagger location of data in grid cells. For valid 
! predefined values see section \ref{const:staggerloc}. 
! To create a custom stagger location see section 
! \ref{sec:usage:staggerloc:adv}. The default 
! value is {\tt ESMF\_STAGGERLOC\_CENTER}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. The default is to 
! map all of the {\tt grid}'s dimensions against the lowest dimensions of 
! the {\tt farrayPtr} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farrayPtr} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farrayPtr} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt grid}. Ungridded dimensions must be in the same order they are 
! stored in the {\t farrayPtr}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the Grid dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Grid dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farrayPtr}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt farrayPtr}. That is, for each gridded dimension the 
! {\tt farrayPtr} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the Field. However, ordering of the elements 
! needs to be the same as they appear in the {\tt farrayPtr}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt farrayPtr}. That is, for each gridded dimension the 
! {\tt farrayPtr} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr1Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr1Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr1Di1 = & 
 ESMF_FieldCreateGBDataPtr1Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr2Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr2Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr2Di1 = & 
 ESMF_FieldCreateGBDataPtr2Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr3Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr3Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr3Di1 = & 
 ESMF_FieldCreateGBDataPtr3Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr4Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr4Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr4Di1 = & 
 ESMF_FieldCreateGBDataPtr4Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr5Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr5Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr5Di1 = & 
 ESMF_FieldCreateGBDataPtr5Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr6Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr6Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr6Di1 = & 
 ESMF_FieldCreateGBDataPtr6Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr7Di1(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr7Di1 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr7Di1 = & 
 ESMF_FieldCreateGBDataPtr7Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr1Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr1Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr1Di2 = & 
 ESMF_FieldCreateGBDataPtr1Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr2Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr2Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr2Di2 = & 
 ESMF_FieldCreateGBDataPtr2Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr3Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr3Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr3Di2 = & 
 ESMF_FieldCreateGBDataPtr3Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr4Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr4Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr4Di2 = & 
 ESMF_FieldCreateGBDataPtr4Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr5Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr5Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr5Di2 = & 
 ESMF_FieldCreateGBDataPtr5Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr6Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr6Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr6Di2 = & 
 ESMF_FieldCreateGBDataPtr6Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr7Di2(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr7Di2 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr7Di2 = & 
 ESMF_FieldCreateGBDataPtr7Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr1Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr1Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr1Di4 = & 
 ESMF_FieldCreateGBDataPtr1Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr1Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr1Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr1Di8 = & 
 ESMF_FieldCreateGBDataPtr1Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr1Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr1Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr1Dr4 = & 
 ESMF_FieldCreateGBDataPtr1Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr1Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr1Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr1Dr8 = & 
 ESMF_FieldCreateGBDataPtr1Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr2Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr2Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr2Di4 = & 
 ESMF_FieldCreateGBDataPtr2Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr2Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr2Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr2Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr2Di8 = & 
 ESMF_FieldCreateGBDataPtr2Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr2Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr2Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr2Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr2Dr4 = & 
 ESMF_FieldCreateGBDataPtr2Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr2Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr2Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr2Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr2Dr8 = & 
 ESMF_FieldCreateGBDataPtr2Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr2Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr3Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr3Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr3Di4 = & 
 ESMF_FieldCreateGBDataPtr3Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr3Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr3Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr3Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr3Di8 = & 
 ESMF_FieldCreateGBDataPtr3Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr3Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr3Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr3Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr3Dr4 = & 
 ESMF_FieldCreateGBDataPtr3Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr3Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr3Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr3Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr3Dr8 = & 
 ESMF_FieldCreateGBDataPtr3Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr3Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr4Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr4Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr4Di4 = & 
 ESMF_FieldCreateGBDataPtr4Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr4Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr4Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr4Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr4Di8 = & 
 ESMF_FieldCreateGBDataPtr4Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr4Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr4Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr4Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr4Dr4 = & 
 ESMF_FieldCreateGBDataPtr4Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr4Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr4Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr4Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr4Dr8 = & 
 ESMF_FieldCreateGBDataPtr4Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr4Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr5Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr5Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr5Di4 = & 
 ESMF_FieldCreateGBDataPtr5Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr5Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr5Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr5Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr5Di8 = & 
 ESMF_FieldCreateGBDataPtr5Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr5Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr5Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr5Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr5Dr4 = & 
 ESMF_FieldCreateGBDataPtr5Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr5Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr5Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr5Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr5Dr8 = & 
 ESMF_FieldCreateGBDataPtr5Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr5Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr6Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr6Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr6Di4 = & 
 ESMF_FieldCreateGBDataPtr6Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr6Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr6Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr6Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr6Di8 = & 
 ESMF_FieldCreateGBDataPtr6Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr6Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr6Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr6Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr6Dr4 = & 
 ESMF_FieldCreateGBDataPtr6Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr6Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr6Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr6Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr6Dr8 = & 
 ESMF_FieldCreateGBDataPtr6Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr6Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr7Di4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr7Di4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr7Di4 = & 
 ESMF_FieldCreateGBDataPtr7Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr7Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr7Di8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr7Di8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr7Di8 = & 
 ESMF_FieldCreateGBDataPtr7Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr7Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr7Dr4(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr7Dr4 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr7Dr4 = & 
 ESMF_FieldCreateGBDataPtr7Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr7Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateGridDataPtr" 
 function ESMF_FieldCreateGridDataPtr7Dr8(grid, & 
 farrayPtr, keywordEnforcer, datacopyflag, staggerloc, gridToFieldMap, & 
 totalLWidth, totalUWidth, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateGridDataPtr7Dr8 

 ! input arguments 
 type(ESMF_Grid), intent(in) :: grid 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_StaggerLoc), intent(in), optional ::staggerloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 type(ESMF_GridDecompType) :: decompType 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_GridGetInit,grid,rc)


 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if ((present(totalLWidth)) .or. (present(totalUWidth))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif
 ! Error check staggerloc 
 if (present(staggerloc)) then 
 if ((decompType .eq. ESMF_GRID_ARBITRARY) .and. & 
 (staggerloc .ne. ESMF_STAGGERLOC_CENTER)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- staggerloc has to be ESMF_STAGGERLOC_CENTER for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Grid
 geom=ESMF_GeomCreate(grid, staggerloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateGridDataPtr7Dr8 = & 
 ESMF_FieldCreateGBDataPtr7Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 totalLWidth=totalLWidth, totalUWidth=totalUWidth, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateGridDataPtr7Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateGridDataPtr7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!------------------------------------------------------------------------------

! ESMF_FieldCreateLo
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateLSTKR"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from LocStream and typekind

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateLSTKR(locstream, typekind, keywordEnforcer, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateLSTKR
!
! !ARGUMENTS:
    type(ESMF_LocStream), intent(in) :: locstream
    type(ESMF_TypeKind_Flag),intent(in) :: typekind
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_locs_tkr}.
!
! The arguments are:
! \begin{description}
! \item [locstream]
! {\tt ESMF\_LocStream} object.
! \item[typekind]
! The typekind of the Field. See section \ref{const:typekind}
! for a list of valid typekind options.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the LocStream dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! LocStream dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_ArraySpec) :: arrayspec
    integer :: rank, ungridded_dimcount, ungridded_ldimcount, ungridded_udimcount
    integer :: memDimCount

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Compute rank
    ungridded_dimcount = 0
    if(present(ungriddedLBound)) then
      ungridded_ldimcount = size(ungriddedLBound)
      ungridded_dimcount = ungridded_ldimcount
    endif
    if(present(ungriddedUBound)) then
      ungridded_udimcount = size(ungriddedUBound)
      ungridded_dimcount = ungridded_udimcount
    endif

    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    rank = ungridded_dimcount + 1

    ! Set ArraySpec from rank and typekind input
    call ESMF_ArraySpecSet(arrayspec, rank=rank, typekind=typekind, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ESMF_FieldCreateLSTKR = ESMF_FieldCreate(locstream, arrayspec, &
        gridToFieldMap=gridToFieldMap, &
        ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, &
        pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

end function ESMF_FieldCreateLSTKR
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateLSArraySpec"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from LocStream and ArraySpec

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateLSArraySpec(locstream, arrayspec, keywordEnforcer, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateLSArraySpec
!
! !ARGUMENTS:
    type(ESMF_LocStream), intent(in) :: locstream
    type(ESMF_ArraySpec), intent(in) :: arrayspec
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_locs_arrayspec}.
!
! The arguments are:
! \begin{description}
! \item [locstream]
! {\tt ESMF\_LocStream} object.
! \item [arrayspec]
! Data type and kind specification.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the LocStream dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! LocStream dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    ! Create Geom from Mesh
    geom=ESMF_GeomCreate(locstream, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateLSArraySpec=ESMF_FieldCreateGBArraySpec(geom, &
        arrayspec, gridToFieldMap, &
        ungriddedLBound, ungriddedUBound, &
        pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateLSArraySpec%ftypep%geomb_internal = .true.

    if(present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateLSArraySpec
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateLSArray"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from LocStream and Array

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateLSArray(locstream, array, &
    keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateLSArray
!
! !ARGUMENTS:
    type(ESMF_LocStream), intent(in) :: locstream
    type(ESMF_Array), intent(in) :: array
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    character (len = *), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field}. This version of creation
! assumes the data exists already and is being
! passed in through an {\tt ESMF\_Array}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_array}.
!
! The arguments are:
! \begin{description}
! \item [locstream]
! {\tt ESMF\_LocStream} object.
! \item [array]
! {\tt ESMF\_Array} object.
! \item [{[datacopyflag]}]
! Indicates whether to copy the contents of the {\tt array} or reference it directly.
! For valid values see \ref{const:datacopyflag}. The default is
! {\tt ESMF\_DATACOPY\_REFERENCE}.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the LocStream dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! LocStream dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if (present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)
    ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,array,rc)

    ! Create Geom from Mesh
    geom=ESMF_GeomCreate(locstream, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateLSArray=ESMF_FieldCreateGBArray(geom, array, datacopyflag, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
       ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateLSArray%ftypep%geomb_internal = .true.

    if (present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateLSArray
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from LocStream and Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateLSData<rank><type><kind>(locstream, farray, & 
! indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
! ungriddedUBound, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateLSData<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_LocStream), intent(in) :: locstream 
! <type> (ESMF_KIND_<kind>),intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! \begin{sloppypar} 
! Create an {\tt ESMF\_Field} from a Fortran data array and {\tt ESMF\_LocStream}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! \end{sloppypar} 
! 
! The arguments are: 
! \begin{description} 
! \item [locstream] 
! {\tt ESMF\_LocStream} object. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item[indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farray} or reference directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt locstream}'s dimCount. The list elements map each dimension 
! of the {\tt locstream} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The default is to 
! map all of the {\tt locstream}'s dimensions against the lowest dimensions of 
! the {\tt farray} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farray} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farray} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt locstream}. Unlocstreamded dimensions must be in the same order they are 
! stored in the {\t farray}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the LocStream dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! LocStream dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than locstream dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than locstream dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData1Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData1Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData1Di1 = & 
 ESMF_FieldCreateGBData1Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData2Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData2Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData2Di1 = & 
 ESMF_FieldCreateGBData2Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData3Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData3Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData3Di1 = & 
 ESMF_FieldCreateGBData3Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData4Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData4Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData4Di1 = & 
 ESMF_FieldCreateGBData4Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData5Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData5Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData5Di1 = & 
 ESMF_FieldCreateGBData5Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData6Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData6Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData6Di1 = & 
 ESMF_FieldCreateGBData6Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData7Di1(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData7Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData7Di1 = & 
 ESMF_FieldCreateGBData7Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData1Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData1Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData1Di2 = & 
 ESMF_FieldCreateGBData1Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData2Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData2Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData2Di2 = & 
 ESMF_FieldCreateGBData2Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData3Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData3Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData3Di2 = & 
 ESMF_FieldCreateGBData3Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData4Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData4Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData4Di2 = & 
 ESMF_FieldCreateGBData4Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData5Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData5Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData5Di2 = & 
 ESMF_FieldCreateGBData5Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData6Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData6Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData6Di2 = & 
 ESMF_FieldCreateGBData6Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData7Di2(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData7Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData7Di2 = & 
 ESMF_FieldCreateGBData7Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData1Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData1Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData1Di4 = & 
 ESMF_FieldCreateGBData1Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData1Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData1Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData1Di8 = & 
 ESMF_FieldCreateGBData1Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData1Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData1Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData1Dr4 = & 
 ESMF_FieldCreateGBData1Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData1Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData1Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData1Dr8 = & 
 ESMF_FieldCreateGBData1Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData2Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData2Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData2Di4 = & 
 ESMF_FieldCreateGBData2Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData2Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData2Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData2Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData2Di8 = & 
 ESMF_FieldCreateGBData2Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData2Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData2Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData2Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData2Dr4 = & 
 ESMF_FieldCreateGBData2Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData2Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData2Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData2Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData2Dr8 = & 
 ESMF_FieldCreateGBData2Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData2Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData3Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData3Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData3Di4 = & 
 ESMF_FieldCreateGBData3Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData3Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData3Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData3Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData3Di8 = & 
 ESMF_FieldCreateGBData3Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData3Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData3Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData3Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData3Dr4 = & 
 ESMF_FieldCreateGBData3Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData3Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData3Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData3Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData3Dr8 = & 
 ESMF_FieldCreateGBData3Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData3Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData4Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData4Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData4Di4 = & 
 ESMF_FieldCreateGBData4Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData4Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData4Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData4Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData4Di8 = & 
 ESMF_FieldCreateGBData4Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData4Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData4Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData4Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData4Dr4 = & 
 ESMF_FieldCreateGBData4Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData4Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData4Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData4Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData4Dr8 = & 
 ESMF_FieldCreateGBData4Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData4Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData5Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData5Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData5Di4 = & 
 ESMF_FieldCreateGBData5Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData5Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData5Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData5Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData5Di8 = & 
 ESMF_FieldCreateGBData5Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData5Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData5Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData5Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData5Dr4 = & 
 ESMF_FieldCreateGBData5Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData5Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData5Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData5Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData5Dr8 = & 
 ESMF_FieldCreateGBData5Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData5Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData6Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData6Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData6Di4 = & 
 ESMF_FieldCreateGBData6Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData6Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData6Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData6Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData6Di8 = & 
 ESMF_FieldCreateGBData6Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData6Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData6Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData6Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData6Dr4 = & 
 ESMF_FieldCreateGBData6Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData6Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData6Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData6Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData6Dr8 = & 
 ESMF_FieldCreateGBData6Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData6Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData7Di4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData7Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData7Di4 = & 
 ESMF_FieldCreateGBData7Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData7Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData7Di8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData7Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData7Di8 = & 
 ESMF_FieldCreateGBData7Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData7Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData7Dr4(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData7Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData7Dr4 = & 
 ESMF_FieldCreateGBData7Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData7Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSData" 
 function ESMF_FieldCreateLSData7Dr8(locstream, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSData7Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSData7Dr8 = & 
 ESMF_FieldCreateGBData7Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSData7Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSData7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from LocStream and Fortran array pointer 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateLSDataPtr<rank><type><kind>(locstream, & 
! keywordEnforcer, farrayPtr, datacopyflag, gridToFieldMap, & 
! name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_LocStream), intent(in) :: locstream 
! <type> (ESMF_KIND_<kind>),pointer :: farrayPtr(<rank>) 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! \begin{sloppypar} 
! Create an {\tt ESMF\_Field} from a Fortran data pointer and {\tt ESMF\_LocStream}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried and deallocated when 
! datacopyflag is {\tt ESMF\_DATACOPY\_REFERENCE}. Note that the {\tt ESMF\_FieldDestroy} call does not 
! deallocate the Fortran data pointer in this case. This gives user more flexibility over memory management. 
! \end{sloppypar} 
! 
! The arguments are: 
! \begin{description} 
! \item [locstream] 
! {\tt ESMF\_LocStream} object. 
! \item [farrayPtr] 
! Native Fortran data pointer to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farrayPtr}. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farrayPtr} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt locstream}'s dimCount. The list elements map each dimension 
! of the {\tt locstream} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. The default is to 
! map all of the {\tt locstream}'s dimensions against the lowest dimensions of 
! the {\tt farrayPtr} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farrayPtr} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farrayPtr} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt locstream}. Unlocstreamded dimensions must be in the same order they are 
! stored in the {\t farrayPtr}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the LocStream dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! LocStream dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr1Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr1Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr1Di1 = & 
 ESMF_FieldCreateGBDataPtr1Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr2Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr2Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr2Di1 = & 
 ESMF_FieldCreateGBDataPtr2Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr3Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr3Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr3Di1 = & 
 ESMF_FieldCreateGBDataPtr3Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr4Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr4Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr4Di1 = & 
 ESMF_FieldCreateGBDataPtr4Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr5Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr5Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr5Di1 = & 
 ESMF_FieldCreateGBDataPtr5Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr6Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr6Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr6Di1 = & 
 ESMF_FieldCreateGBDataPtr6Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr7Di1(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr7Di1 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr7Di1 = & 
 ESMF_FieldCreateGBDataPtr7Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr1Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr1Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr1Di2 = & 
 ESMF_FieldCreateGBDataPtr1Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr2Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr2Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr2Di2 = & 
 ESMF_FieldCreateGBDataPtr2Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr3Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr3Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr3Di2 = & 
 ESMF_FieldCreateGBDataPtr3Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr4Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr4Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr4Di2 = & 
 ESMF_FieldCreateGBDataPtr4Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr5Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr5Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr5Di2 = & 
 ESMF_FieldCreateGBDataPtr5Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr6Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr6Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr6Di2 = & 
 ESMF_FieldCreateGBDataPtr6Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr7Di2(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr7Di2 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr7Di2 = & 
 ESMF_FieldCreateGBDataPtr7Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr1Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr1Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr1Di4 = & 
 ESMF_FieldCreateGBDataPtr1Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr1Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr1Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr1Di8 = & 
 ESMF_FieldCreateGBDataPtr1Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr1Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr1Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr1Dr4 = & 
 ESMF_FieldCreateGBDataPtr1Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr1Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr1Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr1Dr8 = & 
 ESMF_FieldCreateGBDataPtr1Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr2Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr2Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr2Di4 = & 
 ESMF_FieldCreateGBDataPtr2Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr2Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr2Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr2Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr2Di8 = & 
 ESMF_FieldCreateGBDataPtr2Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr2Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr2Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr2Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr2Dr4 = & 
 ESMF_FieldCreateGBDataPtr2Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr2Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr2Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr2Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr2Dr8 = & 
 ESMF_FieldCreateGBDataPtr2Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr2Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr3Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr3Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr3Di4 = & 
 ESMF_FieldCreateGBDataPtr3Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr3Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr3Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr3Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr3Di8 = & 
 ESMF_FieldCreateGBDataPtr3Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr3Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr3Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr3Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr3Dr4 = & 
 ESMF_FieldCreateGBDataPtr3Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr3Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr3Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr3Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr3Dr8 = & 
 ESMF_FieldCreateGBDataPtr3Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr3Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr4Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr4Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr4Di4 = & 
 ESMF_FieldCreateGBDataPtr4Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr4Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr4Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr4Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr4Di8 = & 
 ESMF_FieldCreateGBDataPtr4Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr4Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr4Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr4Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr4Dr4 = & 
 ESMF_FieldCreateGBDataPtr4Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr4Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr4Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr4Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr4Dr8 = & 
 ESMF_FieldCreateGBDataPtr4Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr4Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr5Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr5Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr5Di4 = & 
 ESMF_FieldCreateGBDataPtr5Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr5Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr5Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr5Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr5Di8 = & 
 ESMF_FieldCreateGBDataPtr5Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr5Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr5Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr5Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr5Dr4 = & 
 ESMF_FieldCreateGBDataPtr5Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr5Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr5Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr5Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr5Dr8 = & 
 ESMF_FieldCreateGBDataPtr5Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr5Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr6Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr6Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr6Di4 = & 
 ESMF_FieldCreateGBDataPtr6Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr6Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr6Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr6Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr6Di8 = & 
 ESMF_FieldCreateGBDataPtr6Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr6Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr6Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr6Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr6Dr4 = & 
 ESMF_FieldCreateGBDataPtr6Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr6Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr6Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr6Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr6Dr8 = & 
 ESMF_FieldCreateGBDataPtr6Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr6Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr7Di4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr7Di4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr7Di4 = & 
 ESMF_FieldCreateGBDataPtr7Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr7Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr7Di8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr7Di8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr7Di8 = & 
 ESMF_FieldCreateGBDataPtr7Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr7Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr7Dr4(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr7Dr4 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr7Dr4 = & 
 ESMF_FieldCreateGBDataPtr7Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr7Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateLSDataPtr" 
 function ESMF_FieldCreateLSDataPtr7Dr8(locstream, & 
 farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateLSDataPtr7Dr8 

 ! input arguments 
 type(ESMF_LocStream), intent(in) :: locstream 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_LocStreamGetInit,locstream,rc)


 ! Create Geom from LocStream
 geom=ESMF_GeomCreate(locstream,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateLSDataPtr7Dr8 = & 
 ESMF_FieldCreateGBDataPtr7Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateLSDataPtr7Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateLSDataPtr7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!------------------------------------------------------------------------------

! ESMF_FieldCreateMe
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateMeshTKR"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Mesh and typekind

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateMeshTKR(mesh, typekind, keywordEnforcer, indexflag, &
    meshloc, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateMeshTKR
!
! !ARGUMENTS:
    type(ESMF_Mesh), intent(in) :: mesh
    type(ESMF_TypeKind_Flag), intent(in) :: typekind
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Index_Flag), intent(in), optional :: indexflag
    type(ESMF_MeshLoc), intent(in), optional :: meshloc
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_mesh_tkr}.
!
! The arguments are:
! \begin{description}
! \item [mesh]
! {\tt ESMF\_Mesh} object.
! \item [typekind]
! The typekind of the Field. See section \ref{const:typekind}
! for a list of valid typekind options.
! \item [{[indexflag]}]
! Indicate how DE-local indices are defined. See section
! \ref{const:indexflag} for a list of valid indexflag options.
! \item [{[meshloc]}]
! \begin{sloppypar}
! The part of the Mesh on which to build the Field. For valid
! predefined values see Section~\ref{const:meshloc}.
! If not set, defaults to {\tt ESMF\_MESHLOC\_NODE}.
! \end{sloppypar}
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Mesh dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Mesh dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_ArraySpec) :: arrayspec
    integer :: rank, ungridded_dimcount, ungridded_ldimcount, ungridded_udimcount
    integer :: memDimCount
    type(ESMF_MeshLoc) :: localMeshloc
    type(ESMF_Distgrid) :: localDistgrid
    type(ESMF_Index_Flag) :: l_indexflag

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Compute rank
    ungridded_dimcount = 0
    if(present(ungriddedLBound)) then
      ungridded_ldimcount = size(ungriddedLBound)
      ungridded_dimcount = ungridded_ldimcount
    endif
    if(present(ungriddedUBound)) then
      ungridded_udimcount = size(ungriddedUBound)
      ungridded_dimcount = ungridded_udimcount
    endif

    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    ! Get DimCount from distgrid
    localMeshloc=ESMF_MESHLOC_NODE
    if (present(meshloc)) then
       localMeshloc=meshloc
    endif

    ! Get distgrid
    if (localMeshloc == ESMF_MESHLOC_NODE) then
       call ESMF_MeshGet(mesh, &
            nodalDistgrid=localDistgrid, &
            rc=localrc)
       if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
    else if (localmeshloc == ESMF_MESHLOC_ELEMENT) then
       call ESMF_MeshGet(mesh, &
            elementDistgrid=localDistgrid, &
            rc=localrc)
       if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
    else
       if (ESMF_LogFoundError(ESMF_RC_ARG_VALUE, &
                  msg=" Bad Mesh Location value", &
                  ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Get rank
    call ESMF_DistGridGet(localDistgrid, &
         dimCount=rank, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    ! Add in ungridded dimensions
    rank = rank + ungridded_dimcount

    ! Set ArraySpec from rank and typekind input
    call ESMF_ArraySpecSet(arrayspec, rank=rank, typekind=typekind, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ESMF_FieldCreateMeshTKR = ESMF_FieldCreate(mesh, arrayspec, &
      indexflag=indexflag, meshloc=meshloc, gridToFieldMap=gridToFieldMap, &
      ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, &
      pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

end function ESMF_FieldCreateMeshTKR
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateMeshArraySpec"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Mesh and ArraySpec

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateMeshArraySpec(mesh, arrayspec, keywordEnforcer, &
    indexflag, meshloc, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateMeshArraySpec
!
! !ARGUMENTS:
    type(ESMF_Mesh), intent(in) :: mesh
    type(ESMF_ArraySpec), intent(in) :: arrayspec
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_Index_Flag),intent(in), optional :: indexflag
    type(ESMF_MeshLoc), intent(in), optional :: meshloc
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_mesh_arrayspec} and
! \ref{sec:field:usage:createMeshArrayspecOpt}.
!
! The arguments are:
! \begin{description}
! \item [mesh]
! {\tt ESMF\_Mesh} object.
! \item [arrayspec]
! Data type and kind specification.
! \item [{[indexflag]}]
! Indicate how DE-local indices are defined. See section
! \ref{const:indexflag} for a list of valid indexflag options.
! \item [{[meshloc]}]
! \begin{sloppypar}
! The part of the Mesh on which to build the Field. For valid
! predefined values see Section~\ref{const:meshloc}.
! If not set, defaults to {\tt ESMF\_MESHLOC\_NODE}.
! \end{sloppypar}
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Mesh dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Mesh dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    ! Create Geom from Mesh
    geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateMeshArraySpec=ESMF_FieldCreateGBArraySpec(geom, &
        arrayspec, gridToFieldMap, &
        ungriddedLBound, ungriddedUBound, &
        pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateMeshArraySpec%ftypep%geomb_internal = .true.

    if(present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateMeshArraySpec
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateMeshArray"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from Mesh and Array

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateMeshArray(mesh, array, &
    keywordEnforcer, datacopyflag, meshloc, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    name, vm, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateMeshArray
!
! !ARGUMENTS:
    type(ESMF_Mesh), intent(in) :: mesh
    type(ESMF_Array), intent(in) :: array
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag
    type(ESMF_MeshLoc), intent(in), optional :: meshloc
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    character (len = *), intent(in), optional :: name
    type(ESMF_VM), intent(in), optional :: vm
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field}. This version of creation
! assumes the data exists already and is being
! passed in through an {\tt ESMF\_Array}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_mesh_array}.
!
! The arguments are:
! \begin{description}
! \item [mesh]
! {\tt ESMF\_Mesh} object.
! \item [array]
! {\tt ESMF\_Array} object.
! \item [{[datacopyflag]}]
! Indicates whether to copy the contents of the {\tt array} or reference it directly.
! For valid values see \ref{const:datacopyflag}. The default is
! {\tt ESMF\_DATACOPY\_REFERENCE}.
! \item [{[meshloc]}]
! \begin{sloppypar}
! The part of the Mesh on which to build the Field. For valid
! predefined values see Section~\ref{const:meshloc}.
! If not set, defaults to {\tt ESMF\_MESHLOC\_NODE}.
! \end{sloppypar}
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Mesh dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Mesh dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[name]}]
! Field name.
! \item[{[vm]}]
! If present, the Field object is constructed on the specified
! {\tt ESMF\_VM} object. The default is to construct on the VM of the
! current component context.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if (present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)
    ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,array,rc)

    ! Create Geom from Mesh
    geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateMeshArray=ESMF_FieldCreateGBArray(geom, array, &
      datacopyflag, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
      name=name, vm=vm, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateMeshArray%ftypep%geomb_internal = .true.

    if (present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateMeshArray
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from Mesh and Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateMeshData<rank><type><kind>(mesh, & 
! farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
! gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateMeshData<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_Mesh), intent(in) :: mesh 
! <type> (ESMF_KIND_<kind>),intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! type(ESMF_MeshLoc), intent(in), optional :: meshloc 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data array and {\tt ESMF\_Mesh}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! 
! The arguments are: 
! \begin{description} 
! \item [mesh] 
! {\tt ESMF\_Mesh} object. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item[indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farray} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[meshloc]}] 
! \begin{sloppypar} 
! The part of the Mesh on which to build the Field. For valid 
! predefined values see Section~\ref{const:meshloc}. 
! If not set, defaults to {\tt ESMF\_MESHLOC\_NODE}. 
! \end{sloppypar} 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt mesh}'s dimCount. The list elements map each dimension 
! of the {\tt mesh} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The default is to 
! map all of the {\tt mesh}'s dimensions against the lowest dimensions of 
! the {\tt farray} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farray} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farray} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt mesh}. Unmeshded dimensions must be in the same order they are 
! stored in the {\t farray}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the Mesh dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Mesh dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than mesh dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than mesh dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData1Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData1Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData1Di1 = & 
 ESMF_FieldCreateGBData1Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData2Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData2Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData2Di1 = & 
 ESMF_FieldCreateGBData2Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData3Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData3Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData3Di1 = & 
 ESMF_FieldCreateGBData3Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData4Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData4Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData4Di1 = & 
 ESMF_FieldCreateGBData4Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData5Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData5Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData5Di1 = & 
 ESMF_FieldCreateGBData5Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData6Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData6Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData6Di1 = & 
 ESMF_FieldCreateGBData6Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData7Di1(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData7Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData7Di1 = & 
 ESMF_FieldCreateGBData7Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData1Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData1Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData1Di2 = & 
 ESMF_FieldCreateGBData1Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData2Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData2Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData2Di2 = & 
 ESMF_FieldCreateGBData2Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData3Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData3Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData3Di2 = & 
 ESMF_FieldCreateGBData3Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData4Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData4Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData4Di2 = & 
 ESMF_FieldCreateGBData4Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData5Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData5Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData5Di2 = & 
 ESMF_FieldCreateGBData5Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData6Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData6Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData6Di2 = & 
 ESMF_FieldCreateGBData6Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData7Di2(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData7Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData7Di2 = & 
 ESMF_FieldCreateGBData7Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData1Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData1Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData1Di4 = & 
 ESMF_FieldCreateGBData1Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData1Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData1Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData1Di8 = & 
 ESMF_FieldCreateGBData1Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData1Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData1Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData1Dr4 = & 
 ESMF_FieldCreateGBData1Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData1Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData1Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData1Dr8 = & 
 ESMF_FieldCreateGBData1Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData2Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData2Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData2Di4 = & 
 ESMF_FieldCreateGBData2Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData2Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData2Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData2Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData2Di8 = & 
 ESMF_FieldCreateGBData2Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData2Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData2Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData2Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData2Dr4 = & 
 ESMF_FieldCreateGBData2Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData2Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData2Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData2Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData2Dr8 = & 
 ESMF_FieldCreateGBData2Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData2Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData3Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData3Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData3Di4 = & 
 ESMF_FieldCreateGBData3Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData3Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData3Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData3Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData3Di8 = & 
 ESMF_FieldCreateGBData3Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData3Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData3Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData3Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData3Dr4 = & 
 ESMF_FieldCreateGBData3Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData3Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData3Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData3Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData3Dr8 = & 
 ESMF_FieldCreateGBData3Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData3Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData4Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData4Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData4Di4 = & 
 ESMF_FieldCreateGBData4Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData4Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData4Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData4Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData4Di8 = & 
 ESMF_FieldCreateGBData4Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData4Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData4Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData4Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData4Dr4 = & 
 ESMF_FieldCreateGBData4Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData4Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData4Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData4Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData4Dr8 = & 
 ESMF_FieldCreateGBData4Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData4Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData5Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData5Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData5Di4 = & 
 ESMF_FieldCreateGBData5Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData5Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData5Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData5Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData5Di8 = & 
 ESMF_FieldCreateGBData5Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData5Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData5Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData5Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData5Dr4 = & 
 ESMF_FieldCreateGBData5Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData5Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData5Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData5Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData5Dr8 = & 
 ESMF_FieldCreateGBData5Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData5Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData6Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData6Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData6Di4 = & 
 ESMF_FieldCreateGBData6Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData6Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData6Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData6Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData6Di8 = & 
 ESMF_FieldCreateGBData6Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData6Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData6Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData6Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData6Dr4 = & 
 ESMF_FieldCreateGBData6Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData6Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData6Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData6Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData6Dr8 = & 
 ESMF_FieldCreateGBData6Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData6Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData7Di4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData7Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData7Di4 = & 
 ESMF_FieldCreateGBData7Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData7Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData7Di8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData7Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData7Di8 = & 
 ESMF_FieldCreateGBData7Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData7Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData7Dr4(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData7Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData7Dr4 = & 
 ESMF_FieldCreateGBData7Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData7Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshData" 
 function ESMF_FieldCreateMeshData7Dr8(mesh, & 
 farray, indexflag, keywordEnforcer, datacopyflag, meshloc, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshData7Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshData7Dr8 = & 
 ESMF_FieldCreateGBData7Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshData7Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshData7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from Mesh and Fortran array pointer 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateMeshDataPtr<rank><type><kind>(mesh, & 
! farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
! name, rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_Mesh), intent(in) :: mesh 
! <type> (ESMF_KIND_<kind>),pointer :: farrayPtr(<rank>) 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! type(ESMF_MeshLoc), intent(in), optional :: meshloc 
! integer, intent(in), optional :: gridToFieldMap(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data pointer and {\tt ESMF\_Mesh}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried and deallocated when 
! datacopyflag is {\tt ESMF\_DATACOPY\_REFERENCE}. Note that the {\tt ESMF\_FieldDestroy} call does not 
! deallocate the Fortran data pointer in this case. This gives user more flexibility over memory management. 
! 
! The arguments are: 
! \begin{description} 
! \item [mesh] 
! {\tt ESMF\_Mesh} object. 
! \item [farrayPtr] 
! Native Fortran data pointer to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farrayPtr}. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farrayPtr} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[meshloc]}] 
! \begin{sloppypar} 
! The part of the Mesh on which to build the Field. For valid 
! predefined values see Section~\ref{const:meshloc}. 
! If not set, defaults to {\tt ESMF\_MESHLOC\_NODE}. 
! \end{sloppypar} 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt mesh}'s dimCount. The list elements map each dimension 
! of the {\tt mesh} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. The default is to 
! map all of the {\tt mesh}'s dimensions against the lowest dimensions of 
! the {\tt farrayPtr} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farrayPtr} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farrayPtr} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt mesh}. Unmeshded dimensions must be in the same order they are 
! stored in the {\t farrayPtr}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the Mesh dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Mesh dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr1Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr1Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr1Di1 = & 
 ESMF_FieldCreateGBDataPtr1Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr2Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr2Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr2Di1 = & 
 ESMF_FieldCreateGBDataPtr2Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr3Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr3Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr3Di1 = & 
 ESMF_FieldCreateGBDataPtr3Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr4Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr4Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr4Di1 = & 
 ESMF_FieldCreateGBDataPtr4Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr5Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr5Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr5Di1 = & 
 ESMF_FieldCreateGBDataPtr5Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr6Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr6Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr6Di1 = & 
 ESMF_FieldCreateGBDataPtr6Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr7Di1(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr7Di1 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr7Di1 = & 
 ESMF_FieldCreateGBDataPtr7Di1(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr1Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr1Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr1Di2 = & 
 ESMF_FieldCreateGBDataPtr1Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr2Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr2Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr2Di2 = & 
 ESMF_FieldCreateGBDataPtr2Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr3Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr3Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr3Di2 = & 
 ESMF_FieldCreateGBDataPtr3Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr4Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr4Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr4Di2 = & 
 ESMF_FieldCreateGBDataPtr4Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr5Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr5Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr5Di2 = & 
 ESMF_FieldCreateGBDataPtr5Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr6Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr6Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr6Di2 = & 
 ESMF_FieldCreateGBDataPtr6Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr7Di2(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr7Di2 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr7Di2 = & 
 ESMF_FieldCreateGBDataPtr7Di2(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr1Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr1Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr1Di4 = & 
 ESMF_FieldCreateGBDataPtr1Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr1Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr1Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr1Di8 = & 
 ESMF_FieldCreateGBDataPtr1Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr1Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr1Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr1Dr4 = & 
 ESMF_FieldCreateGBDataPtr1Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr1Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr1Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr1Dr8 = & 
 ESMF_FieldCreateGBDataPtr1Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr2Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr2Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr2Di4 = & 
 ESMF_FieldCreateGBDataPtr2Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr2Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr2Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr2Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr2Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr2Di8 = & 
 ESMF_FieldCreateGBDataPtr2Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr2Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr2Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr2Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr2Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr2Dr4 = & 
 ESMF_FieldCreateGBDataPtr2Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr2Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr2Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr2Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr2Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr2Dr8 = & 
 ESMF_FieldCreateGBDataPtr2Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr2Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr2Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr3Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr3Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr3Di4 = & 
 ESMF_FieldCreateGBDataPtr3Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr3Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr3Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr3Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr3Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr3Di8 = & 
 ESMF_FieldCreateGBDataPtr3Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr3Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr3Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr3Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr3Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr3Dr4 = & 
 ESMF_FieldCreateGBDataPtr3Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr3Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr3Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr3Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr3Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr3Dr8 = & 
 ESMF_FieldCreateGBDataPtr3Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr3Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr3Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr4Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr4Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr4Di4 = & 
 ESMF_FieldCreateGBDataPtr4Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr4Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr4Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr4Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr4Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr4Di8 = & 
 ESMF_FieldCreateGBDataPtr4Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr4Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr4Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr4Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr4Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr4Dr4 = & 
 ESMF_FieldCreateGBDataPtr4Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr4Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr4Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr4Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr4Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr4Dr8 = & 
 ESMF_FieldCreateGBDataPtr4Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr4Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr4Dr8 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr5Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr5Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr5Di4 = & 
 ESMF_FieldCreateGBDataPtr5Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr5Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr5Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr5Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr5Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr5Di8 = & 
 ESMF_FieldCreateGBDataPtr5Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr5Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr5Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr5Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr5Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr5Dr4 = & 
 ESMF_FieldCreateGBDataPtr5Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr5Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr5Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr5Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr5Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr5Dr8 = & 
 ESMF_FieldCreateGBDataPtr5Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr5Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr5Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr6Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr6Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr6Di4 = & 
 ESMF_FieldCreateGBDataPtr6Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr6Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr6Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr6Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr6Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr6Di8 = & 
 ESMF_FieldCreateGBDataPtr6Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr6Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr6Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr6Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr6Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr6Dr4 = & 
 ESMF_FieldCreateGBDataPtr6Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr6Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr6Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr6Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr6Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr6Dr8 = & 
 ESMF_FieldCreateGBDataPtr6Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr6Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr6Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr7Di4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr7Di4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr7Di4 = & 
 ESMF_FieldCreateGBDataPtr7Di4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr7Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr7Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr7Di8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr7Di8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr7Di8 = & 
 ESMF_FieldCreateGBDataPtr7Di8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr7Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr7Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr7Dr4(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr7Dr4 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr7Dr4 = & 
 ESMF_FieldCreateGBDataPtr7Dr4(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr7Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr7Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateMeshDataPtr" 
 function ESMF_FieldCreateMeshDataPtr7Dr8(mesh, & 
 farrayPtr, keywordEnforcer, datacopyflag, meshloc, gridToFieldMap, & 
 name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateMeshDataPtr7Dr8 

 ! input arguments 
 type(ESMF_Mesh), intent(in) :: mesh 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_MeshLoc), intent(in), optional :: meshloc 
 integer, intent(in), optional :: gridToFieldMap(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_MeshGetInit,mesh,rc)


 ! Create Geom from Mesh
 geom=ESMF_GeomCreate(mesh, meshLoc=meshloc, rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateMeshDataPtr7Dr8 = & 
 ESMF_FieldCreateGBDataPtr7Dr8(geom, & 
 farrayPtr, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateMeshDataPtr7Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateMeshDataPtr7Dr8 
!------------------------------------------------------------------------------ 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


!------------------------------------------------------------------------------


! ESMF_FieldCreateXGrid
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateXGTKR"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from XGrid and typekind

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateXGTKR(xgrid, typekind, keywordEnforcer, xgridside, &
    gridindex, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateXGTKR
!
! !ARGUMENTS:
    type(ESMF_XGrid), intent(in) :: xgrid
    type(ESMF_TypeKind_Flag), intent(in) :: typekind
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside
    integer, intent(in), optional :: gridindex
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_locs_tkr}.
!
! The arguments are:
! \begin{description}
! \item [xgrid]
! {\tt ESMF\_XGrid} object.
! \item[typekind]
! The typekind of the Field. See section \ref{const:typekind}
! for a list of valid typekind options.
! \item [{[xgridside]}]
! Which side of the XGrid to create the Field on (either ESMF\_XGRIDSIDE\_A,
! ESMF\_XGRIDSIDE\_B, or ESMF\_XGRIDSIDE\_BALANCED). If not passed in then
! defaults to ESMF\_XGRIDSIDE\_BALANCED.
! \item [{[gridindex]}]
! If xgridSide is ESMF\_XGRIDSIDE\_A or ESMF\_XGRIDSIDE\_B then this index tells which Grid on
! that side to create the Field on. If not provided, defaults to 1.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the XGrid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! XGrid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_ArraySpec) :: arrayspec
    type(ESMF_XGridGeomBase) :: xgeom
    integer :: rank, ungridded_dimcount, ungridded_ldimcount, ungridded_udimcount

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Compute rank
    if(.not. present(xgridside)) then
      ungridded_dimcount = 0
      if(present(ungriddedLBound)) then
        ungridded_ldimcount = size(ungriddedLBound)
        ungridded_dimcount = ungridded_ldimcount
      endif
      if(present(ungriddedUBound)) then
        ungridded_udimcount = size(ungriddedUBound)
        ungridded_dimcount = ungridded_udimcount
      endif

      rank = ungridded_dimcount + 1
    else
      if(xgridside==ESMF_XGRIDSIDE_BALANCED) then
        ungridded_dimcount = 0
        if(present(ungriddedLBound)) then
          ungridded_ldimcount = size(ungriddedLBound)
          ungridded_dimcount = ungridded_ldimcount
        endif
        if(present(ungriddedUBound)) then
          ungridded_udimcount = size(ungriddedUBound)
          ungridded_dimcount = ungridded_udimcount
        endif

        rank = ungridded_dimcount + 1
      else
        call ESMF_XGridGet(xgrid, xgeom, &
          xgridside=xgridside, gridindex=gridindex, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        call ESMF_XGridGeomBaseGet(xgeom, dimCount=rank, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
      endif
    endif

    ! Set ArraySpec from rank and typekind input
    call ESMF_ArraySpecSet(arrayspec, rank=rank, typekind=typekind, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ESMF_FieldCreateXGTKR = ESMF_FieldCreate(xgrid, arrayspec, &
        xgridside=xgridside, gridindex=gridIndex, &
        gridToFieldMap=gridToFieldMap, ungriddedLBound=ungriddedLBound, &
        ungriddedUBound=ungriddedUBound, pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

end function ESMF_FieldCreateXGTKR
!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateXGArraySpec"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from XGrid and ArraySpec

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateXGArraySpec(xgrid, arrayspec, keywordEnforcer, &
    xgridside, gridindex, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    pinflag, name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateXGArraySpec
!
! !ARGUMENTS:
    type(ESMF_XGrid), intent(in) :: xgrid
    type(ESMF_ArraySpec), intent(in) :: arrayspec
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_XGridSide_Flag), intent(in), optional :: xgridSide
    integer, intent(in), optional :: gridIndex
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    character (len=*), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}. Return a new {\tt ESMF\_Field}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_locs_arrayspec}.
!
! The arguments are:
! \begin{description}
! \item [xgrid]
! {\tt ESMF\_XGrid} object.
! \item [arrayspec]
! Data type and kind specification.
! \item [{[xgridside]}]
! Which side of the XGrid to create the Field on (either ESMF\_XGRIDSIDE\_A,
! ESMF\_XGRIDSIDE\_B, or ESMF\_XGRIDSIDE\_BALANCED). If not passed in then
! defaults to ESMF\_XGRIDSIDE\_BALANCED.
! \item [{[gridindex]}]
! If xgridside is ESMF\_XGRIDSIDE\_A or ESMF\_XGRIDSIDE\_B then this index tells which Grid on
! that side to create the Field on. If not provided, defaults to 1.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the XGrid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! XGrid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom
    integer :: ungridded_ldimcount, ungridded_udimcount

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    ! Sanity checking
    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(present(ungriddedLBound)) then
        ungridded_ldimcount = size(ungriddedLBound)
      endif
      if(present(ungriddedUBound)) then
        ungridded_udimcount = size(ungriddedUBound)
      endif
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    ! Create Geom from Xgrid
    geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateXGArraySpec=ESMF_FieldCreateGBArraySpec(geom, &
        arrayspec, gridToFieldMap, &
        ungriddedLBound, ungriddedUBound, &
        pinflag=pinflag, name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateXGArraySpec%ftypep%geomb_internal = .true.

    if(present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateXGArraySpec
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldCreateXGArray"
!BOP
! !IROUTINE: ESMF_FieldCreate - Create a Field from XGrid and Array

! !INTERFACE:
  ! Private name; call using ESMF_FieldCreate()
  function ESMF_FieldCreateXGArray(xgrid, array, &
    keywordEnforcer, datacopyflag, xgridside, gridindex, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    name, rc)
!
! !RETURN VALUE:
    type(ESMF_Field) :: ESMF_FieldCreateXGArray
!
! !ARGUMENTS:
    type(ESMF_XGrid), intent(in) :: xgrid
    type(ESMF_Array), intent(in) :: array
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag
    type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside
    integer, intent(in), optional :: gridindex
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    character (len = *), intent(in), optional :: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Create an {\tt ESMF\_Field}. This version of creation
! assumes the data exists already and is being
! passed in through an {\tt ESMF\_Array}. For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:create_grid_array}.
!
! The arguments are:
! \begin{description}
! \item [xgrid]
! {\tt ESMF\_XGrid} object.
! \item [array]
! {\tt ESMF\_Array} object.
! \item [{[datacopyflag]}]
! Indicates whether to copy the contents of the {\tt array} or reference it directly.
! For valid values see \ref{const:datacopyflag}. The default is
! {\tt ESMF\_DATACOPY\_REFERENCE}.
! \item [{[xgridside]}]
! Which side of the XGrid to create the Field on (either ESMF\_XGRIDSIDE\_A,
! ESMF\_XGRIDSIDE\_B, or ESMF\_XGRIDSIDE\_BALANCED). If not passed in then
! defaults to ESMF\_XGRIDSIDE\_BALANCED.
! \item [{[gridindex]}]
! If xgridSide is ESMF\_XGRIDSIDE\_A or ESMF\_XGRIDSIDE\_B then this index tells which Grid on
! that side to create the Field on. If not provided, defaults to 1.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the XGrid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! XGrid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom
    integer :: ungridded_ldimcount, ungridded_udimcount

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if (present(rc)) then
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)
    ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,array,rc)

    ! Sanity checking
    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(present(ungriddedLBound)) then
        ungridded_ldimcount = size(ungriddedLBound)
      endif
      if(present(ungriddedUBound)) then
        ungridded_udimcount = size(ungriddedUBound)
      endif
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    ! Create Geom from Mesh
    geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Create Field from Geom
    ESMF_FieldCreateXGArray=ESMF_FieldCreateGBArray(geom, array, &
      datacopyflag, gridToFieldMap, ungriddedLBound, ungriddedUBound, &
      name=name, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! remember that geom was created internally
    ESMF_FieldCreateXGArray%ftypep%geomb_internal = .true.

    if (present(rc)) rc = ESMF_SUCCESS

  end function ESMF_FieldCreateXGArray
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldCreate - Create a Field from XGrid and Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldCreate() 
! function ESMF_FieldCreateXGData<rank><type><kind>(xgrid, & 
! farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
! gridToFieldMap, ungriddedLBound, ungriddedUBound, name,& 
! rc) 
! 
! !RETURN VALUE: 
! type(ESMF_Field) :: ESMF_FieldCreateXGData<rank><type><kind> 
! 
! !ARGUMENTS: 
! type(ESMF_XGrid), intent(in) :: xgrid 
! <type> (ESMF_KIND_<kind>), intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
! integer, intent(in), optional :: gridindex 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! character (len=*), intent(in), optional :: name 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! Create an {\tt ESMF\_Field} from a Fortran data array and {\tt ESMF\_Xgrid}. 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! 
! The arguments are: 
! \begin{description} 
! \item [xgrid] 
! {\tt ESMF\_XGrid} object. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the Field 
! The Field dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item[indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item [{[datacopyflag]}] 
! Whether to copy the contents of the {\tt farray} or reference directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[xgridside]}] 
! Which side of the XGrid to create the Field on (either ESMF\_XGRIDSIDE\_A, 
! ESMF\_XGRIDSIDE\_B, or ESMF\_XGRIDSIDE\_BALANCED). If not passed in then 
! defaults to ESMF\_XGRIDSIDE\_BALANCED. 
! \item [{[gridindex]}] 
! If xgridside is ESMF\_XGRIDSIDE\_A or ESMF\_XGRIDSIDE\_B then this index tells which Grid on 
! that side to create the Field on. If not provided, defaults to 1. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt xgrid}'s dimCount. The list elements map each dimension 
! of the {\tt xgrid} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The default is to 
! map all of the {\tt xgrid}'s dimensions against the lowest dimensions of 
! the {\tt farray} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../). 
! The values of all {\tt gridToFieldMap} entries must be greater than or equal 
! to one and smaller than or equal to the {\tt farray} rank. 
! It is erroneous to specify the same {\tt gridToFieldMap} entry 
! multiple times. The total ungridded dimensions in the {\tt field} 
! are the total {\tt farray} dimensions less 
! the total (distributed + undistributed) dimensions in 
! the {\tt xgrid}. Unxgridded dimensions must be in the same order they are 
! stored in the {\t farray}. Permutations of the order of 
! dimensions are handled via individual communication methods. For example, 
! an undistributed dimension can be remapped to a distributed dimension 
! as part of the {\tt ESMF\_ArrayRedist()} operation. 
! If the Field dimCount is less than the Xgrid dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Xgrid dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than xgrid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than xgrid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt farray}. 
! \item [{[name]}] 
! Field name. 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData1Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData1Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData1Di1 = & 
 ESMF_FieldCreateGBData1Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData1Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData1Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData2Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData2Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData2Di1 = & 
 ESMF_FieldCreateGBData2Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData2Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData2Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData3Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData3Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData3Di1 = & 
 ESMF_FieldCreateGBData3Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData3Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData3Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData4Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData4Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData4Di1 = & 
 ESMF_FieldCreateGBData4Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData4Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData4Di1 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData5Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData5Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData5Di1 = & 
 ESMF_FieldCreateGBData5Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData5Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData5Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData6Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData6Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData6Di1 = & 
 ESMF_FieldCreateGBData6Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData6Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData6Di1 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData7Di1(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData7Di1 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i1), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData7Di1 = & 
 ESMF_FieldCreateGBData7Di1(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData7Di1%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData7Di1 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData1Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData1Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData1Di2 = & 
 ESMF_FieldCreateGBData1Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData1Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData1Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData2Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData2Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData2Di2 = & 
 ESMF_FieldCreateGBData2Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData2Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData2Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData3Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData3Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData3Di2 = & 
 ESMF_FieldCreateGBData3Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData3Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData3Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData4Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData4Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData4Di2 = & 
 ESMF_FieldCreateGBData4Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData4Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData4Di2 
!------------------------------------------------------------------------------ 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData5Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData5Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData5Di2 = & 
 ESMF_FieldCreateGBData5Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData5Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData5Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData6Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData6Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData6Di2 = & 
 ESMF_FieldCreateGBData6Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData6Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData6Di2 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData7Di2(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData7Di2 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i2), intent(in), target :: farray(:,:,:,:,:,:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData7Di2 = & 
 ESMF_FieldCreateGBData7Di2(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData7Di2%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData7Di2 
!------------------------------------------------------------------------------ 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData1Di4(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData1Di4 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData1Di4 = & 
 ESMF_FieldCreateGBData1Di4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData1Di4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData1Di4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData1Di8(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData1Di8 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData1Di8 = & 
 ESMF_FieldCreateGBData1Di8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData1Di8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData1Di8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData1Dr4(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData1Dr4 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 real (ESMF_KIND_r4), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData1Dr4 = & 
 ESMF_FieldCreateGBData1Dr4(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData1Dr4%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData1Dr4 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData1Dr8(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData1Dr8 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 real (ESMF_KIND_r8), intent(in), target :: farray(:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = size(ungriddedLBound) 
 endif 
 if(present(ungriddedUBound)) then 
 ungridded_udimcount = size(ungriddedUBound) 
 endif 
 if(ungridded_ldimcount /= ungridded_udimcount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- ungriddedLBound size must equal to ungriddedUBound size", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Create Geom from Xgrid
 geom=ESMF_GeomCreate(xgrid, xgridSide, gridIndex ,rc=localrc)
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
 ESMF_CONTEXT, rcToReturn=rc)) return


 ESMF_FieldCreateXGData1Dr8 = & 
 ESMF_FieldCreateGBData1Dr8(geom, & 
 farray, indexflag=indexflag, datacopyflag=datacopyflag, & 
 gridToFieldMap=gridToFieldMap, & 
 ungriddedLBound=ungriddedLBound, & 
 ungriddedUBound=ungriddedUBound, & 
 name=name, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! remember that geom was created internally 
 ESMF_FieldCreateXGData1Dr8%ftypep%geomb_internal = .true. 

 if (present(rc)) rc = ESMF_SUCCESS 
 end function ESMF_FieldCreateXGData1Dr8 
!------------------------------------------------------------------------------ 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldCreateXGData" 
 function ESMF_FieldCreateXGData2Di4(xgrid, & 
 farray, indexflag, keywordEnforcer, datacopyflag, xgridside, gridindex, & 
 gridToFieldMap, ungriddedLBound, ungriddedUBound, name, rc) 

 ! return value 
 type(ESMF_Field) :: ESMF_FieldCreateXGData2Di4 

 ! input arguments 
 type(ESMF_XGrid), intent(in) :: xgrid 
 integer (ESMF_KIND_i4), intent(in), target :: farray(:,:) 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 type(ESMF_XGridSide_Flag), intent(in), optional :: xgridside 
 integer, intent(in), optional :: gridindex 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 character (len=*), intent(in), optional :: name 
 integer, intent(out), optional :: rc 

 integer :: localrc ! Local error code
 type(ESMF_Geom) :: geom 
 integer :: ungridded_ldimcount, ungridded_udimcount 

 ! Initialize 
 localrc = ESMF_RC_NOT_IMPL

 ! Initialize return code 
 if (present(rc)) then
 rc = ESMF_RC_NOT_IMPL
 endif 

 ! check init status of input parameters
 ESMF_INIT_CHECK_DEEP(ESMF_XGridGetInit,xgrid,rc)


 ! Sanity checking 
 if(present(ungriddedLBound) .and. present(ungriddedUBound)) then 
 if(present(ungriddedLBound)) then 
 ungridded_ldimcount = si