/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.daemon;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.videolan.vlma.common.IVlOrderGiver;
import org.videolan.vlma.common.IVlData;
import org.videolan.vlma.common.VlServer;
import org.videolan.vlma.common.adapters.IVlAdapter;
import org.videolan.vlma.common.medias.IVlMedia;
import org.videolan.vlma.common.medias.VlMediaGroup;
import org.videolan.vlma.common.orders.VlOrder;
import org.videolan.vlma.common.programs.IVlProgram;

/**
 * This class can compute the orders to give to the servers depending on the
 * medias' programmation, on the servers' and peripherics' state, and on the
 * priority of the programms.
 * 
 * @author SylV
 */
public class VlOrderGiver implements IVlOrderGiver {
    private static final Logger logger = Logger.getLogger(VlOrderGiver.class);

    private IVlData data;

    private Thread computingThread;

    /**
     * This constructor needs the data interface it will control orders.
     * 
     * @param data
     */
    public VlOrderGiver(IVlData data) {
        this.data = data;
        computingThread = null;
    }

    /**
     * This method indicates whether or not the object is computing the orders.
     * 
     * @return true only and only if the object is computing the orders. ordres
     */
    public boolean isComputing() {
        return (computingThread != null) && (computingThread.isAlive());
    }

    /**
     * This method is the actual orders' computing method.
     */
    private Runnable orderComputer = new Runnable() {
        public void run() {
            logger.log(Level.DEBUG, "Starting computing channels assignement.");
            Map<Integer, List<VlOrder>> orders = data.getOrders(); // Streaming orders
            List<IVlMedia> medias = data.getMedias(); // Media list
            List<VlServer> servers = data.getServers(); // Servers list
            Map<Integer, List<VlMediaGroup>> programGroups = new HashMap<Integer, List<VlMediaGroup>>();
            Map<Integer, List<IVlAdapter>> readyAdapters = new HashMap<Integer, List<IVlAdapter>>();
            Map<IVlAdapter, VlServer> readyAdaptersServers = new HashMap<IVlAdapter, VlServer>();


            // The ready adapters are listed and grouped by hashType.
            {
                int nbAdapters = 0;
                logger.log(Level.DEBUG, "Analysing available adapters.");
                for (VlServer s : servers) {
                    if (!s.isUp()) {
                        continue;
                    }
                    for (IVlAdapter a : s.getAdapters().values()) {
                        if (!a.isUp()) {
                            continue;                        
                        }
                        List<IVlAdapter> sameHashTypeAdapterList = readyAdapters.get(a.hashType());
                        if (sameHashTypeAdapterList == null) {
                            sameHashTypeAdapterList = new ArrayList<IVlAdapter>();
                            readyAdapters.put(a.hashType(), sameHashTypeAdapterList);
                        }
                        readyAdaptersServers.put(a, s);
                        sameHashTypeAdapterList.add(a);
                        nbAdapters++;
                    }
                }
                logger.log(Level.DEBUG, nbAdapters + " available adapters.");
            }
            
            {
                int nbMedias = 0;
                logger.log(Level.DEBUG, "Putting together media to stream.");
                for (IVlMedia m : medias) {
                    IVlProgram p = m.getProgram();
                    if (p != null && p.isTimeToPlay()) {
                        logger.log(Level.DEBUG, "Media to be streamed: " + m.getName());
                        for (Integer h : readyAdapters.keySet()) {
                            List<IVlAdapter> sameHashTypeAdapterList = readyAdapters.get(h);
                            // If the first adapter of the list can read the
                            // media, then
                            if (sameHashTypeAdapterList.get(0).canRead(m)) {
                                // Get or create the VLMediaGroup list of this
                                // type of media
                                List<VlMediaGroup> sameHashTypeGroupList = programGroups.get(h);
                                if (sameHashTypeGroupList == null) {
                                    sameHashTypeGroupList = new ArrayList<VlMediaGroup>();
                                    programGroups.put(h, sameHashTypeGroupList);
                                }
                                
                                // Inside the the VlMediaGroup list, get the
                                // VlMediaGroup of this type of media if it
                                // already exists.
                                VlMediaGroup mediaGroup = null;    
                                for (VlMediaGroup j : sameHashTypeGroupList) {
                                    if (m.belongsToGroup(j)) {
                                        mediaGroup = j;
                                        break;
                                    }
                                }
                                // Create the mediaGroup if not ...
                                if (mediaGroup == null) {
                                    mediaGroup = new VlMediaGroup();
                                    sameHashTypeGroupList.add(mediaGroup);
                                }
                                // Add the media to its group
                                mediaGroup.medias.add(m);
                                
                                nbMedias++;
                                break;
                            }
                            else {
                                logger.log(Level.DEBUG, "This adapter cannot stream " + m.getName());
                            }
                        }
                    }
                }
                logger.log(Level.DEBUG, nbMedias + " media to stream.");
            }

            // One assings a program group by adapter
            logger.log(Level.DEBUG, "Assigning programms to adapters.");
            for (Integer c : readyAdapters.keySet()) {
                logger.log(Level.DEBUG, "Media type: " + c.toString());
                List<VlMediaGroup> sameHashTypeGroupList = programGroups.get(c);
                List<IVlAdapter> sameHashTypeAdapterList = readyAdapters.get(c);
                logger.log(Level.DEBUG, sameHashTypeAdapterList.size() + " adapters.");
                // Removes low-priority groups
                if (sameHashTypeGroupList != null) {
                    Collections.sort(sameHashTypeGroupList);
                    logger.log(Level.DEBUG, sameHashTypeGroupList.size() + " media groups/");
                    int nbAdapters = sameHashTypeAdapterList.size();
                    while (sameHashTypeGroupList.size() > nbAdapters) {
                        logger.log(Level.DEBUG, "This media group won't be streamed (priority " + sameHashTypeGroupList.get(sameHashTypeGroupList.size() - 1).getPriority() + ") :");
                        for (IVlMedia m : sameHashTypeGroupList.get(sameHashTypeGroupList.size() - 1).medias) {
                            logger.log(Level.DEBUG, "- " + m.getName());
                        }
                        sameHashTypeGroupList.remove(sameHashTypeGroupList.size() - 1);
                    }
                }

                logger.log(Level.DEBUG, "Remove unused orders.");
                List<VlOrder> oldOrders = new ArrayList<VlOrder>();
                // Save the oldOrders list
                if (orders.containsKey(c)) {
                    // Make a copy of the old orders
                    oldOrders.addAll(orders.get(c));
                }
                logger.log(Level.DEBUG, oldOrders.size() + " former orders");
                if (sameHashTypeGroupList != null) {
                    Iterator<VlMediaGroup> groupIt = sameHashTypeGroupList.iterator();
                    while (groupIt.hasNext())
                    {
                        VlMediaGroup g = groupIt.next();
                        /*
                         * The groups that are already diffused are removed from
                         * the list. Their correspondant adpater are also
                         * removed from the list of ready adapters.
                         */
                        Iterator<VlOrder> orderIt = oldOrders.iterator();
                        while (orderIt.hasNext()) {
                            VlOrder order = orderIt.next();
                            logger.log(Level.DEBUG, g.medias.size() + " media against " + order.getMedias().medias.size() + " media");
                            if ((order.getMedias().medias.containsAll(g.medias))
                                    && (g.medias.containsAll(order.getMedias().medias))
                                    && (order.getServer().getIp() == g.medias.get(0).getProgram().getPlayer())) {
                                // Remove the mediaGroup
                                groupIt.remove();
                                // Remove its correspondant adapter
                                sameHashTypeAdapterList.remove(order.getAdapter());
                                // Remove now the order
                                orderIt.remove();
                                break;
                            }
                        }
                    }
                }
                    
                /*
                 * The orders that are no more maintened are stoped and removed
                 * from the order list.
                 */
                for (VlOrder o : oldOrders)
                {
                    try {
                        /* Stop the execution of the order */
                        o.stop();
                        /* Update the medias programs properties */
                        for (IVlMedia media : o.getMedias().medias) {
                            if (media.getProgram() != null) {
                                media.getProgram().setPlayer(null);
                                media.getProgram().setBroadcastState(false);
                            }
                        }
                    } catch (IOException e) {

                    }
                    orders.get(c).remove(o);
                }
                
                
                if (!orders.containsKey(c)) {
                    orders.put(c, new ArrayList<VlOrder>());
                }
                
                    
                // The non existing orders are now created and executed.
                if (sameHashTypeGroupList != null) {
                    logger.log(Level.DEBUG, "Creating new orders.");
                    Iterator<VlMediaGroup> i = sameHashTypeGroupList.iterator();
                    Iterator<IVlAdapter> j = sameHashTypeAdapterList.iterator();
                    while (i.hasNext() && j.hasNext()) {
                        VlMediaGroup g = i.next();
                        IVlAdapter a = j.next();
                        VlServer adapterServer = a.getServer();
                        // VlServer adapterServer = readyAdaptersServers.get(a);
                        VlOrder o = new VlOrder();
                        o.setAdapter(a);
                        o.setServer(adapterServer);
                        o.setMedias(g);
                        logger.log(Level.DEBUG, "Assigning channels on adapter "
                                + a.getName() + " of server " + adapterServer.getName());
                        /*
                         * Set the server to the programs of the diffused medias
                         * for this adapeter
                         */
                        for (IVlMedia media : o.getMedias().medias) {
                            media.getProgram().setPlayer(adapterServer.getIp());
                        }
                        try {
                            o.start();
                        } catch (IOException e) {
                            adapterServer.setUp(false);
                        }
                        orders.get(c).add(o);
                    }
                }
            }
        }
    };

    /**
     * This method launches the computing of orders in a news thread, if it was
     * not already running.
     */
    synchronized public void computeOrders() {        
        if (!isComputing()) {
            computingThread = new Thread(orderComputer);
            computingThread.setName("OrderComputingThread");
            computingThread.start();
        }
    }
}
